<?php

namespace Drupal\schemaapp\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Schema App Configuration Form.
 */
class SchemaAppForm extends ConfigFormBase {

  /**
   * Event Dispatcher.
   *
   * @var object
   *   Drupal Event Dispatcher
   */
  protected $eventDispatcher;

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['schemaapp.config'];
  }

  /**
   * {@inheritdoc}
   */
  public function __construct($eventDispatcher) {
    $this->eventDispatcher = $eventDispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('event_dispatcher')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'schemaapp';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('schemaapp.config');
    $form['account_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Account ID'),
      '#default_value' => $config->get('account_id'),
    ];
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $config->get('api_key'),
    ];
    $form['use_url_params'] = [
      "#type" => 'checkbox',
      '#title' => $this->t('Use get parameters'),
      '#default_value' => $config->get('use_url_params') ?? FALSE,
    ];

    $form['cache_lifetime'] = [
      "#type" => 'number',
      '#title' => $this->t('Cache lifetime'),
      '#description' => $this->t('The number of seconds + 0-25% for an item to remain
        cached locally on the server before fetching from the Schema App CDN. For example if the default of 86400 it can be kept up to 108000'),
      '#min' => 0,
    // No more than 7 days.
      '#max' => 604800,
    // Increment in minutes.
      '#step' => 60,
      '#default_value' => $config->get('cache_lifetime') ?? 86400,
    ];

    $form['api_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout'),
      '#description' => $this->t('The number of seconds to wait for the
        Schema App CDN to respond'),
      '#min' => 1,
      '#max' => 60,
      '#step' => 1,
      '#default_value' => $config->get('api_timeout') ?? 30,
    ];

    $form['use_highlighter_markup'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Highlighter JavaScript Markup'),
      '#description' => $this->t('Use Markup generated by the Highlighter JS script
        stored on Schema App\'s CDN'),
      '#default_value' => $config->get('use_highlighter_markup') ?? FALSE,
    ];

    $form['enable_debug_info'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable Debugging Info'),
        '#description' => $this->t('Include meta elements containing debugging info in HTML head element'),
        '#default_value' => $config->get('enable_debug_info') ?? FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    $config = $this->config('schemaapp.config');

    $config->set('api_key', $form_state->getValue('api_key'))
      ->set('account_id', $form_state->getValue('account_id'))
      ->set('use_url_params', $form_state->getValue('use_url_params'))
      ->set('use_highlighter_markup', $form_state->getValue('use_highlighter_markup'))
      ->set('enable_debug_info', $form_state->getValue('enable_debug_info'))
      ->set('api_timeout', $form_state->getValue('api_timeout'))
      ->set('cache_lifetime', $form_state->getValue('cache_lifetime'));

    $config->save();
  }

}
