<?php

namespace Drupal\scroll_depth_indicator\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\NodeType;
use Drupal\Component\Utility\Html;

/**
 * Defines the configuration form for the Scroll Depth Indicator module.
 *
 * Allows administrators to select content types and configure selector
 * placement settings for the scroll depth indicator.
 */
class ScrollDepthIndicatorConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'scroll_depth_indicator_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['scroll_depth_indicator.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('scroll_depth_indicator.settings');

    // Get all available node bundles (content types).
    $node_bundles = NodeType::loadMultiple();
    $options = [];
    foreach ($node_bundles as $machine_name => $node_bundle) {
      $node_type_class = 'node-' . Html::getClass($machine_name);
      $options[$node_type_class] = $node_bundle->label();
    }

    // Add multi-select field for selecting content types.
    $form['enabled_content_types'] = [
      '#type' => 'select',
      '#title' => $this->t('Enabled Content Types'),
      '#description' => $this->t('Select the content types where the scroll depth indicator should be active.'),
      '#options' => $options,
      '#default_value' => $config->get('enabled_content_types') ?? [],
      '#multiple' => TRUE,
      '#required' => TRUE,
    ];

    // Element Placement Settings fieldset.
    $form['element_placement_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Element Placement Settings'),
    ];

    // Dropdown options.
    $options = [
      'element' => $this->t('Element'),
      'class' => $this->t('Class'),
      'id' => $this->t('ID'),
    ];

    $form['element_placement_settings']['selector_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Selector Type'),
      '#options' => $options,
      '#default_value' => $config->get('element_placement_settings.selector_type') ?? 'element',
      '#description' => $this->t('Select the type of selector to append the scroll depth indicator.'),
      '#required' => TRUE,
    ];

    $form['element_placement_settings']['selector'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Specify the selector to append the Progress indicator'),
      '#default_value' => $config->get('element_placement_settings.selector') ?? 'header',
      '#description' => $this->t('Specify the selector of the element to append the Progress indicator.'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('scroll_depth_indicator.settings')
      ->set('enabled_content_types', $form_state->getValue('enabled_content_types'))
      ->set('element_placement_settings', [
        'selector_type' => $form_state->getValue(['element_placement_settings', 'selector_type']),
        'selector' => $form_state->getValue(['element_placement_settings', 'selector']),
      ])
      ->save();

    parent::submitForm($form, $form_state);
  }

}
