# SDC Inline Editor Route Integration

The SDC Inline Editor can work on custom routes that display nodes, not just the canonical `entity.node.canonical` route. This allows you to use the editor on custom routes like `/lessons/{title}` or any other custom node display route.

## How It Works

The module uses a hook system to allow other modules to register custom routes. When a route is registered, the SDC Inline Editor will:

1. Check if the route has a `node` parameter
2. Verify the node has SDC Inline Editor configuration enabled
3. Check user permissions
4. Attach the editor libraries and settings

## Registering Custom Routes

To register a custom route, implement `hook_sdc_inline_editor_routes()` in your module:

```php
<?php

/**
 * Implements hook_sdc_inline_editor_routes().
 */
function mymodule_sdc_inline_editor_routes() {
  return [
    'mymodule.custom_node_route',
    'mymodule.another_custom_route',
  ];
}
```

## Important: Route Parameter Requirements

For the SDC Inline Editor to work on your custom route, **the route must have a `node` parameter** that contains a `NodeInterface` object.

### Option 1: Use `{node}` in Route Path

Update your route definition to use `{node}` as the parameter:

```yaml
mymodule.custom_node_route:
  path: '/custom-path/{node}'
  defaults:
    _controller: '\Drupal\mymodule\Controller\MyController::view'
  requirements:
    _permission: 'access content'
    node: \d+  # Node ID
```

Then use a parameter converter or load the node in your controller.

### Option 2: Set Node in Controller

If your route uses a different parameter (like `{title}`), you need to load the node and set it as a route parameter in your controller:

```php
<?php

namespace Drupal\mymodule\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class MyController extends ControllerBase {

  public function viewByTitle($title) {
    // Load the node by title (or whatever your route parameter is).
    $nodes = \Drupal::entityTypeManager()
      ->getStorage('node')
      ->loadByProperties([
        'type' => 'your_content_type',
        'title' => urldecode($title),
      ]);
    
    if (empty($nodes)) {
      throw new NotFoundHttpException();
    }
    
    $node = reset($nodes);
    
    // IMPORTANT: Set the node in the request attributes so SDC Inline Editor can find it.
    // RouteMatch reads parameters from request attributes, so this is sufficient.
    \Drupal::request()->attributes->set('node', $node);
    
    // Build and return the node view.
    $build = \Drupal::entityTypeManager()
      ->getViewBuilder('node')
      ->view($node, 'full');
    
    return $build;
  }
}
```

### Option 3: Use a Route Parameter Converter (Recommended)

Create a parameter converter to automatically convert your route parameter to a node:

```php
<?php

namespace Drupal\mymodule\ParamConverter;

use Drupal\Core\ParamConverter\ParamConverterInterface;
use Symfony\Component\Routing\Route;

class TitleToNodeConverter implements ParamConverterInterface {

  public function convert($value, $definition, $name, array $defaults) {
    // Load node by title.
    $nodes = \Drupal::entityTypeManager()
      ->getStorage('node')
      ->loadByProperties([
        'type' => 'your_content_type',
        'title' => urldecode($value),
      ]);
    
    return !empty($nodes) ? reset($nodes) : NULL;
  }

  public function applies($definition, $name, Route $route) {
    return !empty($definition['type']) && $definition['type'] === 'title_to_node';
  }
}
```

Register it in `mymodule.services.yml`:

```yaml
services:
  mymodule.title_to_node_converter:
    class: Drupal\mymodule\ParamConverter\TitleToNodeConverter
    tags:
      - { name: paramconverter }
```

Then update your route:

```yaml
mymodule.custom_node_route:
  path: '/custom-path/{node}'
  defaults:
    _controller: '\Drupal\mymodule\Controller\MyController::view'
  requirements:
    _permission: 'access content'
    node:
      type: title_to_node
```

## Example: Complete Implementation

Here's a complete example for a custom route that displays nodes by title:

### 1. Route Definition (`mymodule.routing.yml`)

```yaml
mymodule.lesson_by_title:
  path: '/lessons/{node}'
  defaults:
    _controller: '\Drupal\mymodule\Controller\LessonController::viewByTitle'
    _title_callback: '\Drupal\mymodule\Controller\LessonController::getTitle'
  requirements:
    _permission: 'access content'
    node:
      type: title_to_node
```

### 2. Register the Route (`mymodule.module`)

```php
<?php

/**
 * Implements hook_sdc_inline_editor_routes().
 */
function mymodule_sdc_inline_editor_routes() {
  return [
    'mymodule.lesson_by_title',
  ];
}
```

### 3. Parameter Converter (`src/ParamConverter/TitleToNodeConverter.php`)

```php
<?php

namespace Drupal\mymodule\ParamConverter;

use Drupal\Core\ParamConverter\ParamConverterInterface;
use Symfony\Component\Routing\Route;

class TitleToNodeConverter implements ParamConverterInterface {

  public function convert($value, $definition, $name, array $defaults) {
    $nodes = \Drupal::entityTypeManager()
      ->getStorage('node')
      ->loadByProperties([
        'type' => 'lesson',
        'title' => urldecode($value),
      ]);
    
    return !empty($nodes) ? reset($nodes) : NULL;
  }

  public function applies($definition, $name, Route $route) {
    return !empty($definition['type']) && $definition['type'] === 'title_to_node';
  }
}
```

### 4. Service Definition (`mymodule.services.yml`)

```yaml
services:
  mymodule.title_to_node_converter:
    class: Drupal\mymodule\ParamConverter\TitleToNodeConverter
    tags:
      - { name: paramconverter }
```

## Troubleshooting

### Editor Not Appearing

1. **Check route registration**: Make sure your route is returned by `hook_sdc_inline_editor_routes()`
2. **Verify node parameter**: Ensure the route has a `node` parameter set
3. **Check permissions**: User must have `sdc_inline_editor.edit_content` permission
4. **Content type configuration**: The content type must have SDC Inline Editor enabled in configuration
5. **Clear cache**: After implementing the hook, clear Drupal cache

### Route Parameter Not Found

If you get errors about missing node parameter:

1. Make sure your controller sets the node: `\Drupal::request()->attributes->set('node', $node);`
2. Or use a parameter converter to automatically convert your route parameter to a node
3. Verify the route parameter name matches what you're setting

## See Also

- `hook_sdc_inline_editor_routes()` - Hook documentation
- `sdc_inline_editor_is_valid_node_view_route()` - Route validation function
