# SDC Attributes Twig Functions

This module provides convenient Twig functions to generate SDC (Single Directory Component) attributes for your templates.

## Available Functions

### `sdc_component_attributes(component_name, additional_attributes)`

Generates component-level SDC attributes.

**Parameters:**
- `component_name` (string|null): The component name
- `additional_attributes` (array): Additional attributes to include

**Example:**
```twig
<div{{ sdc_component_attributes('my-component', {'class': 'custom-class'}) }}>
  Content here
</div>
```

**Output:**
```html
<div data-sdc-component="my-component" class="custom-class">
  Content here
</div>
```

### `sdc_field_attributes(field_name, editable, additional_attributes)`

Generates field-level SDC attributes for inline editing.

**Parameters:**
- `field_name` (string): The field name
- `editable` (bool): Whether the field is editable (default: false)
- `additional_attributes` (array): Additional attributes to include

**Example:**
```twig
<div{{ sdc_field_attributes('title', true) }}>
  {{ title }}
</div>
```

**Output:**
```html
<div data-sdc-field="title" data-sdc-editable="true">
  {{ title }}
</div>
```

#### Sidebar `data-sdc-type` quick reference

The sidebar decides which editor to render based on the `data-sdc-type` attribute. Supported values map to handlers in `sdc_inline_editor/js/src/inline-editor/fieldHandlers/sidebar/`:

- `text-input`
- `number-input`
- `checkbox-toggle`
- `select-dropdown`
- `image-selector`
- `multi-image-selector`
- `icon-selector` *(searchable Font Awesome picker; ensure the icon in your markup uses `data-sdc-icon-preview` so the sidebar can live-update it.)*

### Restricting drop targets with `data-allowed-components`

To limit which child components may be placed inside a nested container, include the `data-allowed-components` attribute on the same element that declares `data-drop-container="true"`. Provide a JSON array containing the component IDs that should be accepted.

```twig
<div
  class="some-nested-container"
  data-drop-container="true"
  data-allowed-components='["mse_sdc_components:link-list-item", "sdc_inline_editor:text-block"]'>
  …
</div>
```

During drag-and-drop, the editor will only show drop zones compatible with the listed components. Attempting to drop a disallowed component (from the toolbar or by rearranging existing blocks) is blocked and the user receives a warning.

### Disabling component encapsulation with `data-sdc-no-encapsulate`

By default, all SDC components are automatically wrapped in a `block-wrapper` div with toolbar controls (delete, move, settings icons) when in editing mode. For nested components that shouldn't have their own wrapper (like list items within a list), add the `data-sdc-no-encapsulate` attribute to the root element.

**Example:**
```twig
<li{{ sdc_component_attributes(component_name) }} data-sdc-no-encapsulate class="mse-link-list__item">
  <a href="{{ url }}">{{ text }}</a>
</li>
```

**Output:**
```html
<li data-sdc-component="mse_sdc_components:link-list-item" data-sdc-no-encapsulate class="mse-link-list__item">
  <a href="#example">Example Link</a>
</li>
```

This prevents the component from being wrapped in a `block-wrapper` div, which is useful for components that are meant to be managed as part of a parent component's collection rather than as standalone editable blocks.

### `sdc_html_editor_attributes(component_name, field_name, editable, type, additional_attributes)`

Generates complete SDC attributes for HTML/text editors. **Note: This function combines both component and field attributes. For better separation, use `sdc_component_attributes()` and `sdc_field_attributes()` separately.**

**Parameters:**
- `component_name` (string|null): The component name
- `field_name` (string): The field name (default: 'html')
- `editable` (bool): Whether the field is editable (default: false)
- `type` (string): The editor type (default: 'html-editor')
- `additional_attributes` (array): Additional attributes to include

**Example:**
```twig
<div{{ sdc_html_editor_attributes(component_name, 'html', editable|default(false)) }}>
  {{ html }}
</div>
```

**Output:**
```html
<div data-sdc-component="my-component" data-sdc-field="html" data-sdc-editable="true" data-sdc-type="html-editor">
  {{ html }}
</div>
```

## Recommended Usage Pattern

**For better separation of concerns, use component and field attributes separately:**

```twig
<div{{ sdc_component_attributes(component_name) }} {{ sdc_field_attributes('html', editable|default(false), {'data-sdc-type': 'html-editor'}) }}>
  {{html}}
</div>
```

**Output:**
```html
<div data-sdc-component="my-component" data-sdc-field="html" data-sdc-editable="true" data-sdc-type="html-editor">
  {{html}}
</div>
```

## Migration from Manual Attributes

**Before:**
```twig
<div{% if component_name is defined %} data-sdc-component="{{component_name}}"{% endif %} data-sdc-field="html" data-sdc-editable="{{editable|default('false')}}" data-sdc-type="html-editor">
  {{html}}
</div>
```

**After (Recommended - Separated):**
```twig
<div{{ sdc_component_attributes(component_name) }} {{ sdc_field_attributes('html', editable|default(false), {'data-sdc-type': 'html-editor'}) }}>
  {{html}}
</div>
```

**After (Alternative - Combined):**
```twig
<div{{ sdc_html_editor_attributes(component_name, 'html', editable|default(false)) }}>
  {{html}}
</div>
```

## Benefits

1. **Cleaner templates**: Reduces repetitive attribute code
2. **Consistent formatting**: Ensures all SDC attributes are properly formatted
3. **Easier maintenance**: Changes to attribute structure only need to be made in one place
4. **Type safety**: Functions handle boolean conversion and HTML escaping
5. **Flexibility**: Easy to add additional attributes when needed
