/**
 * @file
 * Photo Set Block Component JavaScript
 */

(function () {
  'use strict';

  let lightboxState = {
    isOpen: false,
    currentIndex: 0,
    images: [],
    touchStartX: 0,
    touchEndX: 0
  };

  // Initialize photo set blocks
  Drupal.behaviors.photoSetBlock = {
    attach: function (context, settings) {
      const photoSets = context.querySelectorAll('.photo-set-block');
      
      photoSets.forEach(function (photoSet) {
        // Check if already initialized
        if (photoSet.classList.contains('photo-set-initialized')) {
          return;
        }
        
        photoSet.classList.add('photo-set-initialized');
        
        // Initialize scroll animations
        initScrollAnimations(photoSet);
        
        // Initialize lightbox for all images
        initLightbox(photoSet);
        
        // Add click handlers for images in carousel mode
        if (photoSet.classList.contains('photo-set-carousel')) {
          initCarousel(photoSet);
        }
        
        // Add masonry layout calculations if needed
        if (photoSet.classList.contains('photo-set-masonry')) {
          initMasonry(photoSet);
        }
        
        // Grid layout orphan handling is done via Twig logic and CSS
      });
    }
  };
  
  /**
   * Initialize lightbox functionality
   */
  function initLightbox(photoSet) {
    const images = photoSet.querySelectorAll('.photo-set-image');
    
    if (images.length === 0) {
      return;
    }
    
    // Create lightbox modal if it doesn't exist
    let lightbox = document.querySelector('.photo-lightbox');
    if (!lightbox) {
      lightbox = createLightboxModal();
      document.body.appendChild(lightbox);
    }
    
    // Add click handlers to images
    images.forEach(function(image, index) {
      // Make image clickable with cursor pointer
      image.style.cursor = 'pointer';
      image.setAttribute('tabindex', '0');
      image.setAttribute('role', 'button');
      image.setAttribute('aria-label', 'View image in lightbox');
      
      image.addEventListener('click', function(e) {
        e.preventDefault();
        openLightbox(photoSet, index);
      });
      
      // Add keyboard support
      image.addEventListener('keydown', function(e) {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          openLightbox(photoSet, index);
        }
      });
    });
  }
  
  /**
   * Create lightbox modal structure
   */
  function createLightboxModal() {
    const lightbox = document.createElement('div');
    lightbox.className = 'photo-lightbox';
    lightbox.setAttribute('role', 'dialog');
    lightbox.setAttribute('aria-modal', 'true');
    lightbox.setAttribute('aria-label', 'Image lightbox');
    
    lightbox.innerHTML = `
      <div class="lightbox-overlay"></div>
      <div class="lightbox-content">
        <button class="lightbox-close" aria-label="Close lightbox">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <line x1="18" y1="6" x2="6" y2="18"></line>
            <line x1="6" y1="6" x2="18" y2="18"></line>
          </svg>
        </button>
        <button class="lightbox-prev" aria-label="Previous image">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="15 18 9 12 15 6"></polyline>
          </svg>
        </button>
        <button class="lightbox-next" aria-label="Next image">
          <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <polyline points="9 18 15 12 9 6"></polyline>
          </svg>
        </button>
        <div class="lightbox-image-container">
          <img class="lightbox-image" src="" alt="" />
          <div class="lightbox-counter"></div>
        </div>
      </div>
    `;
    
    // Add event listeners
    const closeBtn = lightbox.querySelector('.lightbox-close');
    const overlay = lightbox.querySelector('.lightbox-overlay');
    const prevBtn = lightbox.querySelector('.lightbox-prev');
    const nextBtn = lightbox.querySelector('.lightbox-next');
    
    closeBtn.addEventListener('click', closeLightbox);
    overlay.addEventListener('click', closeLightbox);
    prevBtn.addEventListener('click', showPrevImage);
    nextBtn.addEventListener('click', showNextImage);
    
    // Keyboard navigation
    document.addEventListener('keydown', handleKeyboardNav);
    
    // Touch support for mobile swipe
    const imageContainer = lightbox.querySelector('.lightbox-image-container');
    imageContainer.addEventListener('touchstart', handleTouchStart, false);
    imageContainer.addEventListener('touchend', handleTouchEnd, false);
    
    return lightbox;
  }
  
  /**
   * Open lightbox with specific image
   */
  function openLightbox(photoSet, index) {
    const images = photoSet.querySelectorAll('.photo-set-image');
    
    // Store image data
    lightboxState.images = Array.from(images).map(img => ({
      src: img.getAttribute('data-original-image-url') || img.src,
      alt: img.alt || ''
    }));
    
    lightboxState.currentIndex = index;
    lightboxState.isOpen = true;
    
    // Show lightbox
    const lightbox = document.querySelector('.photo-lightbox');
    lightbox.classList.add('active');
    document.body.style.overflow = 'hidden';
    
    // Update image
    updateLightboxImage();
    
    // Focus on close button for accessibility
    setTimeout(() => {
      lightbox.querySelector('.lightbox-close').focus();
    }, 100);
  }
  
  /**
   * Close lightbox
   */
  function closeLightbox() {
    if (!lightboxState.isOpen) {
      return;
    }
    
    const lightbox = document.querySelector('.photo-lightbox');
    lightbox.classList.remove('active');
    document.body.style.overflow = '';
    lightboxState.isOpen = false;
  }
  
  /**
   * Show previous image
   */
  function showPrevImage() {
    if (lightboxState.currentIndex > 0) {
      lightboxState.currentIndex--;
      updateLightboxImage();
    }
  }
  
  /**
   * Show next image
   */
  function showNextImage() {
    if (lightboxState.currentIndex < lightboxState.images.length - 1) {
      lightboxState.currentIndex++;
      updateLightboxImage();
    }
  }
  
  /**
   * Update lightbox image display
   */
  function updateLightboxImage() {
    const lightbox = document.querySelector('.photo-lightbox');
    const img = lightbox.querySelector('.lightbox-image');
    const counter = lightbox.querySelector('.lightbox-counter');
    const prevBtn = lightbox.querySelector('.lightbox-prev');
    const nextBtn = lightbox.querySelector('.lightbox-next');
    
    const currentImage = lightboxState.images[lightboxState.currentIndex];
    
    // Add loading class
    img.classList.add('loading');
    
    // Update image
    img.onload = function() {
      img.classList.remove('loading');
    };
    img.src = currentImage.src;
    img.alt = currentImage.alt;
    
    // Update counter
    counter.textContent = `${lightboxState.currentIndex + 1} / ${lightboxState.images.length}`;
    
    // Update navigation buttons
    prevBtn.style.display = lightboxState.currentIndex > 0 ? 'flex' : 'none';
    nextBtn.style.display = lightboxState.currentIndex < lightboxState.images.length - 1 ? 'flex' : 'none';
  }
  
  /**
   * Handle keyboard navigation
   */
  function handleKeyboardNav(e) {
    if (!lightboxState.isOpen) {
      return;
    }
    
    switch(e.key) {
      case 'Escape':
        closeLightbox();
        break;
      case 'ArrowLeft':
        showPrevImage();
        break;
      case 'ArrowRight':
        showNextImage();
        break;
    }
  }
  
  /**
   * Handle touch start
   */
  function handleTouchStart(e) {
    lightboxState.touchStartX = e.changedTouches[0].screenX;
  }
  
  /**
   * Handle touch end
   */
  function handleTouchEnd(e) {
    lightboxState.touchEndX = e.changedTouches[0].screenX;
    handleSwipe();
  }
  
  /**
   * Handle swipe gesture
   */
  function handleSwipe() {
    const swipeThreshold = 50;
    const diff = lightboxState.touchStartX - lightboxState.touchEndX;
    
    if (Math.abs(diff) > swipeThreshold) {
      if (diff > 0) {
        // Swiped left - show next
        showNextImage();
      } else {
        // Swiped right - show previous
        showPrevImage();
      }
    }
  }
  
  /**
   * Initialize scroll animations for photo items
   */
  function initScrollAnimations(photoSet) {
    const items = photoSet.querySelectorAll('.photo-set-item');
    
    if (items.length === 0) {
      return;
    }
    
    // Create an Intersection Observer to watch for when items enter the viewport
    const observerOptions = {
      root: null, // Use viewport as root
      rootMargin: '0px 0px -100px 0px', // Trigger slightly before element enters viewport
      threshold: 0.1 // Trigger when 10% of element is visible
    };
    
    const observer = new IntersectionObserver(function(entries) {
      entries.forEach(function(entry) {
        if (entry.isIntersecting) {
          // Get the index for staggered animation
          const item = entry.target;
          const index = Array.from(items).indexOf(item);
          
          // Add staggered delay based on index
          const delay = index * 100; // 100ms delay between each item
          
          setTimeout(function() {
            item.classList.add('animate-in');
          }, delay);
          
          // Stop observing once animated
          observer.unobserve(item);
        }
      });
    }, observerOptions);
    
    // Observe all photo items
    items.forEach(function(item) {
      observer.observe(item);
    });
  }
  
  /**
   * Initialize carousel functionality
   */
  function initCarousel(photoSet) {
    const container = photoSet.querySelector('.photo-set-container');
    const items = container.querySelectorAll('.photo-set-item');
    
    if (items.length === 0) {
      return;
    }
    
    // Add carousel navigation if needed
    // This is a placeholder for carousel functionality
    console.log('Carousel initialized with', items.length, 'images');
  }
  
  /**
   * Initialize masonry layout
   */
  function initMasonry(photoSet) {
    const container = photoSet.querySelector('.photo-set-container');
    
    // Add masonry layout calculations
    // This is a placeholder for masonry functionality
    console.log('Masonry layout initialized');
  }

})();

