/**
 * @file
 * Custom HTML Editor for PR Builder
 * 
 * This editor preserves exact HTML structure without any normalization.
 * It provides a modal-based HTML source editor using CodeMirror 6.
 */

import { EditorView, basicSetup } from 'codemirror';
import { EditorState } from '@codemirror/state';
import { html } from '@codemirror/lang-html';
import { oneDark } from '@codemirror/theme-one-dark';
import js_beautify from 'js-beautify';

export class CustomHtmlEditor {
    constructor(element, config = {}) {
        this.element = element;
        this.config = config;
        this.isEditing = false;
        this.modalOverlay = null;
        this.modalContent = null;
        this.codeEditorView = null;
        this.originalContent = '';
        
        this.init();
    }

    init() {
        // Store original content
        this.originalContent = this.element.innerHTML;
        
        // Make element clickable to open editor
        this.element.style.cursor = 'pointer';
        this.element.style.border = '2px dashed transparent';
        this.element.style.transition = 'border-color 0.2s ease';
        
        // Add hover effects
        this.element.addEventListener('mouseenter', () => {
            if (!this.isEditing) {
                this.element.style.borderColor = '#007cba';
            }
        });
        
        this.element.addEventListener('mouseleave', () => {
            if (!this.isEditing) {
                this.element.style.borderColor = 'transparent';
            }
        });
        
        // Add click handler to open editor
        this.element.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            this.openEditor();
        });
        
        // Add visual indicator
        this.element.setAttribute('data-custom-html-editor', 'true');
    }

    openEditor() {
        if (this.isEditing) return;
        
        this.isEditing = true;
        this.element.style.borderColor = '#007cba';
        this.createModal();
    }

    closeEditor() {
        if (!this.isEditing) return;
        
        this.isEditing = false;
        this.element.style.borderColor = 'transparent';
        this.destroyModal();
    }

    createModal() {
        // Create modal overlay
        this.modalOverlay = document.createElement('div');
        this.modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
        `;

        // Create modal content
        this.modalContent = document.createElement('div');
        this.modalContent.style.cssText = `
            background: white;
            border-radius: 8px;
            width: 90%;
            max-width: 1200px;
            height: 80%;
            max-height: 800px;
            display: flex;
            flex-direction: column;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        `;

        // Create modal header
        const modalHeader = document.createElement('div');
        modalHeader.style.cssText = `
            padding: 20px;
            border-bottom: 1px solid #e0e0e0;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
            border-radius: 8px 8px 0 0;
        `;
        modalHeader.innerHTML = `
            <h3 style="margin: 0; color: #333; font-size: 18px;">HTML Source Editor</h3>
            <button type="button" style="
                background: none;
                border: none;
                font-size: 24px;
                cursor: pointer;
                color: #666;
                padding: 0;
                width: 30px;
                height: 30px;
                display: flex;
                align-items: center;
                justify-content: center;
            ">&times;</button>
        `;

        // Create modal body with CodeMirror
        const modalBody = document.createElement('div');
        modalBody.style.cssText = `
            flex: 1;
            padding: 0;
            overflow: hidden;
            display: flex;
            flex-direction: column;
        `;

        // Create CodeMirror container
        const codeMirrorContainer = document.createElement('div');
        codeMirrorContainer.style.cssText = `
            width: 100%;
            height: 100%;
            overflow: hidden;
            flex: 1;
            position: relative;
        `;

        modalBody.appendChild(codeMirrorContainer);

        // Create modal footer with buttons
        const modalFooter = document.createElement('div');
        modalFooter.style.cssText = `
            padding: 20px;
            border-top: 1px solid #e0e0e0;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
            background: #f8f9fa;
            border-radius: 0 0 8px 8px;
        `;
        modalFooter.innerHTML = `
            <button type="button" style="
                padding: 10px 20px;
                border: 1px solid #ccc;
                background: white;
                border-radius: 4px;
                cursor: pointer;
                font-size: 14px;
            ">Cancel</button>
            <button type="button" style="
                padding: 10px 20px;
                border: none;
                background: #007cba;
                color: white;
                border-radius: 4px;
                cursor: pointer;
                font-size: 14px;
            ">Save Changes</button>
        `;

        // Assemble modal
        this.modalContent.appendChild(modalHeader);
        this.modalContent.appendChild(modalBody);
        this.modalContent.appendChild(modalFooter);
        this.modalOverlay.appendChild(this.modalContent);

        // Add to document
        document.body.appendChild(this.modalOverlay);

        // Initialize CodeMirror
        this.initializeCodeMirror(codeMirrorContainer);

        // Setup event listeners
        this.setupModalEvents();

        // Focus the CodeMirror editor and refresh layout
        setTimeout(() => {
            this.codeEditorView.focus();
            // Force CodeMirror to refresh its layout
            this.codeEditorView.requestMeasure();
            
            // Add resize observer to handle layout changes
            if (window.ResizeObserver) {
                const resizeObserver = new ResizeObserver(() => {
                    this.codeEditorView.requestMeasure();
                });
                resizeObserver.observe(container);
                
                // Store observer for cleanup
                this.resizeObserver = resizeObserver;
            }
        }, 100);
    }

    initializeCodeMirror(container) {
        // Create extensions array
        const extensions = [
            basicSetup,
            html(),
            EditorView.theme({
                '.cm-content': {
                    fontSize: '14px',
                    fontFamily: 'Monaco, Menlo, "Ubuntu Mono", monospace',
                    lineHeight: '1.5',
                    padding: '12px',
                },
                '.cm-focused': {
                    outline: 'none',
                },
                '.cm-editor': {
                    border: 'none',
                    height: '100%',
                    maxHeight: '100%',
                },
                '.cm-scroller': {
                    fontFamily: 'Monaco, Menlo, "Ubuntu Mono", monospace',
                    overflow: 'auto !important',
                    maxHeight: '100%',
                },
                '.cm-editor .cm-scroller': {
                    overflow: 'auto !important',
                },
            }),
        ];

        // Add dark theme if specified
        if (this.config.theme === 'dark') {
            extensions.push(oneDark);
        }

        // Create initial state with current HTML content
        // Use custom beautify options to preserve custom elements and attributes
        const beautifyOptions = {
            indent_size: 2,
            indent_char: ' ',
            max_preserve_newlines: 2,
            preserve_newlines: true,
            keep_array_indentation: false,
            break_chained_methods: false,
            indent_scripts: 'normal',
            brace_style: 'collapse',
            space_before_conditional: true,
            unescape_strings: false,
            jslint_happy: false,
            end_with_newline: true,
            wrap_line_length: 0,
            indent_inner_html: true,
            comma_first: false,
            e4x: false,
            indent_empty_lines: false,
            // Preserve custom elements and attributes
            preserve_attributes: true,
            preserve_custom_elements: true
        };
        
        const startState = EditorState.create({
            doc: this.preserveCustomElements(this.element.innerHTML, beautifyOptions),
            extensions: extensions,
        });

        // Create editor view
        this.codeEditorView = new EditorView({
            state: startState,
            parent: container,
        });
    }

    setupModalEvents() {
        const closeBtn = this.modalContent.querySelector('button');
        const cancelBtn = this.modalContent.querySelectorAll('button')[1];
        const saveBtn = this.modalContent.querySelectorAll('button')[2];

        // Close modal handlers
        const closeModal = () => {
            this.closeEditor();
        };

        closeBtn.addEventListener('click', closeModal);
        cancelBtn.addEventListener('click', closeModal);

        // Overlay click to close
        this.modalOverlay.addEventListener('click', (e) => {
            if (e.target === this.modalOverlay) {
                closeModal();
            }
        });

        // Save handler
        saveBtn.addEventListener('click', () => {
            const newContent = this.codeEditorView.state.doc.toString();
            this.element.innerHTML = newContent;
            this.originalContent = newContent;
            closeModal();
        });

        // ESC key to close
        this.handleEscape = (e) => {
            if (e.key === 'Escape') {
                closeModal();
            }
        };
        document.addEventListener('keydown', this.handleEscape);
    }

    destroyModal() {
        if (this.modalOverlay) {
            document.body.removeChild(this.modalOverlay);
            this.modalOverlay = null;
            this.modalContent = null;
        }

        if (this.codeEditorView) {
            this.codeEditorView.destroy();
            this.codeEditorView = null;
        }

        if (this.resizeObserver) {
            this.resizeObserver.disconnect();
            this.resizeObserver = null;
        }

        if (this.handleEscape) {
            document.removeEventListener('keydown', this.handleEscape);
            this.handleEscape = null;
        }
    }

    // Method to get current content
    getData() {
        return this.element.innerHTML;
    }

    // Method to set content
    setData(data) {
        this.element.innerHTML = data;
        this.originalContent = data;
    }

    // Method to preserve custom elements during beautification
    preserveCustomElements(html, beautifyOptions) {
        try {
            // First, try to beautify the HTML
            const beautified = js_beautify.html_beautify(html, beautifyOptions);
            return beautified;
        } catch (error) {
            console.warn('HTML beautification failed, using original HTML:', error);
            // If beautification fails, return the original HTML with basic formatting
            return this.basicFormatHtml(html);
        }
    }

    // Basic HTML formatting that preserves all elements and attributes
    basicFormatHtml(html) {
        // Simple indentation without using js-beautify
        let formatted = html;
        
        // Add line breaks before opening tags (except inline elements)
        const blockElements = ['div', 'section', 'article', 'header', 'footer', 'nav', 'main', 'aside', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'ul', 'ol', 'li', 'form', 'table', 'thead', 'tbody', 'tr', 'td', 'th'];
        
        blockElements.forEach(tag => {
            const regex = new RegExp(`<${tag}([^>]*)>`, 'gi');
            formatted = formatted.replace(regex, `\n<$1$2>`);
            
            const closingRegex = new RegExp(`</${tag}>`, 'gi');
            formatted = formatted.replace(closingRegex, `\n</$1>`);
        });
        
        // Clean up multiple newlines
        formatted = formatted.replace(/\n\s*\n/g, '\n');
        
        // Add basic indentation
        const lines = formatted.split('\n');
        let indentLevel = 0;
        const indentedLines = lines.map(line => {
            const trimmed = line.trim();
            if (!trimmed) return '';
            
            // Decrease indent for closing tags
            if (trimmed.startsWith('</')) {
                indentLevel = Math.max(0, indentLevel - 1);
            }
            
            const indented = '  '.repeat(indentLevel) + trimmed;
            
            // Increase indent for opening tags (but not self-closing or inline elements)
            if (trimmed.startsWith('<') && !trimmed.startsWith('</') && !trimmed.endsWith('/>') && !trimmed.match(/<(img|br|hr|input|meta|link|script|style)/i)) {
                indentLevel++;
            }
            
            return indented;
        });
        
        return indentedLines.join('\n').trim();
    }

    // Method to destroy the editor
    destroy() {
        this.closeEditor();
        this.element.removeAttribute('data-custom-html-editor');
        this.element.style.cursor = '';
        this.element.style.border = '';
        this.element.style.transition = '';
    }
}

export default CustomHtmlEditor;
