/**
 * @file
 * Modal-based CKEditor 5 text editor for PR Builder text fields.
 *
 * This editor mirrors the behavior of the inline text editor but keeps the
 * editing experience inside a pop-out modal so layout-dependent components
 * (like floated images) maintain their front-end positioning while authors edit.
 */

import { TextEditor } from './ckeditor5-config';

const OVERLAY_Z_INDEX = 10000;

export class CustomTextPopupEditor {
	constructor(element, config = {}) {
		this.element = element;
		this.config = config;
		this.editorInstance = null;
		this.isEditing = false;

		this.modalOverlay = null;
		this.modalContent = null;
		this.modalBody = null;
		this.modalEditorContainer = null;
		this.toolbarElement = null;

		this.handleElementClick = this.handleElementClick.bind(this);
		this.handleEscapeKey = this.handleEscapeKey.bind(this);

		this.init();
	}

	init() {
		this.element.setAttribute('data-text-popup-editor', 'true');
		this.element.style.cursor = 'pointer';
		this.element.style.border = this.element.style.border || '2px dashed transparent';
		this.element.style.transition = 'border-color 0.2s ease';

		this.element.addEventListener('mouseenter', () => {
			if (!this.isEditing) {
				this.element.style.borderColor = '#007cba';
			}
		});

		this.element.addEventListener('mouseleave', () => {
			if (!this.isEditing) {
				this.element.style.borderColor = 'transparent';
			}
		});

		this.element.addEventListener('click', this.handleElementClick, true);
	}

	async handleElementClick(event) {
		event.preventDefault();
		event.stopPropagation();

		if (this.isEditing) {
			return;
		}

		await this.openEditor();
	}

	async openEditor() {
		if (this.isEditing) {
			return;
		}

		this.isEditing = true;
		this.element.style.borderColor = '#007cba';

		this.createModal();

		try {
			await this.initializeCkEditor();
		} catch (error) {
			console.error('Failed to initialise popup text editor', error);
			this.closeEditor();
		}
	}

	async initializeCkEditor() {
		if (!this.modalBody) {
			throw new Error('Modal body element missing for popup text editor.');
		}

		const editableHost = document.createElement('div');
		editableHost.className = 'text-popup-editor__editable';
		Object.assign(editableHost.style, {
			flex: '1 1 auto',
			minHeight: '0',
			padding: '16px',
			overflow: 'auto',
			background: '#fff',
		});
		editableHost.innerHTML = this.element.innerHTML;
		this.modalEditorContainer.appendChild(editableHost);

		this.editorInstance = await TextEditor.create(editableHost, this.config.editorConfig || {});

		this.toolbarElement = this.editorInstance.ui.view.toolbar.element;
		if (this.toolbarElement) {
			this.toolbarElement.classList.add('text-popup-editor__toolbar');
			Object.assign(this.toolbarElement.style, {
				position: 'relative',
				zIndex: '2',
				boxShadow: '0 1px 0 rgba(15, 23, 42, 0.08)',
			});
			this.modalEditorContainer.insertBefore(this.toolbarElement, editableHost);
		}

		// Ensure focus is placed inside the editor once ready.
		this.editorInstance.once('ready', () => {
			setTimeout(() => {
				this.editorInstance.editing.view.focus();
			}, 0);
		});
	}

	createModal() {
		this.modalOverlay = document.createElement('div');
		this.modalOverlay.className = 'text-popup-editor__overlay';
		Object.assign(this.modalOverlay.style, {
			position: 'fixed',
			inset: '0',
			background: 'rgba(0, 0, 0, 0.7)',
			zIndex: OVERLAY_Z_INDEX,
			display: 'flex',
			alignItems: 'center',
			justifyContent: 'center',
			padding: '2rem',
		});

		this.modalContent = document.createElement('div');
		this.modalContent.className = 'text-popup-editor__modal';
		Object.assign(this.modalContent.style, {
			background: '#fff',
			borderRadius: '10px',
			maxWidth: '960px',
			width: '100%',
			maxHeight: '85vh',
			display: 'flex',
			flexDirection: 'column',
			boxShadow: '0 20px 60px rgba(0, 0, 0, 0.35)',
			overflow: 'hidden',
		});

		const modalHeader = document.createElement('div');
		modalHeader.className = 'text-popup-editor__header';
		Object.assign(modalHeader.style, {
			padding: '18px 24px',
			display: 'flex',
			alignItems: 'center',
			justifyContent: 'space-between',
			borderBottom: '1px solid #e0e0e0',
			background: '#f8f9fa',
		});

		const title = document.createElement('h3');
		title.textContent = 'Text Editor';
		Object.assign(title.style, {
			margin: '0',
			fontSize: '18px',
			color: '#1f2937',
		});

		const closeButton = document.createElement('button');
		closeButton.setAttribute('type', 'button');
		closeButton.innerHTML = '&times;';
		closeButton.className = 'text-popup-editor__close';
		Object.assign(closeButton.style, {
			background: 'none',
			border: 'none',
			fontSize: '26px',
			cursor: 'pointer',
			color: '#6b7280',
			lineHeight: '1',
			padding: '0',
			width: '32px',
			height: '32px',
			display: 'flex',
			alignItems: 'center',
			justifyContent: 'center',
		});

		closeButton.addEventListener('click', () => this.closeEditor());

		modalHeader.appendChild(title);
		modalHeader.appendChild(closeButton);

		this.modalBody = document.createElement('div');
		this.modalBody.className = 'text-popup-editor__body';
		Object.assign(this.modalBody.style, {
			padding: '32px',
			flex: '1 1 auto',
			overflow: 'hidden',
			display: 'flex',
			flexDirection: 'column',
		});

		this.modalEditorContainer = document.createElement('div');
		this.modalEditorContainer.className = 'text-popup-editor__container';
		Object.assign(this.modalEditorContainer.style, {
			display: 'flex',
			flexDirection: 'column',
			height: '100%',
			width: '100%',
			overflow: 'hidden',
		});

		this.modalBody.appendChild(this.modalEditorContainer);

		const modalFooter = document.createElement('div');
		modalFooter.className = 'text-popup-editor__footer';
		Object.assign(modalFooter.style, {
			padding: '18px 24px',
			display: 'flex',
			justifyContent: 'flex-end',
			gap: '12px',
			borderTop: '1px solid #e0e0e0',
			background: '#f8f9fa',
		});

		const cancelButton = document.createElement('button');
		cancelButton.type = 'button';
		cancelButton.textContent = 'Cancel';
		cancelButton.className = 'text-popup-editor__cancel';
		Object.assign(cancelButton.style, {
			padding: '10px 18px',
			borderRadius: '6px',
			border: '1px solid #d1d5db',
			background: '#fff',
			color: '#1f2937',
			cursor: 'pointer',
			fontSize: '14px',
		});
		cancelButton.addEventListener('click', () => this.closeEditor(false));

		const saveButton = document.createElement('button');
		saveButton.type = 'button';
		saveButton.textContent = 'Save';
		saveButton.className = 'text-popup-editor__save';
		Object.assign(saveButton.style, {
			padding: '10px 20px',
			borderRadius: '6px',
			border: 'none',
			background: '#2563eb',
			color: '#fff',
			cursor: 'pointer',
			fontSize: '14px',
		});
		saveButton.addEventListener('click', () => this.closeEditor(true));

		modalFooter.appendChild(cancelButton);
		modalFooter.appendChild(saveButton);

		this.modalContent.appendChild(modalHeader);
		this.modalContent.appendChild(this.modalBody);
		this.modalContent.appendChild(modalFooter);
		this.modalOverlay.appendChild(this.modalContent);

		this.modalOverlay.addEventListener('click', (event) => {
			if (event.target === this.modalOverlay) {
				this.closeEditor(false);
			}
		});

		document.addEventListener('keydown', this.handleEscapeKey, true);

		document.body.appendChild(this.modalOverlay);
	}

	handleEscapeKey(event) {
		if (event.key === 'Escape') {
			this.closeEditor(false);
		}
	}

	async closeEditor(shouldSave = false) {
		if (!this.isEditing) {
			return;
		}

		this.isEditing = false;
		this.element.style.borderColor = 'transparent';

		if (shouldSave && this.editorInstance) {
			const newContent = this.editorInstance.getData();
			this.element.innerHTML = newContent;
			
			// Also update the data attribute so getFieldValue can read it correctly
			// Encode HTML entities for storage in the attribute
			const textarea = document.createElement('textarea');
			textarea.textContent = newContent;
			const encodedContent = textarea.innerHTML;
			this.element.setAttribute('data-sdc-prop-text-editor-popup', encodedContent);
			
			// Mark the page as having unsaved changes
			if (window.editor && typeof window.editor.markAsChanged === 'function') {
				window.editor.markAsChanged();
			}
		}

		await this.destroyEditorInstance();
		this.destroyModal();
	}

	async destroyEditorInstance() {
		if (this.editorInstance) {
			try {
				await this.editorInstance.destroy();
			} catch (error) {
				console.warn('Failed to destroy popup text editor instance cleanly.', error);
			}
		}

		this.editorInstance = null;
	}

	destroyModal() {
		document.removeEventListener('keydown', this.handleEscapeKey, true);

		if (this.modalOverlay && this.modalOverlay.parentNode) {
			this.modalOverlay.parentNode.removeChild(this.modalOverlay);
		}

		if (this.toolbarElement && this.toolbarElement.parentNode) {
			this.toolbarElement.parentNode.removeChild(this.toolbarElement);
		}

		this.modalOverlay = null;
		this.modalContent = null;
		this.modalBody = null;
		this.modalEditorContainer = null;
		this.toolbarElement = null;
	}

	destroy() {
		this.closeEditor(false);
		this.element.removeEventListener('click', this.handleElementClick, true);
		this.element.removeAttribute('data-text-popup-editor');
		this.element.style.cursor = '';
		this.element.style.border = '';
		this.element.style.transition = '';
	}
}

export default CustomTextPopupEditor;

