/**
 * @file
 * CKEditor 5 Text Manager for PR Builder
 *
 * This file provides inline editing capabilities for text components
 * using CKEditor 5 with pr Builder-like functionality.
 */

import { TextEditor } from './ckeditor5-config';
import { CustomHtmlEditor } from './CustomHtmlEditor';
import { CustomTextPopupEditor } from './CustomTextPopupEditor';

class CKEditor5TextManager {
	constructor(options = {}) {
		// Prevent multiple instances
		if (window.ckEditor5Manager) {
			return window.ckEditor5Manager;
		}
		
		this.editors = new Map();
		this.isInitialized = false;
		
		// Store permission settings
		this.permissions = options.permissions || {
			hasEditTextEditorFieldPermission: false,
			hasEditHtmlEditorFieldPermission: false,
		};
		
		this.init();
		
		// Store reference globally
		window.ckEditor5Manager = this;
	}

	async init() {
		// Wait for DOM to be ready
		if (document.readyState === 'loading') {
			document.addEventListener('DOMContentLoaded', () => this.initializeEditors());
		} else {
			this.initializeEditors();
		}
	}

	initializeEditors() {
		// Find all elements with the required data attributes for text editors
		if (this.permissions.hasEditTextEditorFieldPermission) {
			const textEditorElements = document.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="text-editor"]:not([data-ckeditor-initialized])');
			textEditorElements.forEach((element) => {
				this.createInlineEditor(element, 'text-editor');
			});

			const popupTextEditors = document.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="text-editor-popup"]:not([data-ckeditor-initialized])');
			popupTextEditors.forEach((element) => {
				this.createInlineEditor(element, 'text-editor-popup');
			});
		} else {
			console.log('Text editor fields disabled - no permission');
		}

		// Find all elements with the required data attributes for HTML editors
		if (this.permissions.hasEditHtmlEditorFieldPermission) {
			const htmlEditorElements = document.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="html-editor"]:not([data-ckeditor-initialized])');
			htmlEditorElements.forEach((element) => {
				this.createInlineEditor(element, 'html-editor');
			});
		} else {
			console.log('HTML editor fields disabled - no permission');
		}

		// Watch for dynamically added elements
		this.observeNewElements();

		this.isInitialized = true;
	}

	async createInlineEditor(element, type) {
		try {
			// Skip if already has an editor
			if (element.hasAttribute('data-ckeditor-initialized')) {
				return;
			}

			// Create a unique ID for the element
			const editorId = element.id || 'sdc-editor-' + Math.random().toString(36).substr(2, 9);
			element.id = editorId;

			// Initialize appropriate editor based on type
			let editor;
			if (type === 'text-editor') {
				// Make the element contenteditable and prepare for CKEditor
				element.setAttribute('contenteditable', 'true');
				element.setAttribute('data-ckeditor-initialized', 'true');
				
				// Initialize CKEditor 5 Inline Editor for text
				editor = await TextEditor.create(element, {});
				
				// Add event listeners for CKEditor
				this.addEditorEventListeners(editor, element);
			} else if (type === 'text-editor-popup') {
				element.setAttribute('data-ckeditor-initialized', 'true');
				editor = new CustomTextPopupEditor(element, {});
			} else if (type === 'html-editor') {
				// Use custom HTML editor that preserves exact structure
				element.setAttribute('data-ckeditor-initialized', 'true');
				
				// Initialize Custom HTML Editor
				editor = new CustomHtmlEditor(element, {
					theme: 'light' // or 'dark' if you prefer
				});
			}

			// Store editor reference
			this.editors.set(editorId, editor);
		} catch (error) {
			console.error('Failed to initialize editor for element:', element, error);
		}
	}

	addEditorEventListeners(editor, element) {
		// Only add CKEditor-specific event listeners for text editors
		if (editor.model && editor.model.document) {
			// Listen for content changes
			editor.model.document.on('change:data', () => {
				console.log('Content changed in element:', element.id, editor.getData());
				
				// Dispatch a custom event to notify that content has changed
				const event = new CustomEvent('ckEditorContentChanged', {
					detail: { element, editor }
				});
				document.dispatchEvent(event);
			});

			// Listen for focus events
			editor.editing.view.document.on('focus', () => {
				element.classList.add('sdc-editing-active');
			});

			// Listen for blur events
			editor.editing.view.document.on('blur', () => {
				element.classList.remove('sdc-editing-active');
			});

			// Listen for editor ready
			editor.on('ready', () => {
				console.log('Editor is ready for element:', element.id);
			});
		}
	}

	// Demo mode - no saving functionality needed

	observeNewElements() {
		// Watch for new elements being added to the DOM
		const observer = new MutationObserver((mutations) => {
			mutations.forEach((mutation) => {
				mutation.addedNodes.forEach((node) => {
					if (node.nodeType === Node.ELEMENT_NODE) {
						// Check if the node itself has the required attributes
						if (
							node.hasAttribute &&
							node.hasAttribute('data-sdc-editable') &&
							node.getAttribute('data-sdc-editable') == 'true' &&
							node.hasAttribute('data-sdc-type')
						) {
							const editorType = node.getAttribute('data-sdc-type');
							if (editorType === 'text-editor' && this.permissions.hasEditTextEditorFieldPermission) {
								this.createInlineEditor(node, editorType);
							} else if (editorType === 'html-editor' && this.permissions.hasEditHtmlEditorFieldPermission) {
								this.createInlineEditor(node, editorType);
							}
						}

						// Check child elements for both editor types
						if (this.permissions.hasEditTextEditorFieldPermission) {
							const textEditorChildren = node.querySelectorAll && node.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="text-editor"]');
							if (textEditorChildren) {
								textEditorChildren.forEach((child) => this.createInlineEditor(child, 'text-editor'));
							}

							const popupTextEditorChildren = node.querySelectorAll && node.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="text-editor-popup"]');
							if (popupTextEditorChildren) {
								popupTextEditorChildren.forEach((child) => this.createInlineEditor(child, 'text-editor-popup'));
							}
						}

						if (this.permissions.hasEditHtmlEditorFieldPermission) {
							const htmlEditorChildren = node.querySelectorAll && node.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="html-editor"]');
							if (htmlEditorChildren) {
								htmlEditorChildren.forEach((child) => this.createInlineEditor(child, 'html-editor'));
							}
						}
					}
				});
			});
		});

		observer.observe(document.body, {
			childList: true,
			subtree: true,
		});
	}

	// Method to destroy editors (useful for cleanup)
	destroyEditor(elementId) {
		const editor = this.editors.get(elementId);
		if (editor) {
			// Handle both CKEditor and Custom HTML Editor
			if (editor.destroy) {
				editor.destroy();
			}
			this.editors.delete(elementId);
		}
	}

	// Method to get editor instance
	getEditor(elementId) {
		return this.editors.get(elementId);
	}

	// Method to enable/disable editing mode
	setEditingMode(enabled) {
		const textEditorElements = document.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="text-editor"]');
		textEditorElements.forEach((element) => {
			if (enabled) {
				element.classList.add('sdc-editing-enabled');
			} else {
				element.classList.remove('sdc-editing-enabled');
			}
		});

		const popupTextEditorElements = document.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="text-editor-popup"]');
		popupTextEditorElements.forEach((element) => {
			if (enabled) {
				element.classList.add('sdc-editing-enabled');
			} else {
				element.classList.remove('sdc-editing-enabled');
			}
		});

		const htmlEditorElements = document.querySelectorAll('[data-sdc-editable="true"][data-sdc-type="html-editor"]');
		htmlEditorElements.forEach((element) => {
			if (enabled) {
				element.classList.add('sdc-editing-enabled');
			} else {
				element.classList.remove('sdc-editing-enabled');
			}
		});
	}
}
// Export for use in other scripts and as default export
export default CKEditor5TextManager;
