import { Plugin, View, ButtonView } from 'ckeditor5';
import { EditorView, basicSetup } from 'codemirror';
import { EditorState } from '@codemirror/state';
import { html } from '@codemirror/lang-html';
import { oneDark } from '@codemirror/theme-one-dark';
import js_beautify from 'js-beautify';

/**
 * HTML CodeMirror Plugin for CKEditor 5
 * Provides a modal-based HTML source editor using CodeMirror 6
 */
export default class HtmlCodeMirrorPlugin extends Plugin {
	static get pluginName() {
		return 'HtmlCodeMirror';
	}

	/**
	 * Initialize the plugin
	 */
	init() {
		this._editing = false;
		this._codeEditorView = null;
		this._editorContainer = null;

		// Add a toggle button to the toolbar
		this.editor.ui.componentFactory.add('inline-html-source', (locale) => {
			const button = new ButtonView(locale);

		button.set({
			label: 'Source',
			icon: `<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
				<path d="M2 4a2 2 0 0 1 2-2h12a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V4zm2-1a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1V4a1 1 0 0 0-1-1H4z"/>
				<path d="M6.5 6.5a.5.5 0 0 1 .5-.5h6a.5.5 0 0 1 0 1H7a.5.5 0 0 1-.5-.5zM6.5 8.5a.5.5 0 0 1 .5-.5h4a.5.5 0 0 1 0 1H7a.5.5 0 0 1-.5-.5zM6.5 10.5a.5.5 0 0 1 .5-.5h5a.5.5 0 0 1 0 1H7a.5.5 0 0 1-.5-.5zM6.5 12.5a.5.5 0 0 1 .5-.5h3a.5.5 0 0 1 0 1H7a.5.5 0 0 1-.5-.5z"/>
				<path d="M11 8.5a.5.5 0 0 1 .5-.5h2.5a.5.5 0 0 1 0 1H11.5a.5.5 0 0 1-.5-.5z"/>
				<path d="M11 10.5a.5.5 0 0 1 .5-.5h1.5a.5.5 0 0 1 0 1h-1.5a.5.5 0 0 1-.5-.5z"/>
			</svg>`,
			tooltip: true,
			isToggleable: true,
		});

			// Toggle CodeMirror mode on click
			this.listenTo(button, 'execute', () => {
				if (this._editing) {
					this.exitCodeMirrorMode();
					button.isOn = false;
				} else {
					this.enterCodeMirrorMode();
					button.isOn = true;
				}
			});

			return button;
		});
	}

	/**
	 * Enter CodeMirror editing mode
	 */
	enterCodeMirrorMode() {
		if (this._editing) return;

		const editor = this.editor;

		// Create modal
		this._createModal(editor);
		this._editing = true;
	}

	/**
	 * Exit CodeMirror editing mode
	 */
	exitCodeMirrorMode() {
		if (!this._editing) return;

		this._closeModal();
		this._editing = false;
	}

	/**
	 * Create the modal dialog for HTML editing
	 * @param {Object} editor - The CKEditor instance
	 */
	_createModal(editor) {
		// Create modal overlay
		this._modalOverlay = document.createElement('div');
		this._modalOverlay.classList.add('ck-codemirror-modal-overlay');

		// Create modal content
		this._modalContent = document.createElement('div');
		this._modalContent.classList.add('ck-codemirror-modal-content');

		// Create modal header
		const modalHeader = document.createElement('div');
		modalHeader.classList.add('ck-codemirror-modal-header');
		modalHeader.innerHTML = `
            <h3>HTML Source Editor</h3>
            <button class="ck-codemirror-modal-close" type="button">&times;</button>
        `;

		// Create modal body with CodeMirror
		const modalBody = document.createElement('div');
		modalBody.classList.add('ck-codemirror-modal-body');

		this._codeEditorView = new CodeEditorView(editor.locale, {
			syntaxLanguage: 'html',
			theme: 'dark',
		});
		this._codeEditorView.render();
		modalBody.appendChild(this._codeEditorView.element);

		// Create modal footer with buttons
		const modalFooter = document.createElement('div');
		modalFooter.classList.add('ck-codemirror-modal-footer');
		modalFooter.innerHTML = `
            <button class="ck-codemirror-modal-cancel" type="button">Cancel</button>
            <button class="ck-codemirror-modal-save" type="button">Save Changes</button>
        `;

		// Assemble modal
		this._modalContent.appendChild(modalHeader);
		this._modalContent.appendChild(modalBody);
		this._modalContent.appendChild(modalFooter);
		this._modalOverlay.appendChild(this._modalContent);

		// Add to document
		document.body.appendChild(this._modalOverlay);

		// Set initial data (prettified)
		this._codeEditorView.setData(js_beautify.html_beautify(editor.getData()));

		// Add event listeners
		this._setupModalEvents(editor);
	}

	/**
	 * Setup event listeners for the modal
	 * @param {Object} editor - The CKEditor instance
	 */
	_setupModalEvents(editor) {
		const closeBtn = this._modalContent.querySelector('.ck-codemirror-modal-close');
		const cancelBtn = this._modalContent.querySelector('.ck-codemirror-modal-cancel');
		const saveBtn = this._modalContent.querySelector('.ck-codemirror-modal-save');

		// Close modal handlers
		const closeModal = () => {
			this._closeModal();
			this._editing = false;
		};

		closeBtn.addEventListener('click', closeModal);
		cancelBtn.addEventListener('click', closeModal);

		// Overlay click to close
		this._modalOverlay.addEventListener('click', (e) => {
			if (e.target === this._modalOverlay) {
				closeModal();
			}
		});

		// Save handler
		saveBtn.addEventListener('click', () => {
			editor.setData(js_beautify.html_beautify(this._codeEditorView.getData()));
			closeModal();
		});

		// ESC key to close
		this._handleEscape = (e) => {
			if (e.key === 'Escape') {
				closeModal();
			}
		};
		document.addEventListener('keydown', this._handleEscape);

		// Focus the CodeMirror editor
		setTimeout(() => {
			this._codeEditorView.focus();
		}, 100);
	}

	/**
	 * Close the modal and cleanup resources
	 */
	_closeModal() {
		if (this._modalOverlay) {
			document.body.removeChild(this._modalOverlay);
			this._modalOverlay = null;
			this._modalContent = null;
		}

		if (this._codeEditorView) {
			this._codeEditorView.destroy();
			this._codeEditorView = null;
		}

		if (this._handleEscape) {
			document.removeEventListener('keydown', this._handleEscape);
			this._handleEscape = null;
		}
	}

	/**
	 * Check if currently in editing mode
	 * @returns {boolean} True if editing
	 */
	isEditing() {
		return this._editing;
	}

	/**
	 * Get the CodeMirror editor instance
	 * @returns {Object|null} The CodeMirror instance or null
	 */
	getCodeMirrorEditor() {
		return this._codeEditorView ? this._codeEditorView.getCodeMirrorInstance() : null;
	}

	/**
	 * Cleanup when plugin is destroyed
	 */
	destroy() {
		if (this._editing) {
			this.exitCodeMirrorMode();
		}
		super.destroy();
	}
}

/**
 * Internal CodeMirror 6 wrapper class
 * Provides a CKEditor 5 View interface for CodeMirror 6
 */
class CodeEditorView extends View {
	/**
	 * Constructor
	 * @param {Object} locale - The locale object
	 * @param {Object} options - Configuration options
	 */
	constructor(locale, options) {
		super(locale);
		this._options = options;
		this._editorView = null;
		this._extensions = [];

		this.setTemplate({
			tag: 'div',
			attributes: {
				class: ['ck', 'ck-reset_all-excluded', 'ck-code-editor'],
				tabindex: -1,
			},
		});
	}

	/**
	 * Render the CodeMirror editor
	 */
	render() {
		super.render();

		// Create extensions array based on options
		this._extensions = [
			basicSetup,
			html(),
			EditorView.theme({
				'.cm-content': {
					fontSize: '14px',
					fontFamily: 'Monaco, Menlo, "Ubuntu Mono", monospace',
					lineHeight: '1.5',
					padding: '12px',
				},
				'.cm-focused': {
					outline: 'none',
				},
				'.cm-editor': {
					border: '1px solid #ddd',
					borderRadius: '4px',
					height: '100%',
				},
				'.cm-scroller': {
					fontFamily: 'Monaco, Menlo, "Ubuntu Mono", monospace',
				},
			}),
		];

		// Add theme if specified
		if (this._options.theme === 'dark') {
			this._extensions.push(oneDark);
		}

		// Create initial state
		const startState = EditorState.create({
			doc: '',
			extensions: this._extensions,
		});

		// Create editor view
		this._editorView = new EditorView({
			state: startState,
			parent: this.element,
		});
	}

	/**
	 * Destroy the editor and cleanup resources
	 */
	destroy() {
		if (this._editorView) {
			this._editorView.destroy();
			this._editorView = null;
		}
		super.destroy();
	}

	/**
	 * Get the current editor content
	 * @returns {string} The editor content
	 */
	getData() {
		return this._editorView ? this._editorView.state.doc.toString() : '';
	}

	/**
	 * Set the editor content
	 * @param {string} data - The content to set
	 */
	setData(data) {
		if (this._editorView) {
			const transaction = this._editorView.state.update({
				changes: {
					from: 0,
					to: this._editorView.state.doc.length,
					insert: data,
				},
			});
			this._editorView.dispatch(transaction);
		}
	}

	/**
	 * Get the CodeMirror editor instance
	 * @returns {Object} The CodeMirror editor view
	 */
	getCodeMirrorInstance() {
		return this._editorView;
	}

	/**
	 * Focus the editor
	 */
	focus() {
		if (this._editorView) {
			this._editorView.focus();
		}
	}
}
