/**
 * @file
 * DataProcessor class for PR Builder
 *
 * This class handles processing different data types from SDC components.
 * It extracts data from elements with sdc-data-field attributes based on their data-sdc-type.
 */

import { dataFieldHandlers, defaultDataFieldHandler } from './fieldHandlers/dataFieldHandlers.js';

export class DataProcessor {
  constructor() {
    this.typeHandlers = { ...dataFieldHandlers };
    this.defaultHandler = defaultDataFieldHandler;
  }

  /**
   * Process all SDC components and extract their field data
   * @param {Element} container - The container to search within (defaults to document.body)
   * @returns {Array} Array of component data objects
   */
  processComponents(container) {
    const components = [];
    
    // Find all top-level elements with data-sdc-component (not nested in other components)
    const sdcComponents = this.getTopLevelComponents(container);
    
    sdcComponents.forEach((component, index) => {
      const componentData = this.processComponent(component, index);
      if (componentData) {
        components.push(componentData);
      }
    });
    
    return components;
  }

  /**
   * Get only top-level components (not nested within other components)
   * @param {Element} container - The container to search within
   * @returns {Array} Array of top-level component elements
   */
  getTopLevelComponents(container) {
    const allComponents = Array.from(container.querySelectorAll('[data-sdc-component]'));
    const topLevelComponents = [];
    
    allComponents.forEach((component) => {
      // Check if this component has a parent component
      let parent = component.parentElement;
      let hasParentComponent = false;
      
      while (parent && parent !== container) {
        if (parent.hasAttribute('data-sdc-component')) {
          hasParentComponent = true;
          break;
        }
        parent = parent.parentElement;
      }
      
      // Only add if it doesn't have a parent component
      if (!hasParentComponent) {
        topLevelComponents.push(component);
      }
    });
    
    return topLevelComponents;
  }

  /**
   * Process a single SDC component
   * @param {Element} component - The component elemen
   * t
   * @param {number} index - The component index
   * @returns {Object|null} Component data object or null if no fields found
   */
  processComponent(component, index = 0) {
    const componentData = {
      index: index,
      componentType: component.getAttribute('data-sdc-component'),
      fields: {}
    };

    // Find all elements with data-sdc-field within this component (including the component itself)
    let fieldElements = component.querySelectorAll('[data-sdc-field]');
    
    // Also check if the component itself has data-sdc-field
    if (component.hasAttribute('data-sdc-field')) {
      // Add the component itself to the fieldElements if it has the attribute
      fieldElements = [component, ...fieldElements];
    }

    
    // Process all field elements
    fieldElements.forEach((fieldElement) => {
      const fieldName = fieldElement.getAttribute('data-sdc-field');
      const fieldType = fieldElement.getAttribute('data-sdc-type');
      
      if (fieldName && fieldType) {
        const fieldValue = this.processField(fieldElement, fieldType);
        componentData.fields[fieldName] = {
          type: fieldType,
          value: fieldValue
        };
      }
    });
    
    // Check if this is a row-block component - process nested components
    if (component.hasAttribute('data-nested-container')) {
      const childrenData = this.processRowChildren(component);
      componentData.fields.children = {
        type: 'nested-components',
        value: childrenData
      };
    }
    
    // Only skip if we have no fields AND it's not a container component
    if (Object.keys(componentData.fields).length === 0 && !component.hasAttribute('data-nested-container')) {
      return null; // Skip components with no fields
    }
    
    return componentData;
  }

  /**
   * Process children components within a row container
   * @param {Element} rowComponent - The row component element
   * @returns {Object} Object with column data and their child components
   */
  processRowChildren(rowComponent) {
    const columns = Array.from(rowComponent.querySelectorAll('[data-drop-container="true"]')).filter((column) => {
      const parentComponent = column.closest('[data-sdc-component]');
      return parentComponent === rowComponent;
    });
    const columnData = [];
    
    columns.forEach((column, columnIndex) => {
      const columnInfo = {
        columnIndex: columnIndex,
        components: []
      };
      
      // Get the index attribute from the drop container
      const columnIndexAttr = this.getIndexAttribute(column);
      
      // If the drop container has an index attribute, look for elements with the same index
      if (columnIndexAttr) {
        const { attrName, attrValue } = columnIndexAttr;
        
        // First, try to find a parent wrapper with the same index attribute (accordion pattern)
        let matchingWrapper = null;
        let parentWrapper = column.parentElement;
        while (parentWrapper && parentWrapper !== rowComponent) {
          if (parentWrapper.hasAttribute(attrName)) {
            const wrapperIndex = parentWrapper.getAttribute(attrName);
            if (wrapperIndex === attrValue) {
              matchingWrapper = parentWrapper;
              break;
            }
          }
          parentWrapper = parentWrapper.parentElement;
        }
        
        // If no parent wrapper found, search siblings in the entire component (tabs pattern)
        if (!matchingWrapper) {
          const allElementsWithIndex = rowComponent.querySelectorAll(`[${attrName}="${attrValue}"]`);
          allElementsWithIndex.forEach((element) => {
            if (element === column) {
              return;
            }
            
            // Process fields in this matching element
            const fieldElements = element.querySelectorAll('[data-sdc-field]');
            fieldElements.forEach((fieldElement) => {
              // Skip fields that are inside nested components
              let parent = fieldElement.parentElement;
              let insideNestedComponent = false;
              while (parent && parent !== element) {
                if (parent.hasAttribute('data-sdc-component')) {
                  insideNestedComponent = true;
                  break;
                }
                parent = parent.parentElement;
              }
              
              if (!insideNestedComponent) {
                const fieldName = fieldElement.getAttribute('data-sdc-field');
                const fieldType = fieldElement.getAttribute('data-sdc-type');
                
                if (fieldName && fieldType) {
                  const fieldValue = this.processField(fieldElement, fieldType);
                  columnInfo[fieldName] = fieldValue;
                }
              }
            });
          });
        } else {
          // Process fields in the parent wrapper (accordion pattern)
          const fieldElements = matchingWrapper.querySelectorAll('[data-sdc-field]');
          fieldElements.forEach((fieldElement) => {
            // Skip fields that are inside nested components or inside the drop container
            if (column.contains(fieldElement)) {
              return;
            }
            
            let parent = fieldElement.parentElement;
            let insideNestedComponent = false;
            while (parent && parent !== matchingWrapper) {
              if (parent.hasAttribute('data-sdc-component')) {
                insideNestedComponent = true;
                break;
              }
              parent = parent.parentElement;
            }
            
            if (!insideNestedComponent) {
              const fieldName = fieldElement.getAttribute('data-sdc-field');
              const fieldType = fieldElement.getAttribute('data-sdc-type');
              
              if (fieldName && fieldType) {
                const fieldValue = this.processField(fieldElement, fieldType);
                columnInfo[fieldName] = fieldValue;
              }
            }
          });
        }
      }
      
      // Also check for any fields within this drop container itself
      const fieldElements = column.querySelectorAll('[data-sdc-field]');
      fieldElements.forEach((fieldElement) => {
        // Skip fields that are inside nested components
        let parent = fieldElement.parentElement;
        let insideNestedComponent = false;
        while (parent && parent !== column) {
          if (parent.hasAttribute('data-sdc-component')) {
            insideNestedComponent = true;
            break;
          }
          parent = parent.parentElement;
        }
        
        if (!insideNestedComponent) {
          const fieldName = fieldElement.getAttribute('data-sdc-field');
          const fieldType = fieldElement.getAttribute('data-sdc-type');
          
          if (fieldName && fieldType) {
            const fieldValue = this.processField(fieldElement, fieldType);
            columnInfo[fieldName] = fieldValue;
          }
        }
      });
      
      // Find direct child components in this column
      const childComponents = Array.from(column.querySelectorAll('[data-sdc-component]')).filter((child) => {
        // Only include direct children of this column (not nested deeper)
        let parent = child.parentElement;
        while (parent && parent !== column) {
          if (parent.hasAttribute('data-sdc-component')) {
            return false; // This is nested in another component
          }
          parent = parent.parentElement;
        }
        return true;
      });
      
      childComponents.forEach((childComponent, childIndex) => {
        const childData = this.processComponent(childComponent, childIndex);
        if (childData) {
          columnInfo.components.push(childData);
        }
      });
      
      columnData.push(columnInfo);
    });
    
    return columnData;
  }

  /**
   * Get the index attribute from an element (e.g., data-accordion-item-index, data-tab-index, etc.)
   * @param {Element} element - The element to check
   * @returns {Object|null} Object with attrName and attrValue, or null if not found
   */
  getIndexAttribute(element) {
    const attributes = element.attributes;
    for (let i = 0; i < attributes.length; i++) {
      const attr = attributes[i];
      // Look for any data-*-index attribute
      if (attr.name.startsWith('data-') && attr.name.endsWith('-index')) {
        return {
          attrName: attr.name,
          attrValue: attr.value
        };
      }
    }
    return null;
  }

  /**
   * Process a single field based on its type
   * @param {Element} element - The field element
   * @param {string} type - The field type
   * @returns {*} The processed field value
   */
  processField(element, type) {
    const handler = this.typeHandlers[type];
    
    if (handler) {
      return handler(element);
    } else {
      console.warn(`Unknown field type: ${type}`);
      return this.processDefault(element);
    }
  }

  /**
   * Process html-editor type fields
   * @param {Element} element - The field element
   * @returns {string} The HTML content
   */
  /**
   * Default processor for unknown types
   * @param {Element} element - The field element
   * @returns {string} The element's text content
   */
  processDefault(element) {
    if (typeof this.defaultHandler === 'function') {
      return this.defaultHandler(element);
    }
    return element.textContent || element.innerText || '';
  }

  /**
   * Add a new type handler
   * @param {string} type - The field type
   * @param {Function} handler - The handler function
   */
  addTypeHandler(type, handler) {
    this.typeHandlers[type] = handler;
  }

  /**
   * Remove a type handler
   * @param {string} type - The field type to remove
   */
  removeTypeHandler(type) {
    delete this.typeHandlers[type];
  }

  /**
   * Get all available type handlers
   * @returns {Array} Array of type names
   */
  getAvailableTypes() {
    return Object.keys(this.typeHandlers);
  }
}
