import DraggableElement from './class/DraggableElement';
import DropZone from './class/DropZone';
import SDCService from './api/SDCService';

const DropType = {
	SDC: 'sdc',
	TOOLBAR: 'toolbar',
};

class DragAndDrop {
	// Variables
	draggingElement = null; // Must be a DraggableElement instance
	dropZones = []; // Array to track active DropZone instances
	sdcService = null; // SDC API service
	editor = null; // Reference to the editor instance

	// Initialize the drag and drop manager
	constructor(editor = null) {
		this.editor = editor;
		this.sdcService = new SDCService();
		this.init();
	}

	getDraggingComponentName() {
		if (!this.draggingElement) {
			return null;
		}

		if (this.draggingElement.dropType === DropType.TOOLBAR) {
			const name = this.draggingElement.element?.getAttribute('data-sdc-name') || null;
			console.log('[getDraggingComponentName] Toolbar element name:', name);
			return name;
		}

		const componentElement = this.draggingElement.element?.querySelector('[data-sdc-component]');
		const name = componentElement?.getAttribute('data-sdc-component') || null;
		console.log('[getDraggingComponentName] SDC component name:', name);
		return name;
	}

	getAllowedComponents(container) {
		if (!container) {
			return null;
		}

		if (Object.prototype.hasOwnProperty.call(container, '__sdcInlineEditorAllowedComponents')) {
			return container.__sdcInlineEditorAllowedComponents;
		}

		const allowedAttr = container.getAttribute('data-allowed-components');

		if (!allowedAttr) {
			container.__sdcInlineEditorAllowedComponents = null;
			return null;
		}

		let allowedList = null;

		try {
			const parsed = JSON.parse(allowedAttr);
			if (Array.isArray(parsed)) {
				allowedList = parsed.map((value) => String(value).trim()).filter(Boolean);
			}
		} catch (error) {
			allowedList = allowedAttr.split(',').map((value) => value.trim()).filter(Boolean);
		}

		container.__sdcInlineEditorAllowedComponents = allowedList && allowedList.length > 0 ? allowedList : null;
		return container.__sdcInlineEditorAllowedComponents;
	}

	isComponentAllowedInContainer(container, componentName) {
		if (!container || !componentName) {
			return true;
		}

		const allowedComponents = this.getAllowedComponents(container);

		if (!allowedComponents || allowedComponents.length === 0) {
			return true;
		}

		return allowedComponents.includes(componentName);
	}

	isTableCell(container) {
		if (!container) {
			return false;
		}
		
		return container.classList.contains('table-cell-content') && 
		       container.hasAttribute('data-locked') && 
		       container.getAttribute('data-locked') === 'true' &&
		       container.hasAttribute('data-drop-container');
	}

	isImageComponent(componentName) {
		return componentName === 'sdc_inline_editor:single-image-block';
	}

	isDropAllowed(targetWrapper, position, componentName) {
		if (!componentName) {
			return true;
		}

		let container = null;

		if (position === 'inside') {
			container = targetWrapper;
			
			if (this.isTableCell(container) && this.isImageComponent(componentName)) {
				return true;
			}
		} else if (targetWrapper) {
			container = targetWrapper.parentNode;
		}

		if (container && !container.hasAttribute('data-drop-container')) {
			const closestContainer = container.closest?.('[data-drop-container="true"]');
			if (closestContainer) {
				container = closestContainer;
			}
		}

		return this.isComponentAllowedInContainer(container, componentName);
	}

	init() {
		this.setupDragEventListeners();
		this.setupToolbarUpdateListener();
		
		document.addEventListener('toolbarPopupOpened', () => {
			this.setupDragEventListeners();
		});
	}

	setupDragEventListeners() {
		const moveIcons = document.querySelectorAll('.sdc-inline-editor-move-icon');
		const toolbarIcons = document.querySelectorAll('.sdc-inline-editor-toolbar-menu-item');

		moveIcons.forEach((icon) => {
			// Set the drop type for move icons (default SDC)
			icon.setAttribute('data-drop-type', DropType.SDC);
			icon.removeEventListener('mousedown', this.boundHandleDragStart);
			if (!this.boundHandleDragStart) {
				this.boundHandleDragStart = this.handleDragStart.bind(this);
			}
			icon.addEventListener('mousedown', this.boundHandleDragStart);
		});

		toolbarIcons.forEach((icon) => {
			// Make toolbar items draggable
			icon.setAttribute('draggable', 'true');
			// Store the drop type as a data attribute
			icon.setAttribute('data-drop-type', DropType.TOOLBAR);
			icon.style.cursor = 'grab';
			icon.style.pointerEvents = 'auto';
			
			if (!this.boundHandleDragStart) {
				this.boundHandleDragStart = this.handleDragStart.bind(this);
			}
			icon.removeEventListener('mousedown', this.boundHandleDragStart);
			icon.addEventListener('mousedown', this.boundHandleDragStart);
		});
	}	onMouseMove(event) {
		const inCancelZone = this.checkIfInCancelZone(event);
		
		if (inCancelZone) {
			this.cleanupDropZones();
		} else {
			this.updateDropZones(event);
		}
	}

	checkIfInCancelZone(event) {
		const toolbar = document.querySelector('.sdc-inline-editor-toolbar-container');
		if (!toolbar) return false;
		
		const toolbarRect = toolbar.getBoundingClientRect();
		let cancelZoneTop = toolbarRect.top;
		
		const folderPopup = document.querySelector('.sdc-inline-editor-folder-popup');
		if (folderPopup) {
			const popupRect = folderPopup.getBoundingClientRect();
			cancelZoneTop = Math.min(toolbarRect.top, popupRect.top);
		}
		
		return event.clientY >= cancelZoneTop;
	}

	onDrop(event) {
		this.handleDrop(event);
		this.cleanupDropZones();

		// Remove dragging class from the dragged element
		if (this.draggingElement?.element) {
			this.draggingElement.element.classList.remove('dragging');
		}
	}

	handleDragStart(event) {
		event.preventDefault();

		// Get the drop type from the data attribute, fallback to parameter
		const elementDropType = event.currentTarget.getAttribute('data-drop-type') || DropType.SDC;

		// Find the actual block wrapper to drag
		let block = null;
		if (elementDropType == DropType.SDC) {
			block = event.currentTarget.closest('.block-wrapper');
			if (!block) return;
		} else if (elementDropType == DropType.TOOLBAR) {
			// For toolbar items, the element itself is what we want to drag
			block = event.currentTarget.closest('.sdc-inline-editor-toolbar-menu-item');
			if (!block) return;
		}

		// Add dragging class to the element being dragged
		block.classList.add('dragging');

		// Bind the callbacks to maintain proper context
		const boundOnMouseMove = this.onMouseMove.bind(this);
		const boundOnDrop = this.onDrop.bind(this);

		// Pass the block (not the icon) to our class
		const draggable = new DraggableElement(block, boundOnMouseMove, boundOnDrop, elementDropType);
		// Store the drop type with the draggable element
		draggable.dropType = elementDropType;
		draggable.start();
		this.draggingElement = draggable;
	}

	setupToolbarUpdateListener() {
		// Listen for toolbar updates to refresh event listeners
		document.addEventListener('toolbarUpdated', () => {
			console.log('Toolbar updated, refreshing drag and drop event listeners');
			this.setupDragEventListeners();
		});
	}

	refresh() {
		// Re-setup event listeners for any new move icons
		this.setupDragEventListeners();
	}

	/**
	 * Update drop zones based on current mouse position
	 */
	updateDropZones(event) {
		// Clear existing drop zones
		this.cleanupDropZones();

		const draggingComponentName = this.getDraggingComponentName();

		// Find all block wrappers (excluding the one being dragged)
		const blockWrappers = Array.from(document.querySelectorAll('.block-wrapper')).filter((wrapper) => !wrapper.classList.contains('dragging'));
		
		// Find all row columns (potential drop containers).
		// Exclude any containers that are inside the dragging ghost element or inside the element being dragged.
		// Also exclude containers that are explicitly locked (data-locked="true").
		// EXCEPTION: Allow table cells when dragging image components
		const isImageBeingDragged = this.isImageComponent(draggingComponentName);
		
		const rowColumns = Array.from(document.querySelectorAll('[data-drop-container="true"]')).filter((col) => {
			const lockedAttr = col.getAttribute('data-locked');
			if (lockedAttr === 'true') {
				// Allow table cells to accept images
				const isTableCell = this.isTableCell(col);
				if (isImageBeingDragged && isTableCell) {
					return true;
				}
				return false;
			}
			// Check if this column is inside a ghost element
			if (col.closest('.dragging-ghost')) {
				return false;
			}
			// Check if this column is inside an element with the dragging class
			if (col.closest('.dragging')) {
				return false;
			}
			if (draggingComponentName && !this.isComponentAllowedInContainer(col, draggingComponentName)) {
				return false;
			}

			return true;
		});

		// If no block wrappers exist, check for empty row columns or create fallback
		if (blockWrappers.length == 0) {
			// Check if there are empty row columns
			const emptyRows = rowColumns.filter(col => col.querySelectorAll('.block-wrapper').length === 0);
			if (emptyRows.length > 0) {
				this.createRowColumnDropZones(emptyRows, event);
			} else {
				this.createFallbackDropZone();
			}
			console.log('Fallback drop zone created for empty page');
			return;
		}

		const mouseY = event.clientY;
		const mouseX = event.clientX;
		let nearestDropPosition = null;
		let nearestDistance = Infinity;

		// Check if mouse is over a row column
		let hoveredColumn = null;
		rowColumns.forEach((column) => {
			const rect = column.getBoundingClientRect();
			if (mouseX >= rect.left && mouseX <= rect.right && 
				mouseY >= rect.top && mouseY <= rect.bottom) {
				hoveredColumn = column;
			}
		});

		// Check if the hovered column uses horizontal flex layout
		let hoveredColumnIsHorizontalFlex = false;
		if (hoveredColumn) {
			const computedStyle = window.getComputedStyle(hoveredColumn);
			const display = computedStyle.display;
			const flexDirection = computedStyle.flexDirection;
			
			if ((display === 'flex' || display === 'inline-flex') && 
				(flexDirection === 'row' || flexDirection === 'row-reverse')) {
				hoveredColumnIsHorizontalFlex = true;
			}
		}

		if (hoveredColumn) {
			// Get block wrappers within this column
			let columnWrappers = Array.from(hoveredColumn.querySelectorAll('.block-wrapper'))
				.filter((wrapper) => !wrapper.classList.contains('dragging'));

			if (draggingComponentName) {
				columnWrappers = columnWrappers.filter((wrapper) => {
					const container = wrapper.parentElement?.closest('[data-drop-container="true"]') || wrapper.parentElement;
					return this.isComponentAllowedInContainer(container, draggingComponentName);
				});
			}
			
			if (columnWrappers.length === 0) {
				// Empty column - create inside drop zone
				// OR table cell (always show drop zone for table cells when dragging images)
				const isTableCellWithImage = this.isTableCell(hoveredColumn) && isImageBeingDragged;
				
				if (isTableCellWithImage || (!draggingComponentName || this.isDropAllowed(hoveredColumn, 'inside', draggingComponentName))) {
					const dropZone = new DropZone(hoveredColumn, 'inside');
					dropZone.isColumnDropZone = true;
					this.dropZones.push(dropZone);
				}

				return;
			} else {
				// Column has wrappers - find nearest position within column
				columnWrappers.forEach((wrapper) => {
					const rect = wrapper.getBoundingClientRect();
					const wrapperTop = rect.top;
					const wrapperBottom = rect.bottom;
					const wrapperLeft = rect.left;
					const wrapperRight = rect.right;
					
					if (hoveredColumnIsHorizontalFlex) {
						// For horizontal flex containers, use left/right distances
						const distanceToLeft = Math.abs(mouseX - wrapperLeft);
						if (distanceToLeft < nearestDistance && (!draggingComponentName || this.isDropAllowed(wrapper, 'above', draggingComponentName))) {
							nearestDistance = distanceToLeft;
							nearestDropPosition = {
								wrapper: wrapper,
								position: 'above', // 'above' maps to 'left' in horizontal flex
								isHorizontal: true,
							};
						}

						const distanceToRight = Math.abs(mouseX - wrapperRight);
						if (distanceToRight < nearestDistance && (!draggingComponentName || this.isDropAllowed(wrapper, 'below', draggingComponentName))) {
							nearestDistance = distanceToRight;
							nearestDropPosition = {
								wrapper: wrapper,
								position: 'below', // 'below' maps to 'right' in horizontal flex
								isHorizontal: true,
							};
						}
					} else {
						// For vertical layouts (default), use top/bottom distances
						const distanceToTop = Math.abs(mouseY - wrapperTop);
						if (distanceToTop < nearestDistance && (!draggingComponentName || this.isDropAllowed(wrapper, 'above', draggingComponentName))) {
							nearestDistance = distanceToTop;
							nearestDropPosition = {
								wrapper: wrapper,
								position: 'above',
								isHorizontal: false,
							};
						}

						const distanceToBottom = Math.abs(mouseY - wrapperBottom);
						if (distanceToBottom < nearestDistance && (!draggingComponentName || this.isDropAllowed(wrapper, 'below', draggingComponentName))) {
							nearestDistance = distanceToBottom;
							nearestDropPosition = {
								wrapper: wrapper,
								position: 'below',
								isHorizontal: false,
							};
						}
					}
				});
			}
		} else {
			// Not over a column - use standard logic for all block wrappers
			blockWrappers.forEach((wrapper) => {
				const container = wrapper.parentElement?.closest('[data-drop-container="true"]');
				if (draggingComponentName && container && !this.isComponentAllowedInContainer(container, draggingComponentName)) {
					return;
				}

				const rect = wrapper.getBoundingClientRect();
				const wrapperTop = rect.top;
				const wrapperBottom = rect.bottom;

				// Calculate distance to top edge (above position)
				const distanceToTop = Math.abs(mouseY - wrapperTop);
				if (distanceToTop < nearestDistance && (!draggingComponentName || this.isDropAllowed(wrapper, 'above', draggingComponentName))) {
					nearestDistance = distanceToTop;
					nearestDropPosition = {
						wrapper: wrapper,
						position: 'above',
						isHorizontal: false,
					};
				}

				// Calculate distance to bottom edge (below position)
				const distanceToBottom = Math.abs(mouseY - wrapperBottom);
				if (distanceToBottom < nearestDistance && (!draggingComponentName || this.isDropAllowed(wrapper, 'below', draggingComponentName))) {
					nearestDistance = distanceToBottom;
					nearestDropPosition = {
						wrapper: wrapper,
						position: 'below',
						isHorizontal: false,
					};
				}
			});
		}

		// Create drop zone for the nearest position
		if (nearestDropPosition) {
			const dropZone = new DropZone(
				nearestDropPosition.wrapper, 
				nearestDropPosition.position,
				nearestDropPosition.isHorizontal || false
			);
			this.dropZones.push(dropZone);
		}
	}
	/**
	 * Create drop zones for empty row columns
	 */
	createRowColumnDropZones(rows, event) {
		const draggingComponentName = this.getDraggingComponentName();
		const availableRows = draggingComponentName
			? rows.filter((row) => this.isComponentAllowedInContainer(row, draggingComponentName))
			: rows;

		if (!availableRows || availableRows.length === 0) {
			return;
		}

		const mouseX = event.clientX;
		const mouseY = event.clientY;
		
		// Find the nearest column to the mouse
		let nearestRow = null;
		let nearestDistance = Infinity;
		
		availableRows.forEach((row) => {
			const rect = row.getBoundingClientRect();
			const centerX = rect.left + rect.width / 2;
			const centerY = rect.top + rect.height / 2;
			const distance = Math.sqrt(Math.pow(mouseX - centerX, 2) + Math.pow(mouseY - centerY, 2));
			
			if (distance < nearestDistance) {
				nearestDistance = distance;
				nearestRow = row;
			}
		});
		
		if (nearestRow) {
			if (!draggingComponentName || this.isDropAllowed(nearestRow, 'inside', draggingComponentName)) {
				const dropZone = new DropZone(nearestRow, 'inside');
				dropZone.isColumnDropZone = true;
				this.dropZones.push(dropZone);
			}
		}
	}

	/**
	 * Create a fallback drop zone for empty pages
	 */
	createFallbackDropZone() {
		// Find the main content area or body to create a drop zone
		const contentArea = document.querySelector('#sdc-inline-editor-content');
		const draggingComponentName = this.getDraggingComponentName();

		if (contentArea) {
			if (draggingComponentName && !this.isComponentAllowedInContainer(contentArea, draggingComponentName)) {
				return;
			}

			// Create a special drop zone that represents the entire content area
			const dropZone = new DropZone(contentArea, 'inside');
			// Mark this as a fallback drop zone
			dropZone.isFallback = true;
			this.dropZones.push(dropZone);
		}
	}

	/**
	 * Handle the actual drop operation
	 */
	handleDrop(event) {
		// Find the nearest drop zone to the mouse position
		const dropZone = this.findNearestDropZone(event.clientX, event.clientY);

		if (dropZone && this.draggingElement) {
			const draggedElement = this.draggingElement.element;
			const elementDropType = this.draggingElement.dropType || DropType.SDC;
			const dropInfo = dropZone.getDropInfo();
			const { targetWrapper, position } = dropInfo;

			// Handle different drop types
			if (elementDropType === DropType.SDC) {
				const componentName = this.getDraggingComponentName();
				if (!this.isDropAllowed(targetWrapper, position, componentName)) {
					this.showError('This component cannot be placed here.');
					return;
				}

				// Standard SDC block reordering
				// Remove the dragged element from its current position
				draggedElement.remove();

				// Insert the dragged element at the new position
				if (position === 'above') {
					targetWrapper.parentNode.insertBefore(draggedElement, targetWrapper);
				} else if (position === 'below') {
					targetWrapper.parentNode.insertBefore(draggedElement, targetWrapper.nextSibling);
				} else if (position === 'inside') {
					// For inside position - either column drop zone or fallback
					if (dropZone.isColumnDropZone || dropZone.isFallback) {
						targetWrapper.appendChild(draggedElement);
					}
				}

				// Mark as changed after reordering
				if (this.editor) {
					this.editor.markAsChanged();
				}
			} else if (elementDropType === DropType.TOOLBAR) {
				// Check if dropping into a table cell
				const componentName = this.getDraggingComponentName();
				if (position === 'inside' && this.isTableCell(targetWrapper) && this.isImageComponent(componentName)) {
					// Special handling for table cell image replacement
					this.handleTableCellImageDrop(draggedElement, targetWrapper);
				} else if (position === 'inside' && (dropZone.isColumnDropZone || dropZone.isFallback)) {
					// Drop into column or fallback area
					this.handleToolbarDropToContainer(draggedElement, targetWrapper);
				} else {
					this.handleToolbarDrop(draggedElement, targetWrapper, position);
				}
			}
		}
	}

	/**
	 * Handle toolbar item drop - create new component block
	 */
	async handleToolbarDrop(toolbarElement, targetWrapper, position) {
		try {
			// Get the component name from the toolbar element
			const componentName = toolbarElement.getAttribute('data-sdc-name');

			if (!componentName) {
				console.error('No component name found on toolbar element');
				return;
			}

			const container = position === 'above' || position === 'below' ? targetWrapper.parentNode : targetWrapper;

			if (!this.isComponentAllowedInContainer(container, componentName)) {
				this.showError('This component cannot be placed here.');
				return;
			}

			console.log('Creating new component:', componentName);

			// Show loading state
			this.showLoadingState(targetWrapper, position);

			// Fetch the component HTML from the SDC service
			const componentData = await this.sdcService.getComponentHtml(componentName);
			const template = document.createElement('template');
			template.innerHTML = componentData.html.trim();
			const componentElement = template.content.firstChild;

			if (!componentData.success || !componentData.html) {
				throw new Error(componentData.error || 'Failed to get component HTML');
			}

			// Insert the new block at the target position
			if (position === 'above') {
				targetWrapper.parentNode.insertBefore(componentElement, targetWrapper);
			} else {
				targetWrapper.parentNode.insertBefore(componentElement, targetWrapper.nextSibling);
			}

			// Remove loading state
			this.hideLoadingState();

			// Trigger a custom event to notify that a new component was added
			const event = new CustomEvent('componentAdded', {
				detail: {
					component: componentName,
					componentElement: componentElement,
					metadata: componentData.metadata,
				},
			});
			document.dispatchEvent(event);

			console.log('Successfully created component:', componentName);
		} catch (error) {
			console.error('Error creating component from toolbar:', error);
			this.hideLoadingState();
			this.showError('Failed to create component: ' + error.message);
		}
	}

	/**
	 * Handle toolbar item drop to empty page - create new component block
	 */
	async handleToolbarDropToEmptyPage(toolbarElement, contentArea) {
		try {
			// Get the component name from the toolbar element
			const componentName = toolbarElement.getAttribute('data-sdc-name');

			if (!componentName) {
				console.error('No component name found on toolbar element');
				return;
			}

			if (!this.isComponentAllowedInContainer(contentArea, componentName)) {
				this.showError('This component cannot be placed here.');
				return;
			}

			console.log('Creating new component on empty page:', componentName);

			// Show loading state
			this.showLoadingStateForEmptyPage(contentArea);

			// Fetch the component HTML from the SDC service
			const componentData = await this.sdcService.getComponentHtml(componentName);
			const template = document.createElement('template');
			template.innerHTML = componentData.html.trim();
			const componentElement = template.content.firstChild;

			if (!componentData.success || !componentData.html) {
				throw new Error(componentData.error || 'Failed to get component HTML');
			}

			// Append the new block to the content area
			contentArea.appendChild(componentElement);

			// Remove loading state
			this.hideLoadingState();

			// Trigger a custom event to notify that a new component was added
			const event = new CustomEvent('componentAdded', {
				detail: {
					component: componentName,
					componentElement: componentElement,
					metadata: componentData.metadata,
				},
			});
			document.dispatchEvent(event);

			console.log('Successfully created component on empty page:', componentName);
		} catch (error) {
			console.error('Error creating component on empty page:', error);
			this.hideLoadingState();
			this.showError('Failed to create component: ' + error.message);
		}
	}

	/**
	 * Handle dropping an image into a table cell (replaces existing content)
	 */
	async handleTableCellImageDrop(toolbarElement, tableCell) {
		try {
			// Get the component name from the toolbar element
			const componentName = toolbarElement.getAttribute('data-sdc-name');

			if (!componentName) {
				console.error('No component name found on toolbar element');
				return;
			}

			// Show loading state
			this.showLoadingStateForEmptyPage(tableCell);

			// Fetch the component HTML from the SDC service
			const componentData = await this.sdcService.getComponentHtml(componentName);
			const template = document.createElement('template');
			template.innerHTML = componentData.html.trim();
			const componentElement = template.content.firstChild;

			if (!componentData.success || !componentData.html) {
				throw new Error(componentData.error || 'Failed to get component HTML');
			}

			// Remove all existing content from the table cell
			while (tableCell.firstChild) {
				tableCell.removeChild(tableCell.firstChild);
			}

			// Add the new image component
			tableCell.appendChild(componentElement);

			// Remove loading state
			this.hideLoadingState();

			// Trigger a custom event to notify that a new component was added
			const event = new CustomEvent('componentAdded', {
				detail: {
					component: componentName,
					componentElement: componentElement,
					metadata: componentData.metadata,
				},
			});
			document.dispatchEvent(event);

		} catch (error) {
			console.error('Error replacing table cell content:', error);
			this.hideLoadingState();
			this.showError('Failed to add image: ' + error.message);
		}
	}

	/**
	 * Handle toolbar item drop to container (column or other container)
	 */
	async handleToolbarDropToContainer(toolbarElement, container) {
		try {
			// Get the component name from the toolbar element
			const componentName = toolbarElement.getAttribute('data-sdc-name');

			if (!componentName) {
				console.error('No component name found on toolbar element');
				return;
			}

			if (!this.isComponentAllowedInContainer(container, componentName)) {
				this.showError('This component cannot be placed here.');
				return;
			}

			console.log('Creating new component in container:', componentName);

			// Show loading state
			this.showLoadingStateForEmptyPage(container);

			// Fetch the component HTML from the SDC service
			const componentData = await this.sdcService.getComponentHtml(componentName);
			const template = document.createElement('template');
			template.innerHTML = componentData.html.trim();
			const componentElement = template.content.firstChild;

			if (!componentData.success || !componentData.html) {
				throw new Error(componentData.error || 'Failed to get component HTML');
			}

			// Append the new block to the container
			container.appendChild(componentElement);

			// Remove loading state
			this.hideLoadingState();

			// Trigger a custom event to notify that a new component was added
			const event = new CustomEvent('componentAdded', {
				detail: {
					component: componentName,
					componentElement: componentElement,
					metadata: componentData.metadata,
				},
			});
			document.dispatchEvent(event);

			console.log('Successfully created component in container:', componentName);
		} catch (error) {
			console.error('Error creating component in container:', error);
			this.hideLoadingState();
			this.showError('Failed to create component: ' + error.message);
		}
	}

	/**
	 * Show loading state at drop position
	 */
	showLoadingState(targetWrapper, position) {
		const loadingElement = document.createElement('div');
		loadingElement.classList.add('sdc-inline-editor-loading-placeholder');
		loadingElement.innerHTML = '<div class="loading-spinner"></div><span>Creating component...</span>';

		if (position === 'above') {
			targetWrapper.parentNode.insertBefore(loadingElement, targetWrapper);
		} else {
			targetWrapper.parentNode.insertBefore(loadingElement, targetWrapper.nextSibling);
		}

		this.loadingElement = loadingElement;
	}

	/**
	 * Show loading state for empty page
	 */
	showLoadingStateForEmptyPage(contentArea) {
		const loadingElement = document.createElement('div');
		loadingElement.classList.add('sdc-inline-editor-loading-placeholder');
		loadingElement.innerHTML = '<div class="loading-spinner"></div><span>Creating component...</span>';

		// Append to the content area
		contentArea.appendChild(loadingElement);

		this.loadingElement = loadingElement;
	}

	/**
	 * Hide loading state
	 */
	hideLoadingState() {
		if (this.loadingElement) {
			this.loadingElement.remove();
			this.loadingElement = null;
		}
	}

	/**
	 * Show error message
	 */
	showError(message) {
		// Create a temporary error notification
		const errorElement = document.createElement('div');
		errorElement.classList.add('sdc-inline-editor-error-notification');
		errorElement.innerHTML = `<span>${message}</span>`;
		errorElement.style.cssText = `
			position: fixed;
			top: 20px;
			right: 20px;
			background: #ff4444;
			color: white;
			padding: 10px 15px;
			border-radius: 4px;
			z-index: 10001;
			box-shadow: 0 2px 10px rgba(0,0,0,0.2);
		`;

		document.body.appendChild(errorElement);

		// Remove after 5 seconds
		setTimeout(() => {
			if (errorElement.parentNode) {
				errorElement.remove();
			}
		}, 5000);
	}

	/**
	 * Find drop zone under mouse coordinates
	 */
	findDropZoneUnderMouse(x, y) {
		return this.dropZones.find((dropZone) => dropZone.isUnderMouse(x, y));
	}

	/**
	 * Find the nearest drop zone to the given coordinates
	 */
	findNearestDropZone(x, y) {
		if (this.dropZones.length === 0) return null;

		let nearestDropZone = null;
		let nearestDistance = Infinity;

		this.dropZones.forEach((dropZone) => {
			if (!dropZone.element) return;

			const rect = dropZone.element.getBoundingClientRect();
			const centerX = rect.left + rect.width / 2;
			const centerY = rect.top + rect.height / 2;

			// Calculate distance to drop zone center
			const distance = Math.sqrt(Math.pow(x - centerX, 2) + Math.pow(y - centerY, 2));

			if (distance < nearestDistance) {
				nearestDistance = distance;
				nearestDropZone = dropZone;
			}
		});

		return nearestDropZone;
	}

	/**
	 * Clean up all drop zones
	 */
	cleanupDropZones() {
		this.dropZones.forEach((dropZone) => {
			dropZone.remove();
		});
		this.dropZones = [];
	}
}

export default DragAndDrop;
