import DragAndDrop from './DragAndDrop';
import CKEditor5TextManager from '../ckeditor5/ckeditor5-text-manager';
import { Toolbar } from './Toolbar';
import { DataProcessor } from './DataProcessor';
import { SideBar } from './SideBar';
import EditorService from './api/EditorService';
import { SessionController } from './SessionController';

/**
 * @file
 * Editor class for SDC Inline Editor
 *
 * This class is responsible for encapsulating elements and adding a toolbar to the wrapper.
 * It also initializes the drag and drop manager and the ckeditor5 text manager.
 */
export class Editor {
	dragAndDropManager = null;
	ckEditor5TextManager = null;
	editingMode = false;
	toolbar = null;
	sidebar = null;
	components = [];
	dataProcessor = null;
	editorService = null;
	nodeId = null;
	hasUnsavedChanges = false;
	constructor(options = {}) {
		this.components = options.components || [];
		this.editingMode = options.editingMode || false;
		this.nodeId = options.nodeId || null;
		this.sessionController = new SessionController({
			window: window,
			storage: localStorage,
			sessionDuration: 60 * 60 * 1000, // 1 hour default
			storageKeyPrefix: 'sdc-inline-editor-session',
			pageIdentifier: `${this.nodeId}`,
		});
		// Store permission settings
		this.hasDeletePermission = options.hasDeletePermission || false;
		this.hasMovePermission = options.hasMovePermission || false;
		this.hasShowToolbarPermission = options.hasShowToolbarPermission || false;
		this.hasEditTextEditorFieldPermission = options.hasEditTextEditorFieldPermission || false;
		this.hasEditHtmlEditorFieldPermission = options.hasEditHtmlEditorFieldPermission || false;
		this.hasEditImageSelectorFieldPermission = options.hasEditImageSelectorFieldPermission || false;
		this.hasEditNumberInputFieldPermission = options.hasEditNumberInputFieldPermission || false;
		this.hasEditCheckboxInputFieldPermission = options.hasEditCheckboxInputFieldPermission || false;
		this.hasEditSelectDropdownFieldPermission = options.hasEditSelectDropdownFieldPermission || false;
		this.hasEditMultiImageSelectorFieldPermission = options.hasEditMultiImageSelectorFieldPermission || false;
		this.hasEditTextInputFieldPermission = options.hasEditTextInputFieldPermission || false;
		this.hasEditIconSelectorFieldPermission = options.hasEditIconSelectorFieldPermission || false;

		this.sessionController.init();
		// This handles whether or not the editor is enabled for the current page.
		if(this.sessionController.isEditingEnabled()) {
			this.dataProcessor = new DataProcessor();
			this.editorService = new EditorService();
			this.init();
		} else {
			this.initEditorButtons();
		}
	}

	/**
	 * Create a button in the button left to enable editing for the current page.
	 */
	initEditorButtons() {
		const editorButton = document.createElement('button');
		editorButton.classList.add('sdc-inline-editor-editor-button');
		editorButton.innerHTML = 'Enable Editing';
		editorButton.addEventListener('click', () => {
			this.sessionController.enableEditing();
		});
		document.body.appendChild(editorButton);
	}

	init() {
		if (this.editingMode) {
			const sdcInlineEditorContent = document.querySelector('#sdc-inline-editor-content');
			if (sdcInlineEditorContent && !sdcInlineEditorContent.classList.contains('editing-mode')) {
				sdcInlineEditorContent.classList.add('editing-mode');
			}
			// Encapsulate elements
			this.encapsulateElements(document.body);

            // Initalize toolbar if not already initialized
            if (!this.toolbar) {
				this.toolbar = new Toolbar({
					components: this.components,
					editor: this,
					permissions: {
						hasShowToolbarPermission: this.hasShowToolbarPermission,
					}
				});
            }

			// Initialize ckeditor5 text manager if not already initialized
			if (!this.ckEditor5TextManager) {
				this.ckEditor5TextManager = new CKEditor5TextManager({
					permissions: {
						hasEditTextEditorFieldPermission: this.hasEditTextEditorFieldPermission,
						hasEditHtmlEditorFieldPermission: this.hasEditHtmlEditorFieldPermission,
					}
				});
			}

			// Initialize drag and drop if not already initialized
			if (!this.dragAndDropManager) {
				this.dragAndDropManager = new DragAndDrop(this);
			} else {
				// Refresh drag and drop for new elements
				this.dragAndDropManager.refresh();
			}

			// Initialize sidebar if not already initialized
			if (!this.sidebar) {
				this.sidebar = new SideBar({
					editor: this,
					permissions: {
						hasEditTextEditorFieldPermission: this.hasEditTextEditorFieldPermission,
						hasEditHtmlEditorFieldPermission: this.hasEditHtmlEditorFieldPermission,
						hasEditImageSelectorFieldPermission: this.hasEditImageSelectorFieldPermission,
						hasEditNumberInputFieldPermission: this.hasEditNumberInputFieldPermission,
						hasEditCheckboxInputFieldPermission: this.hasEditCheckboxInputFieldPermission,
						hasEditSelectDropdownFieldPermission: this.hasEditSelectDropdownFieldPermission,
						hasEditMultiImageSelectorFieldPermission: this.hasEditMultiImageSelectorFieldPermission,
						hasEditTextInputFieldPermission: this.hasEditTextInputFieldPermission,
						hasEditIconSelectorFieldPermission: this.hasEditIconSelectorFieldPermission,
					}
				});
			}

			// Listen for new components added via drag and drop
			this.setupComponentAddedListener();
			
			// Set up beforeunload listener to warn about unsaved changes
			this.setupBeforeUnloadListener();
			
			// Set up CKEditor change listener
			this.setupCKEditorChangeListener();
		}
	}

	close() {

	}

	open() {

	}

	encapsulateElements(element) {
		// Find all elements with the data-sdc-component attribute
		const sdcComponents = element.querySelectorAll('[data-sdc-component]');

		// Prevents the first element from being hidden by the page title, or other similar elements.
		let firstComponent = true;
		sdcComponents.forEach((component) => {


		// Skip if already wrapped
		if (component.parentNode.classList.contains('block-wrapper')) {
			return;
		}

		// Skip if component is inside a locked container, EXCEPT for single-image in table cells
		const lockedContainer = component.closest('[data-locked="true"]');
		if (lockedContainer) {
			// Only allow encapsulation for single-image components in table cells
			const isInTableCell = lockedContainer.classList.contains('table-cell-content');
			const isSingleImage = component.getAttribute('data-sdc-component') === 'sdc_inline_editor:single-image-block';
			
			if (!isInTableCell || !isSingleImage) {
				return;
			}
		}

		// Skip if component has data-sdc-no-encapsulate attribute
		if (component.hasAttribute('data-sdc-no-encapsulate')) {
			return;
		}		// Wrap the element in a div with the class block-wrapper and also add a toolbar into the block-wrapper
			const wrapper = document.createElement('div');
			wrapper.classList.add('block-wrapper');
			// If the component has a data-block-wrapper-auto, change the styling of the block-wrapper to be width:auto
			if (component.hasAttribute('data-block-wrapper-auto')) {
				wrapper.style.width = 'auto';
			}
			// Add a toolbar to the wrapper, toolbar will have a trash icon, a move icon, and a duplicate icon.
			const toolbar = document.createElement('div');
			toolbar.classList.add('block-wrapper-toolbar');

			// Count number of data-nested-container inside the component
			const nestedContainers = component.querySelectorAll('[data-nested-container="true"]');
			let nestedContainersCount = nestedContainers.length;
			// Also count if the component itself has a data-nested-container="true" attribute
			if (component.hasAttribute('data-nested-container')) {
				nestedContainersCount++;
			}
			
			// If the nested containers count is greater than 0, translate the toolbar up and right a little to prevent overlapping
			// Of other toolbars
			if (nestedContainersCount > 0) {
				
				const up = firstComponent ? 0 : 32;
				const right = 0;
				toolbar.style.transform = `translate(${right}px, -${up}px)`;
			}
			
			// Add trash icon only if user has delete permission
			if (this.hasDeletePermission) {
				const trashIcon = document.createElement('div');
				trashIcon.classList.add('sdc-inline-editor-trash-icon');
				trashIcon.addEventListener('click', this.onDelete.bind(this));
				trashIcon.innerHTML =
					'<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 640"><path d="M232.7 69.9L224 96L128 96C110.3 96 96 110.3 96 128C96 145.7 110.3 160 128 160L512 160C529.7 160 544 145.7 544 128C544 110.3 529.7 96 512 96L416 96L407.3 69.9C402.9 56.8 390.7 48 376.9 48L263.1 48C249.3 48 237.1 56.8 232.7 69.9zM512 208L128 208L149.1 531.1C150.7 556.4 171.7 576 197 576L443 576C468.3 576 489.3 556.4 490.9 531.1L512 208z"/></svg>';
				toolbar.appendChild(trashIcon);
			}

			// Add move icon only if user has move permission
			if (this.hasMovePermission) {
				const moveIcon = document.createElement('div');
				moveIcon.classList.add('sdc-inline-editor-move-icon');
				moveIcon.setAttribute('draggable', 'true');
				moveIcon.innerHTML =
					'<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 640"><path d="M96 160C96 142.3 110.3 128 128 128L512 128C529.7 128 544 142.3 544 160C544 177.7 529.7 192 512 192L128 192C110.3 192 96 177.7 96 160zM96 320C96 302.3 110.3 288 128 288L512 288C529.7 288 544 302.3 544 320C544 337.7 529.7 352 512 352L128 352C110.3 352 96 337.7 96 320zM544 480C544 497.7 529.7 512 512 512L128 512C110.3 512 96 497.7 96 480C96 462.3 110.3 448 128 448L512 448C529.7 448 544 462.3 544 480z"/></svg>';
				toolbar.appendChild(moveIcon);
			}

			// Check if component has direct sidebar elements and add settings icon
			const directSidebarElements = this.getDirectSidebarElements(component);
			if (directSidebarElements.length > 0) {
				const settingsIcon = document.createElement('div');
				settingsIcon.classList.add('sdc-inline-editor-settings-icon');
				settingsIcon.innerHTML =
					'<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24"><path d="M12 15.5A3.5 3.5 0 0 1 8.5 12A3.5 3.5 0 0 1 12 8.5a3.5 3.5 0 0 1 3.5 3.5a3.5 3.5 0 0 1-3.5 3.5m7.43-2.53c.04-.32.07-.64.07-.97c0-.33-.03-.66-.07-1l2.11-1.63c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.31-.61-.22l-2.49 1c-.52-.39-1.06-.73-1.69-.98l-.37-2.65A.506.506 0 0 0 14 2h-4c-.25 0-.46.18-.5.42l-.37 2.65c-.63.25-1.17.59-1.69.98l-2.49-1c-.22-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64L4.57 11c-.04.34-.07.67-.07 1c0 .33.03.65.07.97l-2.11 1.66c-.19.15-.25.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1.01c.52.4 1.06.74 1.69.99l.37 2.65c.04.24.25.42.5.42h4c.25 0 .46-.18.5-.42l.37-2.65c.63-.26 1.17-.59 1.69-.99l2.49 1.01c.22.08.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.66Z"/></svg>';
				toolbar.appendChild(settingsIcon);
			}

			if(this.hasDeletePermission || this.hasMovePermission || directSidebarElements.length > 0) {
				wrapper.appendChild(toolbar);
			}

			// Add visual indicators
			component.classList.add('sdc-editable-container');

			// Insert the wrapper before the component
			component.parentNode.insertBefore(wrapper, component);

			// Move the component into the wrapper
			wrapper.appendChild(component);

			if (firstComponent) {
				firstComponent = false;
			}
		});
	}

	/**
	 * Get sidebar elements that belong directly to this component, not nested components
	 */
	getDirectSidebarElements(component) {
		const allSidebarElements = component.querySelectorAll('[data-sdc-sidebar]');
		const directSidebarElements = [];

		allSidebarElements.forEach(element => {
			// Check if this sidebar element is nested inside a child component
			let parent = element.parentElement;
			let isNested = false;

			while (parent && parent !== component) {
				// If we find a component that's not the current one, this element is nested
				if (parent.hasAttribute('data-sdc-component') && parent !== component) {
					isNested = true;
					break;
				}
				parent = parent.parentElement;
			}

			// Only include non-nested elements
			if (!isNested) {
				directSidebarElements.push(element);
			}
		});

		return directSidebarElements;
	}

	onDelete(event) {
		console.log('Delete', event);
		const blockWrapper = event.target.closest('.block-wrapper');
		
		// Check if this component is inside a table cell
		const tableCell = blockWrapper?.closest('.table-cell-content');
		if (tableCell && tableCell.hasAttribute('data-locked') && tableCell.getAttribute('data-locked') === 'true') {
			// This is inside a table cell - restore default text-block instead of just removing
			blockWrapper.remove();
			this.restoreTableCellText(tableCell);
		} else {
			// Normal deletion
			blockWrapper.remove();
		}
		
		this.markAsChanged();
	}

	/**
	 * Restore default text-block component to a table cell after deletion
	 */
	async restoreTableCellText(tableCell) {
		try {
			// Import and create SDC service
			const { default: SDCService } = await import('./api/SDCService');
			const sdcService = new SDCService();
			const componentData = await sdcService.getComponentHtml('sdc_inline_editor:text-block');
			
			if (componentData.success && componentData.html) {
				const template = document.createElement('template');
				template.innerHTML = componentData.html.trim();
				const componentElement = template.content.firstChild;
				
				// Clear the cell and add the text block
				while (tableCell.firstChild) {
					tableCell.removeChild(tableCell.firstChild);
				}
				tableCell.appendChild(componentElement);
				
				// Trigger componentAdded event
				const event = new CustomEvent('componentAdded', {
					detail: {
						component: 'sdc_inline_editor:text-block',
						componentElement: componentElement,
					},
				});
				document.dispatchEvent(event);
			} else {
				console.error('Failed to restore text-block:', componentData);
			}
		} catch (error) {
			console.error('Error restoring table cell text:', error);
		}
	}

	/**
	 * Set up beforeunload listener to warn about unsaved changes
	 */
	setupBeforeUnloadListener() {
		this.boundBeforeUnloadHandler = (event) => {
			if (this.hasUnsavedChanges) {
				// Modern browsers require returnValue to be set
				event.preventDefault();
				event.returnValue = '';
				// Some browsers show a custom message, but most now show a standard message
				return '';
			}
		};
		window.addEventListener('beforeunload', this.boundBeforeUnloadHandler);
	}

	/**
	 * Set up CKEditor change listener to detect text edits
	 */
	setupCKEditorChangeListener() {
		// Listen for CKEditor content changes
		document.addEventListener('ckEditorContentChanged', () => {
			this.markAsChanged();
		});
	}

	/**
	 * Mark content as having unsaved changes
	 */
	markAsChanged() {
		if (!this.hasUnsavedChanges) {
			this.hasUnsavedChanges = true;
		}
	}

	setupComponentAddedListener() {
		// Remove existing listener if it exists to prevent duplicates
		if (this.boundComponentAddedHandler) {
			document.removeEventListener('componentAdded', this.boundComponentAddedHandler);
		}

		// Create bound handler that we can remove later
		this.boundComponentAddedHandler = (event) => {
			console.log('Component added event received:', event.detail);
			const { componentElement } = event.detail;
			
			// Encapsulate the new component element
			if (componentElement) {
				this.encapsulateElements(document.body);
				
				// Refresh drag and drop for new elements
				if (this.dragAndDropManager) {
					this.dragAndDropManager.refresh();
				}

				// Initialize inline editors for new elements
				if (this.ckEditor5TextManager) {
					this.ckEditor5TextManager.initializeEditors();
				}

				// Re-attach Drupal behaviors to new components (for row-block resize handles, etc.)
				if (typeof Drupal !== 'undefined' && Drupal.attachBehaviors) {
					console.log('Re-attaching Drupal behaviors to new component');
					Drupal.attachBehaviors(componentElement);
				}

				// Mark as changed when component is added
				this.markAsChanged();
			}
		};

		document.addEventListener('componentAdded', this.boundComponentAddedHandler);
	}

	save() {
		console.log('Saving SDC Inline Editor content...');
		
		// Process all SDC components and extract their field data
        const sdcInlineEditorContent = document.querySelector('#sdc-inline-editor-content');
		const componentsData = this.dataProcessor.processComponents(sdcInlineEditorContent);

		console.log('Components data:', componentsData);
		
		// Create the ordered list of components with their specific fields
		const bodyContent = {
			components: componentsData,
			timestamp: new Date().toISOString()
		};
		
		// This will replace the body content with the processed SDC components
		this.sendToServer(bodyContent);
		
		return bodyContent;
	}

	/**
	 * Send the processed data to the server
	 * @param {Object} data - The processed component data
	 */
	async sendToServer(data) {
		try {
			console.log('Sending to server:', data);
			
			// Use the stored node ID
			if (!this.nodeId) {
				console.error('No node ID available. Cannot save content.');
				this.editorService.showNotification('No node ID available. Cannot save content.', 'error');
				return;
			}

			// Use the editor service to save components
			const result = await this.editorService.saveComponents(this.nodeId, data.components);

			if (result.success) {
				console.log('Save successful:', result);
				this.editorService.showNotification('Content saved successfully!', 'success');
				// Clear unsaved changes flag on successful save
				this.hasUnsavedChanges = false;
			} else {
				console.error('Save failed:', result);
				this.editorService.showNotification(`Save failed: ${result.message}`, 'error');
			}
		} catch (error) {
			console.error('Network error:', error);
			this.editorService.showNotification('Network error occurred while saving', 'error');
		}
	}

	/**
	 * Destroy the editor and clean up event listeners
	 */
	destroy() {
		// Remove componentAdded listener
		if (this.boundComponentAddedHandler) {
			document.removeEventListener('componentAdded', this.boundComponentAddedHandler);
			this.boundComponentAddedHandler = null;
		}

		// Remove beforeunload listener
		if (this.boundBeforeUnloadHandler) {
			window.removeEventListener('beforeunload', this.boundBeforeUnloadHandler);
			this.boundBeforeUnloadHandler = null;
		}

		// Destroy sidebar
		if (this.sidebar) {
			this.sidebar.destroy();
			this.sidebar = null;
		}

		// Destroy toolbar
		if (this.toolbar) {
			this.toolbar.destroy();
			this.toolbar = null;
		}

		// Clean up other components as needed
		this.dragAndDropManager = null;
		this.ckEditor5TextManager = null;
	}

}
