/**
 * @file
 * SessionController class for SDC Inline Editor
 *
 * This class is responsible for managing the editing sessions per page.
 * 
 * If the editor has not been enabled for a certain page it will not show up in the local Storage.
 * Editing sessions expire after an hour of being created. 
 * 
 * You will need to enable the editor for the specific page, this will handle all interaction between the local storage
 * and the editor.
 */
export class SessionController {
	constructor(options = {}) {
		this.window = options.window || (typeof window !== 'undefined' ? window : null);
		this.storage = options.storage || (this.window ? this.window.localStorage : null);
		this.sessionDuration =
			typeof options.sessionDuration === 'number'
				? options.sessionDuration
				: 60 * 60 * 1000; // 1 hour default
		this.storageKeyPrefix = options.storageKeyPrefix || 'sdc-inline-editor-session';
		this.pageIdentifier =
			options.pageIdentifier ||
			(this.window
				? `${this.window.location.pathname}${this.window.location.search}`
				: 'unknown-page');

		this.init();
	}

	init() {
		this.cleanupExpiredSession();
	}

	get storageKey() {
		return `${this.storageKeyPrefix}:${this.pageIdentifier}`;
	}

	enableEditing() {
		if (!this.storage) {
			console.warn('SessionController: Unable to access storage to enable editing.');
			return;
		}

		const sessionData = {
			enabled: true,
			timestamp: Date.now(),
		};

		try {
			this.storage.setItem(this.storageKey, JSON.stringify(sessionData));
		} catch (error) {
			console.error('SessionController: Failed to enable editing for page.', error);
			return;
		}

		this.showLoadingOverlay();

		if (this.window && this.window.location && typeof this.window.location.reload === 'function') {
			this.window.location.reload();
		}
	}

	showLoadingOverlay() {
		if (!this.window || !this.window.document) {
			return;
		}

		const doc = this.window.document;
		const styleId = 'sdc-inline-editor-loading-overlay-style';
		const overlayId = 'sdc-inline-editor-loading-overlay';

		if (!doc.getElementById(styleId)) {
			const style = doc.createElement('style');
			style.id = styleId;
			style.textContent = `
				.sdc-inline-editor-loading-overlay {
					position: fixed;
					inset: 0;
					display: flex;
					align-items: center;
					justify-content: center;
					background: rgba(17, 17, 17, 0.85);
					color: #ffffff;
					font-family: "Helvetica Neue", Arial, sans-serif;
					font-size: 1.25rem;
					z-index: 999999;
					opacity: 0;
					pointer-events: none;
					transition: opacity 150ms ease-in-out;
				}

				.sdc-inline-editor-loading-overlay.is-visible {
					opacity: 1;
					pointer-events: auto;
				}

				.sdc-inline-editor-loading-overlay__content {
					display: flex;
					align-items: center;
					gap: 12px;
				}

				.sdc-inline-editor-loading-overlay__spinner {
					width: 32px;
					height: 32px;
					border-radius: 50%;
					border: 4px solid rgba(255, 255, 255, 0.25);
					border-top-color: #ffffff;
					animation: sdc-inline-editor-loading-spin 1s linear infinite;
				}

				@keyframes sdc-inline-editor-loading-spin {
					to {
						transform: rotate(360deg);
					}
				}
			`;
			(doc.head || doc.documentElement || doc.body).appendChild(style);
		}

		let overlay = doc.getElementById(overlayId);

		if (!overlay) {
		overlay = doc.createElement('div');
		overlay.id = overlayId;
		overlay.className = 'sdc-inline-editor-loading-overlay';

		const content = doc.createElement('div');
		content.className = 'sdc-inline-editor-loading-overlay__content';

		const spinner = doc.createElement('span');
		spinner.className = 'sdc-inline-editor-loading-overlay__spinner';
		spinner.setAttribute('aria-hidden', 'true');

		const label = doc.createElement('span');
		label.className = 'sdc-inline-editor-loading-overlay__label';
			label.textContent = 'Loading editor...';

			content.append(spinner, label);

			overlay.appendChild(content);

			(doc.body || doc.documentElement).appendChild(overlay);
		}

		// Ensure overlay becomes fully visible on the next frame.
		const schedule = this.window.requestAnimationFrame
			? (callback) => this.window.requestAnimationFrame(callback)
			: (callback) => this.window.setTimeout(callback, 0);

		schedule(() => {
			overlay.classList.add('is-visible');
		});
	}

	disableEditing() {
		if (!this.storage) {
			console.warn('SessionController: Unable to access storage to disable editing.');
			return;
		}

		try {
			this.storage.removeItem(this.storageKey);
            if (this.window && this.window.location && typeof this.window.location.reload === 'function') {
                this.window.location.reload();
            }
		} catch (error) {
			console.error('SessionController: Failed to disable editing for page.', error);
		}
	}

	isEditingEnabled() {
		if (!this.storage) {
			return false;
		}

		const sessionData = this.getSession();
		return Boolean(sessionData && sessionData.enabled);
	}

	getSession() {
		if (!this.storage) {
			return null;
		}

		const rawValue = this.storage.getItem(this.storageKey);

		if (!rawValue) {
			return null;
		}

		try {
			const parsed = JSON.parse(rawValue);

			if (!parsed.timestamp) {
				return parsed;
			}

			if (Date.now() - parsed.timestamp > this.sessionDuration) {
				this.storage.removeItem(this.storageKey);
				return null;
			}

			return parsed;
		} catch (error) {
			console.error('SessionController: Failed to parse session data.', error);
			this.storage.removeItem(this.storageKey);
			return null;
		}
	}

	cleanupExpiredSession() {
		this.getSession();
	}
}