/**
 * @file
 * SideBar class for SDC Inline Editor
 *
 * This class handles the sidebar interface that appears when users click on block-wrappers
 * containing elements with data-sdc-sidebar attributes.
 */

import SDCService from './api/SDCService.js';
import { DataProcessor } from './DataProcessor.js';
import { sidebarFieldHandlers } from './fieldHandlers/sidebarFieldHandlers.js';

export class SideBar {
	constructor(options = {}) {
		this.editor = options.editor || null;
		this.isOpen = false;
		this.currentComponent = null;
		this.sidebarElement = null;
		this.editorInstances = new Map();
		this.sdcService = new SDCService();
		this.dataProcessor = new DataProcessor();
		
		// Store permission settings
		this.permissions = options.permissions || {
			hasEditTextEditorFieldPermission: false,
			hasEditHtmlEditorFieldPermission: false,
			hasEditImageSelectorFieldPermission: false,
			hasEditNumberInputFieldPermission: false,
			hasEditCheckboxInputFieldPermission: false,
			hasEditSelectDropdownFieldPermission: false,
			hasEditMultiImageSelectorFieldPermission: false,
			hasEditTextInputFieldPermission: false,
			hasEditIconSelectorFieldPermission: false,
		};

		this.init();
	}

	init() {
		this.createSidebarElement();
		this.setupClickListeners();
	}

	/**
	 * Create the sidebar DOM element
	 */
	createSidebarElement() {
		this.sidebarElement = document.createElement('div');
		this.sidebarElement.id = 'sdc-inline-editor-sidebar';
		this.sidebarElement.className = 'sdc-inline-editor-sidebar';
		this.sidebarElement.innerHTML = `
      <div class="sidebar-header">
        <h3>Component Settings</h3>
        <div class="sidebar-header-actions">
          <button class="sidebar-save" type="button">Save</button>
          <button class="sidebar-close" type="button">&times;</button>
        </div>
      </div>
      <div class="sidebar-content">
        <div class="sidebar-editors"></div>
      </div>
    `;

		// Add to body
		document.body.appendChild(this.sidebarElement);

		// Setup close button
		const closeBtn = this.sidebarElement.querySelector('.sidebar-close');
		closeBtn.addEventListener('click', () => this.close());

		// Setup save button
		const saveBtn = this.sidebarElement.querySelector('.sidebar-save');
		saveBtn.addEventListener('click', () => this.saveComponent());
	}

	/**
	 * Setup click listeners for settings icons using event delegation
	 */
	setupClickListeners() {
		// Remove existing listener if it exists to prevent duplicates
		if (this.boundClickHandler) {
			document.removeEventListener('click', this.boundClickHandler);
		}

		// Create bound handler that we can remove later
		this.boundClickHandler = (event) => {
			// Check if the clicked element is a settings icon or contains one
			const settingsIcon = event.target.closest('.sdc-inline-editor-settings-icon');

			if (settingsIcon) {
				event.preventDefault();
				event.stopPropagation();

				// Find the closest block wrapper
				const blockWrapper = settingsIcon.closest('.block-wrapper');
				if (blockWrapper) {
					this.open(blockWrapper);
				}
			}
		};

		document.addEventListener('click', this.boundClickHandler);
	}

	/**
	 * Check if sidebar should open for this block-wrapper
	 */
	shouldOpenSidebar(blockWrapper) {
		// Look for elements with data-sdc-sidebar attribute (excluding nested components)
		const sidebarElements = this.getDirectSidebarElements(blockWrapper);
		return sidebarElements.length > 0;
	}

	/**
	 * Open sidebar for a specific block-wrapper
	 */
	open(blockWrapper) {
		this.currentComponent = blockWrapper;
		this.isOpen = true;

		// Get sidebar elements from the component, excluding those in nested components
		const sidebarElements = this.getDirectSidebarElements(blockWrapper);

		// Clear previous editors
		this.clearEditors();

		// Create editors for each sidebar element
		this.createEditors(sidebarElements);

		// Show sidebar
		this.sidebarElement.classList.add('open');
		document.body.classList.add('sidebar-open');
	}

	/**
	 * Get sidebar elements that belong directly to this component, not nested components
	 */
	getDirectSidebarElements(blockWrapper) {
		const allSidebarElements = blockWrapper.querySelectorAll('[data-sdc-sidebar]');
		const directSidebarElements = [];

		// Get the primary component element for this wrapper
		const primaryComponent = blockWrapper.querySelector('[data-sdc-component]');

		allSidebarElements.forEach(element => {
			// Check if this sidebar element is nested inside a child component
			let parent = element.parentElement;
			let isNested = false;

			while (parent && parent !== blockWrapper) {
				// If we find a component that's not the primary one, this element is nested
				if (parent.hasAttribute('data-sdc-component') && parent !== primaryComponent) {
					isNested = true;
					break;
				}
				parent = parent.parentElement;
			}

			// Only include non-nested elements
			if (!isNested) {
				directSidebarElements.push(element);
			}
		});

		return directSidebarElements;
	}

	/**
	 * Close the sidebar
	 */
	close() {
		this.isOpen = false;
		this.currentComponent = null;

		// Hide sidebar
		this.sidebarElement.classList.remove('open');
		document.body.classList.remove('sidebar-open');

		// Clear editors
		this.clearEditors();
	}

	/**
	 * Clear all editor instances
	 */
	clearEditors() {
		const editorsContainer = this.sidebarElement.querySelector('.sidebar-editors');
		editorsContainer.innerHTML = '';

		// Destroy editor instances and call cleanup functions
		this.editorInstances.forEach((editor) => {
			if (editor.destroy) {
				editor.destroy();
			}
			// Call cleanup function if it exists (for custom field handlers like link-list-manager)
			if (editor.containerWithCleanup && editor.containerWithCleanup._cleanup) {
				editor.containerWithCleanup._cleanup();
			} else if (editor.element && editor.element._cleanup) {
				editor.element._cleanup();
			}
		});
		this.editorInstances.clear();
	}

	/**
	 * Create editors for sidebar elements
	 */
	createEditors(sidebarElements) {
		const editorsContainer = this.sidebarElement.querySelector('.sidebar-editors');

		sidebarElements.forEach((element) => {
			const fieldName = element.getAttribute('data-sdc-field');
			const fieldType = element.getAttribute('data-sdc-type');

			if (!fieldName || !fieldType) {
				console.warn('Sidebar element missing required attributes:', element);
				return;
			}

			// Check permissions for this field type
			if (!this.hasPermissionForFieldType(fieldType)) {
				return;
			}

			// Create editor based on type
			const editorElement = this.createEditorElement(fieldName, fieldType, element);
			if (editorElement) {
				editorsContainer.appendChild(editorElement);

				// Initialize the editor
				this.initializeEditor(fieldType, editorElement, element);
			}
		});
	}

	/**
	 * Create editor element based on type
	 */
	createEditorElement(fieldName, fieldType, originalElement) {
		const editorWrapper = document.createElement('div');
		editorWrapper.className = 'sidebar-editor-wrapper';
		editorWrapper.setAttribute('data-field-name', fieldName);
		editorWrapper.setAttribute('data-field-type', fieldType);

		const label = document.createElement('label');
		label.textContent = this.formatFieldLabel(fieldName);
		label.className = 'sidebar-editor-label';

		const editorContainer = document.createElement('div');
		editorContainer.className = 'sidebar-editor-container';

		// Create editor input based on type
		const editorInput = this.createEditorInput(fieldType, fieldName, originalElement);
		if (editorInput) {
			editorContainer.appendChild(editorInput);
		} else {
			console.warn(`Unknown sidebar editor type: ${fieldType}`);
			return null;
		}

		editorWrapper.appendChild(label);
		editorWrapper.appendChild(editorContainer);

		return editorWrapper;
	}

	/**
	 * Create editor input based on type
	 */
	createEditorInput(fieldType, fieldName, originalElement) {
		const handler = sidebarFieldHandlers[fieldType] || sidebarFieldHandlers.default;

		if (!handler) {
			console.warn(`Unknown sidebar editor type: ${fieldType}`);
			return null;
		}

		return handler(this, fieldName, originalElement);
	}

	/**
	 * Initialize editor based on type
	 */
	initializeEditor(fieldType, editorElement, originalElement) {
		// Store editor instance for cleanup
		// Find elements that might have cleanup functions (like link-list-manager container)
		const containerWithCleanup = editorElement.querySelector('.sidebar-link-list-manager');
		
		this.editorInstances.set(originalElement, {
			type: fieldType,
			element: editorElement,
			containerWithCleanup: containerWithCleanup, // Store for cleanup
		});
	}

	/**
	 * Check if user has permission for a specific field type
	 */
	hasPermissionForFieldType(fieldType) {
		switch (fieldType) {
			case 'text-input':
				return this.permissions.hasEditTextInputFieldPermission;
			case 'textarea':
				return this.permissions.hasEditTextInputFieldPermission; // Reuse text input permission
			case 'date-table':
				return this.permissions.hasEditTextInputFieldPermission; // Reuse text input permission
			case 'image-selector':
				return this.permissions.hasEditImageSelectorFieldPermission;
			case 'multi-image-selector':
				return this.permissions.hasEditMultiImageSelectorFieldPermission;
			case 'select-dropdown':
				return this.permissions.hasEditSelectDropdownFieldPermission;
			case 'number-input':
				return this.permissions.hasEditNumberInputFieldPermission;
			case 'checkbox-toggle':
				return this.permissions.hasEditCheckboxInputFieldPermission;
			case 'icon-selector':
				return this.permissions.hasEditIconSelectorFieldPermission;
			case 'link-list-manager':
				return this.permissions.hasEditTextInputFieldPermission; // Reuse text input permission
			default:
				// For unknown field types, allow by default (backward compatibility)
				return true;
		}
	}

	/**
	 * Get field label from field name
	 */
	getFieldLabel(fieldName) {
		return this.formatFieldLabel(fieldName
			.replace(/([A-Z])/g, ' $1')
			.replace(/^./, (str) => str.toUpperCase())
			.trim());
	}

	/**
	 * Get field value from original element
	 */
	getFieldValue(originalElement, fieldType) {
		switch (fieldType) {
			case 'text-input':
				return originalElement.getAttribute('data-sdc-prop-text-input') || '';

		case 'textarea':
			return originalElement.getAttribute('data-sdc-prop-textarea') || '';

		case 'text-editor':
			// Decode HTML entities from the data attribute (browsers automatically escape HTML in attributes)
			const encodedValue = originalElement.getAttribute('data-sdc-prop-text-editor') || '';
			if (encodedValue) {
				const textarea = document.createElement('textarea');
				textarea.innerHTML = encodedValue;
				return textarea.value;
			}
			return '';

		case 'text-editor-popup':
			// Read from data attribute (same as text-editor)
			// The CustomTextPopupEditor saves to innerHTML for display, but we need to read from the data attribute
			// which is updated when the sidebar field changes
			const popupEncodedValue = originalElement.getAttribute('data-sdc-prop-text-editor-popup') || '';
			if (popupEncodedValue) {
				// Decode HTML entities from the data attribute
				const textarea = document.createElement('textarea');
				textarea.innerHTML = popupEncodedValue;
				return textarea.value;
			}
			// Fallback to innerHTML if data attribute is not set (for backward compatibility)
			return originalElement.innerHTML || '';

		case 'date-table':
				return originalElement.getAttribute('data-sdc-prop-date-table') || '';

			case 'image-selector':
				return originalElement.getAttribute('data-sdc-prop-image-selector') || '';

		case 'image-crop':
			const cropData = originalElement.getAttribute('data-sdc-prop-image-crop') || '';
			if (cropData) {
				try {
					const parsed = JSON.parse(cropData);
					return parsed;
				} catch (e) {
					console.warn('Failed to parse image-crop data:', cropData);
					return {};
				}
			}
			return {};		case 'multi-image-selector':
			try {
				const value = JSON.parse(originalElement.getAttribute('data-sdc-prop-multi-image-selector') || '[]');
				return value;
			} catch (e) {
				console.warn('Failed to parse multi-image-selector value:', originalElement.getAttribute('data-sdc-prop-multi-image-selector'));
				return [];
			}

			case 'image-alignment':
				return originalElement.getAttribute('data-sdc-prop-image-alignment') || 'left';

			case 'color-picker':
				return originalElement.getAttribute('data-sdc-prop-color-picker') || '#000000';

			case 'select-dropdown':
				return originalElement.getAttribute('data-sdc-prop-select-dropdown') || '';

			case 'number-input':
				return originalElement.getAttribute('data-sdc-prop-number-input') || '0';

			case 'checkbox-toggle':
				const value = originalElement.getAttribute('data-sdc-prop-checkbox-toggle');
				// Handle string values properly - "false" should be false, not true
				if (value == 'true' || value == '1' || value == 'on') return true;
				if (value == 'false' || value == '0' || value == 'off' || value == '' || value == null) return false;
				return Boolean(value);

			case 'icon-selector':
				return originalElement.getAttribute('data-sdc-prop-icon-selector') || '';

			case 'link-list-manager':
				return originalElement.getAttribute('data-sdc-prop-link-list-manager') || '[]';

			default:
				return '';
		}
	}

	/**
	 * Update field value in original element
	 */
	updateFieldValue(originalElement, value, fieldType) {
		const propAttribute = `data-sdc-prop-${fieldType}`;
		originalElement.setAttribute(propAttribute, value);

		// For text-editor-popup, also update innerHTML so the display matches
		if (fieldType === 'text-editor-popup') {
			originalElement.innerHTML = value;
		}

		if (fieldType === 'icon-selector') {
			this.updateComponentIconDisplay(originalElement, value);
		}

		if (fieldType === 'image-crop') {
			this.updateImageCrop(originalElement, value);
		}

		// Trigger change event for data processing
		const event = new CustomEvent('sidebar-field-changed', {
			detail: {
				element: originalElement,
				fieldType: fieldType,
				value: value,
			},
		});
		document.dispatchEvent(event);
	}

	/**
	 * Update the visible icon within the component when using the icon selector
	 */
	updateComponentIconDisplay(originalElement, iconClassName) {
		if (!originalElement) {
			return;
		}

		const component = originalElement.closest('[data-sdc-component]');
		if (!component) {
			return;
		}

		const applyIconClass = (target) => {
			if (!target) {
				return;
			}

			const baseClasses = target.getAttribute('data-sdc-icon-preview-base');
			const classes = baseClasses
				? `${baseClasses} ${iconClassName}`.trim()
				: iconClassName;

			target.setAttribute('class', classes);
			target.setAttribute('aria-hidden', 'true');
		};

		const explicitTargets = component.querySelectorAll('[data-sdc-icon-preview]');
		if (explicitTargets.length) {
			explicitTargets.forEach(applyIconClass);
			return;
		}

		// Fallback: update the first icon element found within the component
		const fallbackIcon = component.querySelector('i');
		if (fallbackIcon) {
			applyIconClass(fallbackIcon);
		}
	}

	/**
	 * Update image alignment in the component
	 */
	updateImageAlignment(component, alignment) {
		const images = component.querySelectorAll('img');
		images.forEach((img) => {
			img.style.float = alignment;
			img.style.display = alignment === 'center' ? 'block' : 'inline';
			img.style.margin = alignment === 'center' ? '0 auto' : '';
		});
	}

	/**
	 * Update image crop positioning in the component
	 */
	updateImageCrop(originalElement, cropDataString) {
		if (!originalElement) {
			return;
		}

		const component = originalElement.closest('[data-sdc-component]');
		if (!component) {
			return;
		}

		// Parse crop data
		let cropData;
		try {
			cropData = typeof cropDataString === 'string' ? JSON.parse(cropDataString) : cropDataString;
		} catch (e) {
			console.warn('Failed to parse crop data:', e);
			return;
		}

		if (!cropData) {
			return;
		}

		// Find images in the component that correspond to this field
		const fieldName = originalElement.getAttribute('data-sdc-field');
		const images = component.querySelectorAll('img');
		
		images.forEach((img) => {
			if (cropData.x !== undefined && cropData.y !== undefined) {
				img.style.objectFit = 'cover';
				img.style.objectPosition = `${cropData.x}% ${cropData.y}%`;
			}
		});
	}

	/**
	 * Update color property in the component
	 */
	updateColorProperty(component, fieldName, color) {
		// This would need to be customized based on what the color affects
		// For now, we'll update a CSS custom property
		component.style.setProperty(`--${fieldName}-color`, color);
	}

	/**
	 * Get dropdown options from element attributes
	 */
	getDropdownOptions(element) {
		const optionsAttr = element.getAttribute('data-sdc-options');
		if (optionsAttr) {
			try {
				return JSON.parse(optionsAttr);
			} catch (e) {
				console.warn('Invalid options JSON:', optionsAttr);
			}
		}
		return null;
	}

	/**
	 * Load image preview from UUID
	 */
	loadImagePreview(uuid, previewElement) {
		// For now, we'll show a placeholder since we don't have a direct API
		// to get the image URL from UUID in the frontend
		previewElement.innerHTML = `<div class="no-image"></div>`;

		// TODO: Implement API call to get image URL from UUID
		// This would require a custom endpoint or service
	}

	/**
	 * Open media library using Drupal's media library
	 * @param {Function} callback - Callback function to handle selected media
	 * @param {Boolean} allowMultiple - Whether to allow multiple image selection
	 * @param {Number} maxImages - Maximum number of images to allow (default 10)
	 */
	openMediaLibrary(callback, allowMultiple = false, maxImages = 10) {
		// Get the media library URL from drupalSettings
		let mediaLibraryUrl = drupalSettings.sdcInlineEditor?.mediaLibraryUrl;

		if (mediaLibraryUrl) {
			// Add multiple selection parameter to URL if needed
			if (allowMultiple) {
				const separator = mediaLibraryUrl.includes('?') ? '&' : '?';
				// Use 'remaining_slots' to match what the PHP controller expects
				mediaLibraryUrl += `${separator}remaining_slots=${maxImages}`;
			}

			// Open the media library in a modal/iframe
			const modal = document.createElement('div');
			modal.className = 'sdc-inline-editor-media-modal';
			modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        z-index: 10000;
        display: flex;
        align-items: center;
        justify-content: center;
      `;

			const iframe = document.createElement('iframe');
			iframe.src = mediaLibraryUrl;
			iframe.style.cssText = `
        width: 90%;
        height: 90%;
        border: none;
        background: white;
        border-radius: 8px;
      `;

			const closeButton = document.createElement('button');
			closeButton.innerHTML = '×';
			closeButton.style.cssText = `
        position: absolute;
        top: 20px;
        right: 20px;
        background: #333;
        color: white;
        border: none;
        width: 40px;
        height: 40px;
        border-radius: 50%;
        font-size: 24px;
        cursor: pointer;
        z-index: 10001;
      `;

			closeButton.addEventListener('click', () => {
				document.body.removeChild(modal);
			});

			modal.appendChild(iframe);
			modal.appendChild(closeButton);
			document.body.appendChild(modal);

			// Listen for the custom event from the media library opener
			const handleMediaSelection = (event) => {
				if (event.detail && event.detail.entities && callback) {
					// Convert the entities to a format the callback expects
					const mediaData = event.detail.entities.map((entity) => ({
						id: entity[0],
						uuid: entity[1],
						type: entity[2],
						thumbnail: entity[3],
					}));
					callback(mediaData);
					// Close modal after selection
					document.body.removeChild(modal);
				}
				document.removeEventListener('sdcInlineEditorMediaSelected', handleMediaSelection);
			};

			document.addEventListener('sdcInlineEditorMediaSelected', handleMediaSelection);
		}
	}

	/**
	 * Save component changes and update HTML
	 */
	async saveComponent() {
		if (!this.currentComponent) {
			console.warn('No component selected for saving');
			return;
		}

		const saveBtn = this.sidebarElement.querySelector('.sidebar-save');
		const originalText = saveBtn.textContent;

		try {
			// Show loading state
			saveBtn.textContent = 'Saving...';
			saveBtn.disabled = true;

			// Get component type
			const componentElement = this.currentComponent.querySelector('[data-sdc-component]');
			const componentType = componentElement?.getAttribute('data-sdc-component');
			if (!componentType) {
				throw new Error('Component type not found');
			}

		// Collect all field data from the sidebar (which has the current values)
		const fieldData = this.collectFieldData();
		
		// Parse JSON string values to proper types before sending to backend
		const parsedFieldData = this.parseFieldDataForBackend(fieldData);
		
		// Get updated HTML from SDCService
		const result = await this.sdcService.getComponentHtml(componentType, parsedFieldData);
		if (result && result.html) {
			// Find the actual component element (not the wrapper)
			const componentElement = this.currentComponent.querySelector('[data-sdc-component]');
			if (componentElement) {
				// Create a temporary container to parse the new HTML
				const tempContainer = document.createElement('div');
				tempContainer.innerHTML = result.html;

				// Get the new component element from the parsed HTML
				const newComponentElement = tempContainer.querySelector('[data-sdc-component]');

				if (newComponentElement) {
					// Get the old wrapper (if it exists) to preserve its position
					const oldWrapper = componentElement.closest('.block-wrapper');
					const parentNode = oldWrapper ? oldWrapper.parentNode : componentElement.parentNode;
					
					// Remove the old wrapper completely (or just the component if no wrapper)
					if (oldWrapper) {
						// Insert the new component where the old wrapper was
						parentNode.insertBefore(newComponentElement, oldWrapper);
						// Remove the old wrapper
						oldWrapper.remove();
					} else {
						// No wrapper, just replace the component
						componentElement.parentNode.replaceChild(newComponentElement, componentElement);
					}

					// Re-attach Drupal behaviors to the updated component
					if (typeof Drupal !== 'undefined' && Drupal.attachBehaviors) {
						Drupal.attachBehaviors(newComponentElement);
					}

					// Re-encapsulate the new component to add the wrapper and toolbar
					// We need to encapsulate the entire document to ensure proper wrapping
					if (this.editor && this.editor.encapsulateElements) {
						// Encapsulate the entire document - this will properly wrap the new component
						this.editor.encapsulateElements(document.body);
					}

					// Update currentComponent reference to point to the new wrapper
					const newWrapper = newComponentElement.closest('.block-wrapper');
					if (newWrapper) {
						this.currentComponent = newWrapper;
					} else {
						// If no wrapper was created, something went wrong - use the component itself
						this.currentComponent = newComponentElement;
					}

					// Recreate the editors to reference the new elements
					const finalComponent = this.currentComponent.querySelector('[data-sdc-component]') || this.currentComponent;
					const sidebarElements = this.getDirectSidebarElements(finalComponent);
					this.clearEditors();

					// Mark as changed when component settings are saved
					if (this.editor) {
						this.editor.markAsChanged();
					}

					// Show success feedback
					this.showSaveFeedback('Component saved successfully!', 'success');

					// Close the sidebar after a successful save to return focus to the page
					// After 400ms
					setTimeout(() => {
						this.close();
					}, 400);
				} else {
					throw new Error('New component element not found in returned HTML');
				}
			} else {
				throw new Error('Component element not found');
			}
		} else {
			throw new Error('No HTML returned from service');
		}
		} catch (error) {
			console.error('Error saving component:', error);
			this.showSaveFeedback(`Save failed: ${error.message}`, 'error');
		} finally {
			// Reset button state
			saveBtn.textContent = originalText;
			saveBtn.disabled = false;
		}
	}

	/**
	 * Collect field data from the current component
	 */
	collectFieldData() {
		const fieldData = {};

		if (!this.currentComponent) {
			return fieldData;
		}

		// Get all field elements from the component (both sidebar and inline), excluding nested ones
		const allFieldElements = this.currentComponent.querySelectorAll('[data-sdc-field]');
		const primaryComponent = this.currentComponent.querySelector('[data-sdc-component]');

		allFieldElements.forEach((element) => {
			const fieldName = element.getAttribute('data-sdc-field');
			const fieldType = element.getAttribute('data-sdc-type');

			if (fieldName && fieldType) {
				// Check if this field element is nested inside a child component
				let parent = element.parentElement;
				let isNested = false;

				while (parent && parent !== this.currentComponent) {
					// If we find a component that's not the primary one, this element is nested
					if (parent.hasAttribute('data-sdc-component') && parent !== primaryComponent) {
						isNested = true;
						break;
					}
					parent = parent.parentElement;
				}

				// Only include non-nested fields
				if (!isNested) {
					let value = this.getFieldValue(element, fieldType);
					
					// Special handling for link-list-manager: get current table data before collecting
					if (fieldType === 'link-list-manager') {
						// Get the current table data directly (this preserves user edits in the table)
						const managerContainer = this.sidebarElement.querySelector('.sidebar-link-list-manager');
						if (managerContainer && managerContainer._getTableData) {
							const tableData = managerContainer._getTableData();
							if (tableData && tableData !== '[]' && tableData !== '') {
								value = tableData;
								// Also update the field value so it's stored
								this.updateFieldValue(element, tableData, fieldType);
							}
						}
						
						// If value is still empty or '[]', try to read from existing DOM children
						if (!value || value === '[]' || value === '') {
							const component = element.closest('[data-sdc-component]');
							const dropContainer = component?.querySelector('[data-drop-container="true"]');
							if (dropContainer) {
								const existingListItems = dropContainer.querySelectorAll('[data-sdc-component="mse_sdc_components:link-list-item"]');
								if (existingListItems.length > 0) {
									// Import the extract function logic (inline for now)
									const items = [];
									existingListItems.forEach((item) => {
										const linkElement = item.querySelector('a');
										const text = linkElement?.textContent?.trim() || '';
										const url = linkElement?.getAttribute('href') || '#';
										const target = linkElement?.getAttribute('target') || '_self';
										
										// Get popover data from hidden fields
										const getFieldValue = (fieldName, fieldType) => {
											const fieldEl = item.querySelector(`[data-sdc-field="${fieldName}"]`);
											if (!fieldEl) return null;
											const propAttr = `data-sdc-prop-${fieldType}`;
											return fieldEl.getAttribute(propAttr);
										};
										
										const hoverText = getFieldValue('hover_text', 'text-input') || item.getAttribute('data-bs-content') || '';
										const popoverPlacement = getFieldValue('popover_placement', 'select-dropdown') || item.getAttribute('data-bs-placement') || 'left';
										const popoverTrigger = getFieldValue('popover_trigger', 'select-dropdown') || item.getAttribute('data-bs-trigger') || 'hover focus';
										const enablePopoverValue = getFieldValue('enable_popover', 'checkbox-toggle');
										const enablePopover = enablePopoverValue === 'true' || enablePopoverValue === '1' || enablePopoverValue === true;
										
										items.push({
											text: text,
											url: url,
											target: target,
											hover_text: hoverText,
											popover_placement: popoverPlacement,
											popover_trigger: popoverTrigger,
											enable_popover: enablePopover
										});
									});
									value = JSON.stringify(items);
								}
							}
						}
					}
					
					fieldData[fieldName] = value;
				}
			}
		});

		// Special handling for nested components (like "children")
		// If the component has data-nested-container, we need to preserve children
		if (primaryComponent && primaryComponent.hasAttribute('data-nested-container')) {
			// Use DataProcessor to collect children data
			const childrenData = this.dataProcessor.processRowChildren(primaryComponent);
			if (childrenData && childrenData.length > 0) {
				fieldData.children = childrenData;
			}
		}

		return fieldData;
	}

	/**
	 * Parse field data for backend - convert JSON strings to proper types
	 * @param {Object} fieldData - The raw field data
	 * @returns {Object} Parsed field data ready for backend
	 */
	parseFieldDataForBackend(fieldData) {
		const parsed = {};
		
		for (const [fieldName, value] of Object.entries(fieldData)) {
			// For link_list_items, parse JSON string to array
			if (fieldName === 'link_list_items') {
				if (typeof value === 'string') {
					try {
						const parsedValue = JSON.parse(value);
						parsed[fieldName] = Array.isArray(parsedValue) ? parsedValue : [];
					} catch (e) {
						console.warn('Failed to parse link_list_items:', e);
						parsed[fieldName] = [];
					}
				} else if (Array.isArray(value)) {
					parsed[fieldName] = value;
				} else {
					parsed[fieldName] = [];
				}
			} else if (fieldName === 'children') {
				// Children data is already in the correct format from processRowChildren
				// Just pass it through as-is
				parsed[fieldName] = value;
			} else {
				// For other fields, keep as-is or parse if it's a JSON string
				if (typeof value === 'string' && (value.startsWith('[') || value.startsWith('{'))) {
					try {
						parsed[fieldName] = JSON.parse(value);
					} catch (e) {
						// Not JSON, keep as string
						parsed[fieldName] = value;
					}
				} else {
					parsed[fieldName] = value;
				}
			}
		}
		
		return parsed;
	}

	/**
	 * Show save feedback to user
	 */
	showSaveFeedback(message, type = 'info') {
		const content = this.sidebarElement.querySelector('.sidebar-content');

		let feedbackElement = this.sidebarElement.querySelector('.save-feedback');
		if (!feedbackElement) {
			feedbackElement = document.createElement('div');
			feedbackElement.className = 'save-feedback';
			feedbackElement.setAttribute('role', 'status');
		}

		feedbackElement.textContent = message;
		feedbackElement.className = `save-feedback ${type}`;
		if (content) {
			content.insertBefore(feedbackElement, content.firstChild);
		}

		if (this.saveFeedbackTimeout) {
			clearTimeout(this.saveFeedbackTimeout);
		}

		this.saveFeedbackTimeout = setTimeout(() => {
			if (feedbackElement.parentNode) {
				feedbackElement.parentNode.removeChild(feedbackElement);
			}
			this.saveFeedbackTimeout = null;
		}, 3000);
	}

	/**
	 * Because of the way the dataProcessor works, we need to extract the values from the object
	 * and give it to the backend in a format that can be used to update the component.
	 * @param {Object} input - The input object
	 * @returns {Object} The extracted values
	 */
	extractValues(input) {
		// Base case: if input is not an object, return as-is
		if (typeof input !== 'object' || input === null) return input;
		
		// If the object has a "value" field, return that directly
		if ('value' in input && Object.keys(input).length === 2 && 'type' in input) {
			return input.value;
		}
		
		// Special handling for nested components (like "children")
		if (Array.isArray(input)) {
			return input.map(item => this.extractValues(item));
		}
		
		// Otherwise, recursively process each property
		const result = {};
		for (const [key, value] of Object.entries(input)) {
			result[key] = this.extractValues(value);
		}
		
		return result;
	}


	/**
	 * Destroy the sidebar
	 */
	destroy() {
		this.close();
		
		// Remove click listener
		if (this.boundClickHandler) {
			document.removeEventListener('click', this.boundClickHandler);
			this.boundClickHandler = null;
		}
		
		if (this.sidebarElement && this.sidebarElement.parentNode) {
			this.sidebarElement.parentNode.removeChild(this.sidebarElement);
		}
		this.editorInstances.clear();

		// Destroy save feedback timeout
		if (this.saveFeedbackTimeout) {
			clearTimeout(this.saveFeedbackTimeout);
			this.saveFeedbackTimeout = null;
		}
	}

	/**
	 * Format the field label to be more readable
	 * @param {String} fieldName - The field name
	 * @returns {String} The formatted field label
	 */
	formatFieldLabel(fieldName) {
		return fieldName.replace(/_/g, ' ').replace(/\b\w/g, char => char.toUpperCase());
	}

}
