

/**
 * @file
 * Toolbar class for SDC Inline Editor
 *
 * This class is responsible for adding the ability to add new SDCs to the page.
 */
export class Toolbar {

    components = [];
    editor = null;
    hasUnsavedChanges = false;
    GROUP_THRESHOLD = 10;

    constructor(options = {}) {
        this.components = options.components || [];
        this.editor = options.editor || null;
        this.permissions = options.permissions || {
            hasShowToolbarPermission: false,
        };
        this.init();
    }

    init() {
        this.createToolbar();
        // Only populate toolbar options if user has permission
        if (this.permissions && this.permissions.hasShowToolbarPermission) {
            this.populateToolbar();
        }
        // Setup change tracking
        this.setupChangeTracking();
    }

    createToolbar() {
        // Remove existing toolbar if it exists
        const existingToolbar = document.querySelector('.sdc-inline-editor-toolbar-container');
        if (existingToolbar) {
            existingToolbar.remove();
        }
        

        // Create toolbar container
        const toolbarContainer = document.createElement('div');
        toolbarContainer.classList.add('sdc-inline-editor-toolbar-container');

        
        
        // Create save button (positioned above toolbar)
        const saveButton = document.createElement('button');
        saveButton.classList.add('sdc-inline-editor-save-button');
        saveButton.classList.add('sdc-inline-editor-save-and-not-close-button')
        saveButton.type = 'button';
        saveButton.innerHTML = `
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M17 3H5C3.89 3 3 3.9 3 5V19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V7L17 3ZM19 19H5V5H16.17L19 7.83V19ZM12 12C10.34 12 9 13.34 9 15S10.34 18 12 18 15 16.66 15 15 13.66 12 12 12ZM6 6H15V10H6V6Z"/>
            </svg>
            <span class="save-button-text">Save</span>
            <span class="unsaved-indicator" style="display: none;">●</span>
        `;

        const saveAndCloseButton = document.createElement('button');
        saveAndCloseButton.classList.add('sdc-inline-editor-save-button');
        saveAndCloseButton.classList.add('sdc-inline-editor-save-and-close-button');
        saveAndCloseButton.type = 'button';
        saveAndCloseButton.innerHTML = `
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M17 3H5C3.89 3 3 3.9 3 5V19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V7L17 3ZM19 19H5V5H16.17L19 7.83V19ZM12 12C10.34 12 9 13.34 9 15S10.34 18 12 18 15 16.66 15 15 13.66 12 12 12ZM6 6H15V10H6V6Z"/>
            </svg>
            <span class="save-button-text">Save and Close</span>
            <span class="unsaved-indicator" style="display: none;">●</span>
        `;
        
        // Create edit node button
        const editNodeButton = document.createElement('button');
        editNodeButton.classList.add('sdc-inline-editor-edit-node-button');
        editNodeButton.type = 'button';
        editNodeButton.innerHTML = `
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M20.71,7.04C21.1,6.65 21.1,6 20.71,5.63L18.37,3.29C18,2.9 17.35,2.9 16.96,3.29L15.12,5.12L18.87,8.87M3,17.25V21H6.75L17.81,9.93L14.06,6.18L3,17.25Z"/>
            </svg>
            Edit page settings
        `;

        const closeButton = document.createElement('button');
        closeButton.classList.add('sdc-inline-editor-close-button');
        closeButton.type = 'button';
        closeButton.innerHTML = `
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
            </svg>
            Close without saving
        `;


        // Create button container for styling
        const buttonContainer = document.createElement('div');
        buttonContainer.classList.add('sdc-inline-editor-button-container');
        buttonContainer.appendChild(saveButton);
        buttonContainer.appendChild(saveAndCloseButton);
        buttonContainer.appendChild(editNodeButton);
        buttonContainer.appendChild(closeButton);

        // Append button container and toolbar to container
        toolbarContainer.appendChild(buttonContainer);
        
        // Create toolbar element
        if(this.permissions && this.permissions.hasShowToolbarPermission) {
            const toolbar = document.createElement('div');
                toolbar.classList.add('sdc-inline-editor-toolbar');
                toolbar.innerHTML = `
                    <div class="sdc-inline-editor-toolbar-menu"></div>
                `;
                toolbarContainer.appendChild(toolbar);
            this.toolbar = toolbar;
        }


        // Append to the body or specific container
        const targetContainer = document.body;
        targetContainer.appendChild(toolbarContainer);
        
        
        this.toolbarContainer = toolbarContainer;
        
        // Add event listener for save button and edit node button
        this.setupSaveButtons();
        this.setupEditNodeButton();
        this.setupCloseButton();
    }

    populateToolbar() {
        if(this.toolbar) {
            const toolbarMenu = this.toolbar.querySelector('.sdc-inline-editor-toolbar-menu');
            // Handle both array format and object format
            const componentsArray = Object.values(this.components);

            // If there are too many components, group them into folders by category
            if (componentsArray.length > this.GROUP_THRESHOLD) {
                const groupsMap = new Map();
                componentsArray.forEach(component => {
                    const cat = this.categorizeComponent(component) || 'misc';
                    if (!groupsMap.has(cat)) groupsMap.set(cat, []);
                    groupsMap.get(cat).push(component);
                });

                const ordered = Array.from(groupsMap.keys()).sort((a, b) => a.localeCompare(b));
                ordered.forEach(groupName => {
                    const items = groupsMap.get(groupName) || [];
                    if (items.length) {
                        const folderTrigger = this.createFolder(groupName, items);
                        toolbarMenu.appendChild(folderTrigger);
                    }
                });
            }
            else {
                // Render flat list when small
                componentsArray.forEach(component => {
                    const menuItem = this.createToolbarItem(component);
                    toolbarMenu.appendChild(menuItem);
                });
            }
        }
    }

    categorizeComponent(component) {
        if (component && component.category) {
            return String(component.category).toLowerCase();
        }
        
        return 'misc';
    }

    createFolder(groupName, components) {
        // Create a trigger element
        const trigger = document.createElement('div');
        trigger.classList.add('sdc-inline-editor-toolbar-folder-trigger');
        trigger.setAttribute('role', 'button');
        trigger.setAttribute('draggable', 'false');
        trigger.tabIndex = 0;

        // Optional folder icon (small triangle) and name
        const icon = document.createElement('span');
        icon.classList.add('sdc-inline-editor-folder-icon');
        icon.innerHTML = '▸';
        trigger.appendChild(icon);

        const title = document.createElement('span');
        title.classList.add('sdc-inline-editor-component-name');
        title.textContent = `${this.capitalize(groupName)} (${components.length})`;
        trigger.appendChild(title);

        // Click opens popup outside toolbar
        const openHandler = (e) => {
            this.openFolderPopup(groupName, components, trigger);
        };

        trigger.addEventListener('click', openHandler);
        trigger.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                openHandler(e);
            }
        });

        return trigger;
    }

    openFolderPopup(groupName, components, triggerEl) {
        if (this._folderPopup && this._activeFolderTrigger === triggerEl) {
            return;
        }
        
        const hadPopupOpen = !!this._folderPopup;
        
        if (this._folderPopup) {
            const { overlay, popup } = this._folderPopup;
            overlay.remove();
            popup.remove();
            this._folderPopup = null;
        }

        const overlay = document.createElement('div');
        overlay.classList.add('sdc-inline-editor-folder-overlay');

        const popup = document.createElement('div');
        popup.classList.add('sdc-inline-editor-folder-popup');
        popup.setAttribute('role', 'dialog');
        popup.setAttribute('aria-label', `${this.capitalize(groupName)} components`);
        
        this._activeFolderTrigger = triggerEl;

        const header = document.createElement('div');
        header.style.display = 'flex';
        header.style.justifyContent = 'space-between';
        header.style.alignItems = 'center';

        const hTitle = document.createElement('strong');
        hTitle.textContent = `${this.capitalize(groupName)}`;
        header.appendChild(hTitle);

        const closeBtn = document.createElement('button');
        closeBtn.type = 'button';
        closeBtn.textContent = 'Close';
        closeBtn.addEventListener('click', () => this.closeFolderPopup());
        header.appendChild(closeBtn);

        popup.appendChild(header);

        components.forEach(component => {
            const item = this.createToolbarItem(component);
            popup.appendChild(item);
        });

        document.body.appendChild(overlay);
        document.body.appendChild(popup);

        requestAnimationFrame(() => {
            const rect = triggerEl.getBoundingClientRect();
            const popupRect = popup.getBoundingClientRect();
            
            let left = rect.left + (rect.width / 2) - (popupRect.width / 2);
            
            const margin = 20;
            left = Math.max(margin, Math.min(left, window.innerWidth - popupRect.width - margin));
            
            const spacing = 20;
            let top = rect.top - popupRect.height - spacing;
            
            if (top < margin) {
                top = rect.bottom + spacing;
            }
            
            popup.style.left = `${left}px`;
            popup.style.top = `${top}px`;
            
            if (!hadPopupOpen) {
                this.moveButtonsAbovePopup(popup);
            }
        });

        // Store references for later close
        this._folderPopup = { overlay, popup };

        // Dismiss on ESC
        this._folderPopupKeyHandler = (e) => {
            if (e.key === 'Escape') this.closeFolderPopup();
        };
        document.addEventListener('keydown', this._folderPopupKeyHandler);

        // Notify other code so drag/drop managers can bind to new elements
        const event = new CustomEvent('toolbarPopupOpened', { detail: { groupName, popup, toolbar: this } });
        document.dispatchEvent(event);
    }

    closeFolderPopup() {
        // Only reset and clean up if there's actually a popup open
        if (!this._folderPopup) {
            return;
        }
        
        const { overlay, popup } = this._folderPopup;
        overlay.remove();
        popup.remove();
        this._folderPopup = null;
        
        if (this._folderPopupKeyHandler) {
            document.removeEventListener('keydown', this._folderPopupKeyHandler);
            this._folderPopupKeyHandler = null;
        }
        if (this._activeFolderTrigger) {
            this._activeFolderTrigger = null;
        }
        
        // Reset button positions only when actually closing
        this.resetButtonPositions();

        const event = new CustomEvent('toolbarPopupClosed', { detail: { toolbar: this } });
        document.dispatchEvent(event);
    }

    moveButtonsAbovePopup(popup) {
        const buttonContainer = this.toolbarContainer?.querySelector('.sdc-inline-editor-button-container');
        if (!buttonContainer) return;
        
        const popupRect = popup.getBoundingClientRect();
        const buttonRect = buttonContainer.getBoundingClientRect();
        
        // Calculate how much to move buttons up
        // Move them to be above the popup with some spacing
        const spacing = 10;
        const targetBottom = popupRect.top - spacing;
        const currentBottom = buttonRect.bottom;
        const moveDistance = currentBottom - targetBottom;
        
        if (moveDistance > 0) {
            buttonContainer.style.transition = 'transform 0.3s ease';
            buttonContainer.style.transform = `translateY(-${moveDistance}px)`;
        }
    }

    resetButtonPositions() {
        const buttonContainer = this.toolbarContainer?.querySelector('.sdc-inline-editor-button-container');
        if (!buttonContainer) return;
        
        buttonContainer.style.transition = 'transform 0.3s ease';
        buttonContainer.style.transform = 'translateY(0)';
    }

    capitalize(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
    }

    createToolbarItem(component) {
        const menuItem = document.createElement('div');
        menuItem.classList.add('sdc-inline-editor-toolbar-menu-item');
        menuItem.setAttribute('data-sdc-name', component.name);
        menuItem.setAttribute('draggable', 'true');
        if (component.thumbnail) {
            const thumbnail = document.createElement('img');
            thumbnail.src = component.thumbnail;
            thumbnail.classList.add('sdc-inline-editor-component-thumbnail');
            thumbnail.alt = component.friendlyName || 'Component';
            menuItem.appendChild(thumbnail);
        }
        const componentName = document.createElement('span');
        componentName.classList.add('sdc-inline-editor-component-name');
        componentName.innerHTML = component.friendlyName || 'Unknown Component';
        menuItem.appendChild(componentName);
        return menuItem;
    }

    setupSaveButtons() {
        const saveButton = this.toolbarContainer.querySelector('.sdc-inline-editor-save-and-not-close-button');
        if (saveButton && this.editor) {
            saveButton.addEventListener('click', () => {
                this.handleSave();
            });
        }
        const saveAndCloseButton = this.toolbarContainer.querySelector('.sdc-inline-editor-save-and-close-button');
        if (saveAndCloseButton && this.editor) {
            saveAndCloseButton.addEventListener('click', () => {
                this.handleSaveAndClose();
            });
        }
    }
    
    setupCloseButton() {
        const closeButton = this.toolbarContainer.querySelector('.sdc-inline-editor-close-button');
        if (closeButton && this.editor) {
            closeButton.addEventListener('click', () => {
                this.handleClose();
            });
        }
    }

    handleSave() {
        if (this.editor && typeof this.editor.save === 'function') {
            console.log('Save button clicked');
            this.editor.save();
            // Clear the unsaved indicator after save
            this.updateUnsavedIndicator(false);
        } else {
            console.warn('Editor not available or save method not found');
        }
    }

    handleSaveAndClose() {
        if (this.editor) {
            console.log('Save and close button clicked');
            this.editor.save();
            this.updateUnsavedIndicator(false);
            // Wait 500 ms before disabling editing
            setTimeout(() => {
            this.editor.sessionController.disableEditing();
            }, 500);
        } else {
            console.warn('Editor not available or saveAndClose method not found');
        }
    }

    handleClose() {
        if (this.editor) {
            this.editor.sessionController.disableEditing();
        } else {
            console.warn('Editor not available or close method not found');
        }
    }

    setupChangeTracking() {
        if (!this.editor) {
            return;
        }

        // Poll the editor's hasUnsavedChanges property
        this.changeCheckInterval = setInterval(() => {
            if (this.editor && this.editor.hasUnsavedChanges !== this.hasUnsavedChanges) {
                this.updateUnsavedIndicator(this.editor.hasUnsavedChanges);
            }
        }, 500);
    }

    updateUnsavedIndicator(hasChanges) {
        this.hasUnsavedChanges = hasChanges;
        
        const saveButtons = this.toolbarContainer
            ? Array.from(this.toolbarContainer.querySelectorAll('.sdc-inline-editor-save-button'))
            : [];
        
        if (!saveButtons.length) {
            return;
        }
        
        saveButtons.forEach((button) => {
            const indicator = button.querySelector('.unsaved-indicator');
            if (!indicator) {
                return;
            }
            
            const defaultTitle = button.dataset.defaultTitle
                || button.title
                || button.querySelector('.save-button-text')?.textContent?.trim()
                || '';
            
            button.dataset.defaultTitle = defaultTitle;
            
            if (hasChanges) {
                indicator.style.display = 'inline';
                button.classList.add('has-unsaved-changes');
                button.title = 'You have unsaved changes';
            } else {
                indicator.style.display = 'none';
                button.classList.remove('has-unsaved-changes');
                button.title = defaultTitle || 'Save';
            }
        });
    }

    setupEditNodeButton() {
        const editNodeButton = this.toolbarContainer.querySelector('.sdc-inline-editor-edit-node-button');
        if (editNodeButton && this.editor) {
            editNodeButton.addEventListener('click', () => {
                this.handleEditNode();
            });
        }
    }

    handleEditNode() {
        if (this.editor && this.editor.nodeId) {
            // Navigate to the node edit page
            const editUrl = `/node/${this.editor.nodeId}/edit`;
            window.location.href = editUrl;
        } else {
            console.warn('Editor or node ID not available');
        }
    }

    show() {
        if (this.toolbarContainer) {
            this.toolbarContainer.style.display = 'block';
        }
    }

    hide() {
        if (this.toolbarContainer) {
            this.toolbarContainer.style.display = 'none';
        }
    }

    destroy() {
        // Clear the change check interval
        if (this.changeCheckInterval) {
            clearInterval(this.changeCheckInterval);
            this.changeCheckInterval = null;
        }

        if (this.toolbarContainer) {
            this.toolbarContainer.remove();
            this.toolbarContainer = null;
            this.toolbar = null;
        }
    }

    updateComponents(newComponents) {
        this.components = newComponents;
        this.createToolbar();
        if (this.permissions && this.permissions.hasShowToolbarPermission) {
            this.populateToolbar();
        }
        
        // Trigger a custom event to notify that toolbar has been updated
        // This allows the drag and drop manager to refresh its event listeners
        const event = new CustomEvent('toolbarUpdated', {
            detail: { toolbar: this }
        });
        document.dispatchEvent(event);
    }
}
