/**
 * @file
 * Editor API Service for SDC Inline Editor
 *
 * This service handles communication with the Editor controller endpoints.
 */
class EditorService {
    constructor() {
        this.baseUrl = '/sdc-inline-editor/api/editor';
    }

    /**
     * Save component data to a node
     * @param {number} nodeId - The node ID to save to
     * @param {Array} components - Array of component data
     * @returns {Promise<Object>} - Promise resolving to save result
     */
    async saveComponents(nodeId, components) {
        try {
            const payload = {
                node_id: nodeId,
                components: components
            };

            const response = await fetch(`${this.baseUrl}/save-components`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': this.getCsrfToken()
                },
                body: JSON.stringify(payload)
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.message || 'Failed to save components');
            }

            return data;
        } catch (error) {
            console.error('Error saving components:', error);
            throw error;
        }
    }

    /**
     * Get CSRF token for API requests
     * @returns {string} - The CSRF token
     */
    getCsrfToken() {
        // Get CSRF token from Drupal settings
        if (typeof drupalSettings !== 'undefined' && drupalSettings.csrfToken) {
            return drupalSettings.csrfToken;
        }
        
        // Fallback: try to get from meta tag
        const metaToken = document.querySelector('meta[name="csrf-token"]');
        if (metaToken) {
            return metaToken.getAttribute('content');
        }
        
        // If no token found, return empty string (will likely cause server error)
        console.warn('CSRF token not found. Request may be rejected by server.');
        return '';
    }

    /**
     * Show save notification to user
     * @param {string} message - The message to show
     * @param {string} type - The notification type (success, error, info)
     */
    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `sdc-inline-editor-notification sdc-inline-editor-notification--${type}`;
        notification.textContent = message;

        // Style the notification
        Object.assign(notification.style, {
            position: 'fixed',
            top: '20px',
            right: '20px',
            padding: '12px 20px',
            borderRadius: '4px',
            color: 'white',
            fontWeight: 'bold',
            zIndex: '10000',
            transition: 'opacity 0.3s ease',
            backgroundColor: type === 'success' ? '#28a745' : type === 'error' ? '#dc3545' : '#007cba'
        });

        // Add to page
        document.body.appendChild(notification);

        // Remove after 3 seconds
        setTimeout(() => {
            notification.style.opacity = '0';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }
}

export default EditorService;
