/**
 * @file
 * SDC API Service for SDC Inline Editor
 *
 * This service handles communication with the SDC controller endpoints.
 */
class SDCService {
    constructor() {
        this.baseUrl = '/sdc-inline-editor/api/sdc';
    }

    /**
     * Get the HTML for a specific component
     * @param {string} componentName - The name of the component
     * @returns {Promise<Object>} - Promise resolving to component data with HTML
     */
    async getComponentHtml(componentName, props = {}) {
        try {
            const url = `${this.baseUrl}/component/html`;
            
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({
                    component: componentName,
                    props,
                }),
            });
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'Failed to fetch component HTML');
            }
            
            return data;
        } catch (error) {
            console.error('Error fetching component HTML:', error);
            throw error;
        }
    }

    /**
     * Get all accessible components
     * @param {string} contentType - Optional content type filter
     * @returns {Promise<Object>} - Promise resolving to list of accessible components
     */
    async getAccessibleComponents(contentType = null) {
        try {
            let url = `${this.baseUrl}/components`;
            if (contentType) {
                url += `?content_type=${encodeURIComponent(contentType)}`;
            }
            
            const response = await fetch(url);
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'Failed to fetch accessible components');
            }
            
            return data;
        } catch (error) {
            console.error('Error fetching accessible components:', error);
            throw error;
        }
    }

    /**
     * Check if a component is accessible
     * @param {string} componentName - The name of the component
     * @param {string} contentType - Optional content type
     * @returns {Promise<Object>} - Promise resolving to access check result
     */
    async checkComponentAccess(componentName, contentType = null) {
        try {
            let url = `${this.baseUrl}/component/access?component=${encodeURIComponent(componentName)}`;
            if (contentType) {
                url += `&content_type=${encodeURIComponent(contentType)}`;
            }
            
            const response = await fetch(url);
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.error || 'Failed to check component access');
            }
            
            return data;
        } catch (error) {
            console.error('Error checking component access:', error);
            throw error;
        }
    }
}

export default SDCService;
