/**
 * @file
 * DraggableElement class for SDC Inline Editor
 *
 * This class is responsible for creating a ghost element that follows the cursor when dragging an element.
 * It also creates a callback function that is called when the element is dropped.
 * 
 * It can be used for any element that needs to be dragged. 
 */
class DraggableElement {
    constructor(element, onMouseMoveCallback, onDropCallback, dropType) {
        this.element = element;
        this.ghostElement = null;
        this.handleMouseMove = this.onMouseMove.bind(this);
        this.handleMouseUp = this.onMouseUp.bind(this);
        this.onMouseMoveCallack = onMouseMoveCallback;
        this.onDropCallback = onDropCallback;
        this.dropType = dropType;
        this.isInCancelZone = false;
        this.cancelZoneElement = null;
    }

    createGhostElement() {
        if (this.ghostElement) this.ghostElement.remove();

        const rect = this.element.getBoundingClientRect();
        this.ghostElement = this.element.cloneNode(true);
        this.ghostElement.classList.add('dragging-ghost');
        
        // Store dimensions for centering
        this.ghostWidth = rect.width;
        this.ghostHeight = rect.height;
        
        Object.assign(this.ghostElement.style, {
            position: 'absolute',
            left: (rect.left + window.scrollX - this.ghostWidth / 2) + 'px',
            top: (rect.top + window.scrollY - this.ghostHeight / 2) + 'px',
            zIndex: 10000,
            pointerEvents: 'none',
            opacity: 0.8,
            width: this.ghostWidth + 'px',
            cursor: 'grabbing',
            height: this.ghostHeight + 'px',
            backgroundColor: 'rgba(0, 0, 0, 0.5)',
        });
        document.body.appendChild(this.ghostElement);
    }

    start() {
        this.createGhostElement();
        this.createCancelZone();
        document.addEventListener('mousemove', this.handleMouseMove);
        document.addEventListener('mouseup', this.handleMouseUp);
    }

    createCancelZone() {
        if (this.cancelZoneElement) {
            this.cancelZoneElement.remove();
        }
        
        const toolbar = document.querySelector('.sdc-inline-editor-toolbar-container');
        if (!toolbar) return;
        
        const toolbarRect = toolbar.getBoundingClientRect();
        
        const folderPopup = document.querySelector('.sdc-inline-editor-folder-popup');
        let topPosition = toolbarRect.top;
        
        if (folderPopup) {
            const popupRect = folderPopup.getBoundingClientRect();
            topPosition = Math.min(toolbarRect.top, popupRect.top);
        }
        
        this.cancelZoneElement = document.createElement('div');
        this.cancelZoneElement.classList.add('sdc-inline-editor-cancel-zone');
        
        Object.assign(this.cancelZoneElement.style, {
            position: 'fixed',
            top: topPosition + 'px',
            left: '0',
            right: '0',
            bottom: '0',
            zIndex: 9998,
            pointerEvents: 'none',
        });
        
        document.body.appendChild(this.cancelZoneElement);
    }

    onMouseMove(e) {
        // Center the ghost element on the cursor
        this.ghostElement.style.left = (e.clientX + window.scrollX - this.ghostWidth / 2) + 'px';
        this.ghostElement.style.top = (e.clientY + window.scrollY - this.ghostHeight / 2) + 'px';
        
        this.checkCancelZone(e);
        
        if(this.onMouseMoveCallack) {
            this.onMouseMoveCallack(e);
        }
    }

    checkCancelZone(e) {
        const toolbar = document.querySelector('.sdc-inline-editor-toolbar-container');
        if (!toolbar) return;
        
        const toolbarRect = toolbar.getBoundingClientRect();
        let cancelZoneTop = toolbarRect.top;
        
        const folderPopup = document.querySelector('.sdc-inline-editor-folder-popup');
        if (folderPopup) {
            const popupRect = folderPopup.getBoundingClientRect();
            cancelZoneTop = Math.min(toolbarRect.top, popupRect.top);
        }
        
        if (e.clientY >= cancelZoneTop) {
            if (this.ghostElement) {
                this.ghostElement.classList.add('in-cancel-zone');
            }
            this.isInCancelZone = true;
        } else {
            if (this.ghostElement) {
                this.ghostElement.classList.remove('in-cancel-zone');
            }
            this.isInCancelZone = false;
        }
    }

    onMouseUp(e) {
        if (this.cancelZoneElement) {
            this.cancelZoneElement.remove();
            this.cancelZoneElement = null;
        }
        
        this.ghostElement.remove();
        this.ghostElement = null;
        document.removeEventListener('mousemove', this.handleMouseMove);
        document.removeEventListener('mouseup', this.handleMouseUp);
        
        if (this.onDropCallback && !this.isInCancelZone) {
            this.onDropCallback(e);
        }
    }
}
export default DraggableElement;
