/**
 * @file
 * DropZone class for SDC Inline Editor
 *
 * This class is responsible for creating and managing drop zone indicators
 * that show where elements can be dropped during drag operations.
 */
class DropZone {
    constructor(targetWrapper, position, isHorizontal = false) {
        this.targetWrapper = targetWrapper;
        this.position = position; // 'above' or 'below'
        this.isHorizontal = isHorizontal; // Whether to display as vertical bar (for horizontal flex)
        this.element = null;
        this.className = 'sdc-inline-editor-drop-zone';
        
        this.create();
    }

    /**
     * Create the drop zone element
     */
    create() {
        this.element = document.createElement('div');
        this.element.classList.add(this.className);
        this.element.classList.add(`drop-zone-${this.position}`);
        this.element.setAttribute('data-target-wrapper', '');
        this.element.setAttribute('data-position', this.position);

        // Position the drop zone
        this.updatePosition();

        // Store reference to target wrapper
        this.element.targetWrapper = this.targetWrapper;
        this.element.position = this.position;

        document.body.appendChild(this.element);
    }

    /**
     * Update the position of the drop zone
     */
    updatePosition() {
        if (!this.element || !this.targetWrapper) return;

        const rect = this.targetWrapper.getBoundingClientRect();
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        // Handle 'inside' position for empty containers
        if (this.position === 'inside') {
            Object.assign(this.element.style, {
                position: 'absolute',
                left: rect.left + 'px',
                top: (rect.top + scrollTop) + 'px',
                width: rect.width + 'px',
                height: Math.max(rect.height, 100) + 'px',
                zIndex: 9999,
                pointerEvents: 'none'
            });
            
            // Add helpful text if the drop zone is empty
            if (!this.element.textContent) {
                this.element.textContent = 'Drop here to add content';
            }
        } else if (this.isHorizontal) {
            // Horizontal flex container - show vertical bar (for left/right positioning)
            Object.assign(this.element.style, {
                position: 'absolute',
                top: (rect.top + scrollTop) + 'px',
                height: rect.height + 'px',
                width: '4px',
                zIndex: 9999,
                pointerEvents: 'none'
            });

            this.element.style.setProperty('height', rect.height + 'px', 'important');

            if (this.position === 'above') {
                // 'above' in horizontal context means 'left'
                this.element.style.left = (rect.left - 2) + 'px';
            } else {
                // 'below' in horizontal context means 'right'
                this.element.style.left = (rect.right - 2) + 'px';
            }
        } else {
            // Standard vertical 'above' or 'below' positioning - show horizontal bar
            Object.assign(this.element.style, {
                position: 'absolute',
                left: rect.left + 'px',
                width: rect.width + 'px',
                height: '4px',
                zIndex: 9999,
                pointerEvents: 'none'
            });

            if (this.position === 'above') {
                this.element.style.top = (rect.top + scrollTop - 2) + 'px';
            } else {
                this.element.style.top = (rect.bottom + scrollTop - 2) + 'px';
            }
        }
    }

    /**
     * Check if the drop zone is under the given mouse coordinates
     */
    isUnderMouse(x, y) {
        if (!this.element) return false;
        
        const rect = this.element.getBoundingClientRect();
        return x >= rect.left && x <= rect.right && y >= rect.top && y <= rect.bottom;
    }

    /**
     * Get the target wrapper and position for dropping
     */
    getDropInfo() {
        return {
            targetWrapper: this.targetWrapper,
            position: this.position
        };
    }

    /**
     * Remove the drop zone from the DOM
     */
    remove() {
        if (this.element && this.element.parentNode) {
            this.element.parentNode.removeChild(this.element);
        }
        this.element = null;
    }

    /**
     * Check if the drop zone is still valid (target wrapper still exists)
     */
    isValid() {
        return this.targetWrapper && this.targetWrapper.parentNode && this.element;
    }
}

export default DropZone;
