/**
 * Date Table Field Handler
 * Creates a table-based editor for dates and descriptions
 */

const createDateTable = (sidebar, fieldName, originalElement) => {
	const container = document.createElement('div');
	container.className = 'sidebar-date-table';

	// Parse existing data
	const existingValue = sidebar.getFieldValue(originalElement, 'date-table') || '';
	const rows = parseDataToRows(existingValue);

	// Create table
	const table = document.createElement('table');
	table.className = 'date-table-editor';

	// Table header
	const thead = document.createElement('thead');
	thead.innerHTML = `
		<tr>
			<th>Date</th>
			<th>Description</th>
			<th></th>
		</tr>
	`;
	table.appendChild(thead);

	// Table body
	const tbody = document.createElement('tbody');
	tbody.className = 'date-table-body';
	table.appendChild(tbody);

	container.appendChild(table);

	// Add row button
	const addButton = document.createElement('button');
	addButton.type = 'button';
	addButton.className = 'btn btn-sm btn-primary mt-2 date-table-add-btn';
	addButton.textContent = '+ Add Row';
	addButton.addEventListener('click', () => {
		addRow(tbody, '', '', () => updateValue());
	});
	container.appendChild(addButton);

	// Initialize rows
	rows.forEach((row) => {
		addRow(tbody, row.date, row.text, () => updateValue());
	});

	// If no rows, add one empty row
	if (rows.length === 0) {
		addRow(tbody, '', '', () => updateValue());
	}

	// Function to update the field value
	const updateValue = () => {
		const value = serializeTableToData(tbody);
		sidebar.updateFieldValue(originalElement, value, 'date-table');
	};

	return container;
};

/**
 * Parse stored data format into rows
 * Format: "2025-10-15 | Description\n2025-11-04 | Another description"
 */
function parseDataToRows(data) {
	if (!data || !data.trim()) return [];

	return data
		.split('\n')
		.map((line) => line.trim())
		.filter((line) => line && line.includes('|'))
		.map((line) => {
			const parts = line.split('|').map((p) => p.trim());
			return {
				date: parts[0] || '',
				text: parts[1] || '',
			};
		});
}

/**
 * Serialize table rows back to data format
 */
function serializeTableToData(tbody) {
	const rows = Array.from(tbody.querySelectorAll('tr'));
	const data = rows
		.map((row) => {
			const dateInput = row.querySelector('.date-table-date');
			const textInput = row.querySelector('.date-table-text');
			if (!dateInput || !textInput) return null;

			const date = dateInput.value.trim();
			const text = textInput.value.trim();

			if (!date && !text) return null; // Skip empty rows

			return `${date} | ${text}`;
		})
		.filter((line) => line !== null);

	return data.join('\n');
}

/**
 * Add a row to the table
 */
function addRow(tbody, dateValue = '', textValue = '', onChange) {
	const row = document.createElement('tr');

	// Date cell
	const dateCell = document.createElement('td');
	const dateInput = document.createElement('input');
	dateInput.type = 'date';
	dateInput.className = 'date-table-date form-control form-control-sm';
	dateInput.value = dateValue;
	dateInput.addEventListener('change', onChange);
	dateCell.appendChild(dateInput);

	// Description cell
	const textCell = document.createElement('td');
	const textInput = document.createElement('input');
	textInput.type = 'text';
	textInput.className = 'date-table-text form-control form-control-sm';
	textInput.value = textValue;
	textInput.placeholder = 'Description';
	textInput.addEventListener('input', onChange);
	textCell.appendChild(textInput);

	// Delete button cell
	const deleteCell = document.createElement('td');
	deleteCell.className = 'text-end';
	const deleteBtn = document.createElement('button');
	deleteBtn.type = 'button';
	deleteBtn.className = 'btn btn-sm btn-danger date-table-delete-btn';
	deleteBtn.innerHTML = '&times;';
	deleteBtn.title = 'Delete row';
	deleteBtn.addEventListener('click', () => {
		row.remove();
		onChange();
	});
	deleteCell.appendChild(deleteBtn);

	row.appendChild(dateCell);
	row.appendChild(textCell);
	row.appendChild(deleteCell);

	tbody.appendChild(row);
}

export default createDateTable;
