import {
	ICONS,
	GENERAL_STYLE_VARIANTS,
	getIconByClassName,
	getIconClassName,
} from '../../icons.js';

const createIconSelector = (sidebar, fieldName, originalElement) => {
	const container = document.createElement('div');
	container.className = 'sidebar-icon-selector';

	const searchInput = document.createElement('input');
	searchInput.type = 'search';
	searchInput.className = 'sidebar-icon-search';
	searchInput.placeholder = 'Search icons...';
	searchInput.setAttribute('aria-label', 'Search icons');

	const grid = document.createElement('div');
	grid.className = 'sidebar-icon-grid';
	grid.setAttribute('role', 'listbox');
	grid.setAttribute('aria-label', 'Icon options');

	let selectedValue = sidebar.getFieldValue(originalElement, 'icon-selector') || '';
	const selectedIcon = getIconByClassName(selectedValue);
	let selectedVariant =
		(selectedIcon && selectedIcon.matchedVariant) ||
		(GENERAL_STYLE_VARIANTS[0] && GENERAL_STYLE_VARIANTS[0].key) ||
		null;
	let activeIcons = ICONS;

	const renderIcons = (iconList) => {
		activeIcons = iconList;
		grid.innerHTML = '';

		if (!iconList.length) {
			const emptyState = document.createElement('div');
			emptyState.className = 'sidebar-icon-empty';
			emptyState.textContent = 'No icons found. Try a different search.';
			grid.appendChild(emptyState);
			return;
		}

		iconList.forEach((icon) => {
			const className = getIconClassName(icon, selectedVariant);

			const button = document.createElement('button');
			button.type = 'button';
			button.className = 'sidebar-icon-option';
			button.setAttribute('role', 'option');
			button.setAttribute('aria-label', icon.label);
			button.setAttribute('data-icon-class', className);

			if (className === selectedValue) {
				button.classList.add('selected');
				button.setAttribute('aria-selected', 'true');
			} else {
				button.setAttribute('aria-selected', 'false');
			}

			const iconPreview = document.createElement('i');
			iconPreview.className = className;
			iconPreview.setAttribute('aria-hidden', 'true');

			const nameLabel = document.createElement('span');
			nameLabel.className = 'sidebar-icon-name';
			nameLabel.textContent = icon.label;

			button.appendChild(iconPreview);
			button.appendChild(nameLabel);

			button.addEventListener('click', () => {
				selectedValue = className;
				sidebar.updateFieldValue(originalElement, selectedValue, 'icon-selector');
				sidebar.updateComponentIconDisplay(originalElement, selectedValue);

				grid.querySelectorAll('.sidebar-icon-option').forEach((option) => {
					option.classList.remove('selected');
					option.setAttribute('aria-selected', 'false');
				});
				button.classList.add('selected');
				button.setAttribute('aria-selected', 'true');

				if (sidebar.editor && typeof sidebar.editor.markAsChanged === 'function') {
					sidebar.editor.markAsChanged();
				}
			});

			grid.appendChild(button);
		});
	};

	const handleSearch = (event) => {
		const query = event.target.value.trim().toLowerCase();

		if (!query) {
			renderIcons(ICONS);
			return;
		}

		const filtered = ICONS.filter((icon) => icon.searchTerms.includes(query));
		renderIcons(filtered);
	};

	const styleToggleButtons = new Map();

	const updateStyleToggleState = () => {
		styleToggleButtons.forEach((button, key) => {
			const isActive = key === selectedVariant;
			button.classList.toggle('active', isActive);
			button.setAttribute('aria-pressed', isActive ? 'true' : 'false');
		});
	};

	const applyVariantToSelection = () => {
		if (!selectedValue) {
			return;
		}

		const match = getIconByClassName(selectedValue);

		if (!match || match.style !== 'general') {
			return;
		}

		const updatedClass = getIconClassName(match, selectedVariant);

		if (updatedClass === selectedValue) {
			return;
		}

		selectedValue = updatedClass;
		sidebar.updateFieldValue(originalElement, selectedValue, 'icon-selector');
		sidebar.updateComponentIconDisplay(originalElement, selectedValue);

		if (sidebar.editor && typeof sidebar.editor.markAsChanged === 'function') {
			sidebar.editor.markAsChanged();
		}
	};

	const handleVariantChange = (variantKey) => {
		if (variantKey === selectedVariant) {
			return;
		}

		const variantExists = GENERAL_STYLE_VARIANTS.some((variant) => variant.key === variantKey);

		if (!variantExists) {
			return;
		}

		selectedVariant = variantKey;
		updateStyleToggleState();
		applyVariantToSelection();
		renderIcons(activeIcons);
	};

	const styleToggle =
		GENERAL_STYLE_VARIANTS.length > 1 ? document.createElement('div') : null;

	if (styleToggle) {
		styleToggle.className = 'sidebar-icon-style-toggle';
		styleToggle.setAttribute('role', 'group');
		styleToggle.setAttribute('aria-label', 'Icon style');

		GENERAL_STYLE_VARIANTS.forEach((variant) => {
			const button = document.createElement('button');
			button.type = 'button';
			button.className = 'sidebar-icon-style-option';
			button.textContent = variant.label;
			button.setAttribute('data-variant', variant.key);
			button.setAttribute('aria-pressed', 'false');

			button.addEventListener('click', () => {
				handleVariantChange(variant.key);
			});

			styleToggleButtons.set(variant.key, button);
			styleToggle.appendChild(button);
		});
	}

	searchInput.addEventListener('input', handleSearch);

	container.appendChild(searchInput);
	if (styleToggle) {
		container.appendChild(styleToggle);
		updateStyleToggleState();
	}
	container.appendChild(grid);

	renderIcons(ICONS);

	return container;
};

export default createIconSelector;


