import { ImageCropper } from '../../utils/ImageCropper.js';

const createImageSelector = (sidebar, fieldName, originalElement) => {
	const container = document.createElement('div');
	container.className = 'sidebar-image-selector';

	const currentImageUuid = sidebar.getFieldValue(originalElement, 'image-selector');
	
	// Find the imageCrop field element
	const component = originalElement.closest('[data-sdc-component]');
	const imageCropElement = component ? component.querySelector('[data-sdc-field="imageCrop"]') : null;
	const currentCropData = imageCropElement ? sidebar.getFieldValue(imageCropElement, 'image-crop') : null;
	
	let selectedImageData = null; // Store full image data
	let cropperInstance = null;

	const preview = document.createElement('div');
	preview.className = 'image-preview';
	if (currentImageUuid) {
		sidebar.loadImagePreview(currentImageUuid, preview);
	} else {
		preview.innerHTML = '<div class="no-image">No image selected</div>';
	}

	// Button container
	const buttonContainer = document.createElement('div');
	buttonContainer.className = 'image-selector-buttons';

	const selectBtn = document.createElement('button');
	selectBtn.type = 'button';
	selectBtn.className = 'select-image-btn';
	selectBtn.textContent = 'Select Image';

	const cropBtn = document.createElement('button');
	cropBtn.type = 'button';
	cropBtn.className = 'crop-image-btn';
	cropBtn.innerHTML = '<i class="fa-solid fa-crop" aria-hidden="true"></i> Crop Image';
	cropBtn.style.display = currentImageUuid ? 'block' : 'none';

	// Cropper container (initially hidden)
	const cropperContainer = document.createElement('div');
	cropperContainer.className = 'image-cropper-wrapper';
	cropperContainer.style.display = 'none';

	selectBtn.addEventListener('click', () => {
		sidebar.openMediaLibrary((selectedImage) => {
			selectedImageData = selectedImage[0];
			sidebar.updateFieldValue(originalElement, selectedImageData.uuid, 'image-selector');
			preview.innerHTML = `<img src="${selectedImageData.thumbnail}" alt="Preview" />`;
			cropBtn.style.display = 'block';
		}, false);
	});

	cropBtn.addEventListener('click', () => {
		if (!selectedImageData && !currentImageUuid) {
			return;
		}
		
		if (!imageCropElement) {
			console.error('Could not find imageCrop field element');
			return;
		}

		// Get the image URL from the rendered component
		let imageUrl = '';
		if (component) {
			const img = component.querySelector('img');
			if (img && img.src) {
				imageUrl = img.src;
			}
		}
		
		// Fallback to selectedImageData if available
		if (!imageUrl && selectedImageData) {
			imageUrl = selectedImageData.url || selectedImageData.thumbnail || '';
		}
		
		if (!imageUrl) {
			console.error('No image URL found for cropping');
			alert('Please save the component first, then try cropping the image.');
			return;
		}

		// Show cropper interface
		cropperContainer.style.display = 'block';
		buttonContainer.style.display = 'none';
		preview.style.display = 'none';

		if (!cropperInstance) {
			// Get current crop data from the imageCrop field
			const latestCropData = imageCropElement ? sidebar.getFieldValue(imageCropElement, 'image-crop') : null;
			
			// Parse existing crop data if available
			let initialCrop = { x: 50, y: 50, scale: 1 };
			if (latestCropData && typeof latestCropData === 'object') {
				initialCrop = { ...initialCrop, ...latestCropData };
			}

			cropperInstance = new ImageCropper({
				imageUrl: imageUrl,
				aspectRatio: null, // Allow free cropping
				initialCrop: initialCrop,
				onCropChange: (cropData) => {
					// Store crop data in the imageCrop field element
					console.log('[imageSelector] Crop changed:', cropData);
					sidebar.updateFieldValue(imageCropElement, JSON.stringify(cropData), 'image-crop');
				},
			});

			const cropperElement = cropperInstance.createCropperElement();
			cropperContainer.appendChild(cropperElement);

			// Done button
			const doneBtn = document.createElement('button');
			doneBtn.type = 'button';
			doneBtn.className = 'crop-done-btn';
			doneBtn.textContent = 'Done';

			doneBtn.addEventListener('click', () => {
				cropperContainer.style.display = 'none';
				buttonContainer.style.display = 'flex';
				preview.style.display = 'flex';
			});

			cropperContainer.appendChild(doneBtn);
		} else {
			// Update existing cropper with new image if needed
			if (imageUrl && imageUrl !== cropperInstance.imageUrl) {
				cropperInstance.updateImage(imageUrl);
			}
		}
	});

	buttonContainer.appendChild(selectBtn);
	buttonContainer.appendChild(cropBtn);

	container.appendChild(preview);
	container.appendChild(buttonContainer);
	container.appendChild(cropperContainer);

	return container;
};

export default createImageSelector;


