/**
 * Link List Manager Field Handler
 * Creates a table-based editor for managing link list items
 */

/**
 * Extract link item data from a DOM element (link-list-item component)
 */
function extractItemFromDOM(itemElement) {
	// Helper to get field value from hidden field elements
	const getFieldValue = (fieldName, fieldType) => {
		const fieldElement = itemElement.querySelector(`[data-sdc-field="${fieldName}"]`);
		if (!fieldElement) return null;
		
		const propAttribute = `data-sdc-prop-${fieldType}`;
		const value = fieldElement.getAttribute(propAttribute);
		return value !== null ? value : null;
	};
	
	// Get values from hidden fields first (source of truth), fallback to rendered DOM
	const text = getFieldValue('text', 'text-input') || itemElement.querySelector('a')?.textContent?.trim() || '';
	const url = getFieldValue('url', 'text-input') || itemElement.querySelector('a')?.getAttribute('href') || '#';
	const target = getFieldValue('target', 'select-dropdown') || itemElement.querySelector('a')?.getAttribute('target') || '_self';
	
	// Get popover data from hidden fields
	const hoverText = getFieldValue('hover_text', 'text-editor') || itemElement.getAttribute('data-bs-content') || '';
	const popoverPlacement = getFieldValue('popover_placement', 'select-dropdown') || itemElement.getAttribute('data-bs-placement') || 'left';
	const popoverTrigger = getFieldValue('popover_trigger', 'select-dropdown') || itemElement.getAttribute('data-bs-trigger') || 'hover focus';
	
	// Check enable_popover from hidden field
	const enablePopoverValue = getFieldValue('enable_popover', 'checkbox-toggle');
	let enablePopover = false;
	if (enablePopoverValue !== null) {
		enablePopover = enablePopoverValue === 'true' || enablePopoverValue === true || enablePopoverValue === '1';
	} else {
		// Fallback: check if popover is enabled based on DOM attributes
		enablePopover = itemElement.hasAttribute('data-bs-toggle') && hoverText !== '';
	}
	
	return {
		text: text,
		url: url,
		target: target,
		hover_text: hoverText,
		popover_placement: popoverPlacement,
		popover_trigger: popoverTrigger,
		enable_popover: enablePopover
	};
}

/**
 * Read all link items from the DOM
 */
function readItemsFromDOM(component) {
	const dropContainer = component?.querySelector('[data-drop-container="true"]');
	if (!dropContainer) return [];
	
	const existingListItems = dropContainer.querySelectorAll('[data-sdc-component="mse_sdc_components:link-list-item"]');
	const items = [];
	existingListItems.forEach((item) => {
		items.push(extractItemFromDOM(item));
	});
	return items;
}

const createLinkListManager = (sidebar, fieldName, originalElement) => {
	const container = document.createElement('div');
	container.className = 'sidebar-link-list-manager';
	container.style.cssText = 'width: 100%;';

	// Get the component element to find existing link items
	const component = originalElement.closest('[data-sdc-component]');
	const dropContainer = component?.querySelector('[data-drop-container="true"]');
	
	// Parse existing data - always sync from DOM first (most up-to-date)
	let existingItems = readItemsFromDOM(component);
	
	// Ensure existingItems is always an array
	if (!Array.isArray(existingItems)) {
		existingItems = [];
	}
	
	// If no items in DOM, try to get from stored value
	if (existingItems.length === 0) {
		const storedValue = sidebar.getFieldValue(originalElement, 'link-list-manager');
		if (storedValue && storedValue !== '[]' && storedValue !== '') {
			try {
				const parsed = JSON.parse(storedValue);
				// Ensure parsed value is an array
				if (Array.isArray(parsed)) {
					existingItems = parsed;
				}
			} catch (e) {
				console.warn('Failed to parse stored link list data:', e);
				existingItems = [];
			}
		}
	}

	// Create table
	const table = document.createElement('table');
	table.className = 'link-list-manager-table';
	table.style.cssText = 'width: 100%; border-collapse: collapse; margin-bottom: 12px; background: white; border-radius: 4px; overflow: hidden;';

	// Table header
	const thead = document.createElement('thead');
	thead.style.cssText = 'background: #f8f9fa;';
	thead.innerHTML = `
		<tr>
			<th style="padding: 10px 12px; text-align: left; border-bottom: 2px solid #e9ecef; font-weight: 600; font-size: 13px; color: #495057;">Text</th>
			<th style="padding: 10px 12px; text-align: left; border-bottom: 2px solid #e9ecef; font-weight: 600; font-size: 13px; color: #495057;">URL</th>
			<th style="padding: 10px 12px; text-align: center; border-bottom: 2px solid #e9ecef; font-weight: 600; font-size: 13px; color: #495057; width: 120px;">Target</th>
			<th style="padding: 10px 12px; text-align: center; border-bottom: 2px solid #e9ecef; font-weight: 600; font-size: 13px; color: #495057; width: 100px;"></th>
		</tr>
	`;
	table.appendChild(thead);

	// Table body
	const tbody = document.createElement('tbody');
	tbody.className = 'link-list-manager-body';
	table.appendChild(tbody);

	container.appendChild(table);

	// Bulk input container for updating link items
	const bulkContainer = document.createElement('div');
	bulkContainer.style.cssText =
		'margin-top: 16px; padding: 10px 12px; border-radius: 4px; background: #f8f9fa; border: 1px solid #e9ecef;';

	const bulkLabel = document.createElement('div');
	bulkLabel.textContent = 'Bulk links (one per line)';
	bulkLabel.style.cssText =
		'font-size: 12px; font-weight: 600; margin-bottom: 4px; color: #495057;';

	const bulkHelp = document.createElement('div');
	bulkHelp.textContent =
		'Use "Populate Text" to fill the Text column, or "Populate URLs" to fill the URL column from each line.';
	bulkHelp.style.cssText =
		'font-size: 11px; color: #6c757d; margin-bottom: 6px;';

	const bulkTextarea = document.createElement('textarea');
	bulkTextarea.className = 'link-list-bulk-input';
	bulkTextarea.rows = 4;
	bulkTextarea.placeholder =
		'One value per line\nExample:\nMy Link Text\nhttps://example.com';
	bulkTextarea.style.cssText =
		'width: 100%; padding: 8px 10px; border-radius: 4px; border: 1px solid #ddd; font-size: 13px; resize: vertical; margin-bottom: 8px;';

	const bulkButtonRow = document.createElement('div');
	bulkButtonRow.style.cssText =
		'display: flex; justify-content: flex-end; gap: 8px;';

	const getBulkLines = () => {
		const raw = bulkTextarea.value;
		if (raw === null || raw === undefined || raw === '') {
			return [];
		}
		return raw.split(/\r?\n/).map((line) => line.trim());
	};

	const getMainRows = () => {
		const rows = Array.from(tbody.querySelectorAll('tr'));
		const mainRows = [];
		for (let i = 0; i < rows.length; i += 2) {
			const row = rows[i];
			if (row && !row.classList.contains('link-list-advanced-row')) {
				mainRows.push(row);
			}
		}
		return mainRows;
	};

	const populateTextButton = document.createElement('button');
	populateTextButton.type = 'button';
	populateTextButton.className = 'btn btn-sm';
	populateTextButton.textContent = 'Populate Text';
	populateTextButton.title = 'Use each line to populate the Text field of existing items.';
	populateTextButton.style.cssText =
		'background: #007cba; color: white; border: none; padding: 6px 12px; border-radius: 4px; font-size: 12px; cursor: pointer; transition: background-color 0.2s ease;';
	populateTextButton.addEventListener('mouseenter', () => {
		populateTextButton.style.background = '#0062a1';
	});
	populateTextButton.addEventListener('mouseleave', () => {
		populateTextButton.style.background = '#007cba';
	});

	populateTextButton.addEventListener('click', () => {
		const lines = getBulkLines();
		if (!lines.length) {
			return;
		}

		const mainRows = getMainRows();
		const existingCount = mainRows.length;
		const lineCount = lines.length;

		if (!existingCount) {
			lines.forEach((line) => {
				const value = line || '';
				addRow(
					tbody,
					{
						text: value,
						url: '',
						target: '_self',
						hover_text: '',
						popover_placement: 'left',
						popover_trigger: 'hover focus',
						enable_popover: false,
					},
					() => updateValue()
				);
			});
		} else {
			const common = Math.min(existingCount, lineCount);
			for (let i = 0; i < common; i++) {
				const row = mainRows[i];
				const textInput = row.querySelector('.link-list-text');
				if (textInput) {
					textInput.value = lines[i] || '';
				}
			}

			if (lineCount > existingCount) {
				for (let i = existingCount; i < lineCount; i++) {
					const value = lines[i] || '';
					addRow(
						tbody,
						{
							text: value,
							url: '',
							target: '_self',
							hover_text: '',
							popover_placement: 'left',
							popover_trigger: 'hover focus',
							enable_popover: false,
						},
						() => updateValue()
					);
				}
			}
		}

		updateValue();
	});

	const populateUrlButton = document.createElement('button');
	populateUrlButton.type = 'button';
	populateUrlButton.className = 'btn btn-sm';
	populateUrlButton.textContent = 'Populate URLs';
	populateUrlButton.title =
		'Use each line to populate the URL field of existing items (Text will default to the same value when creating new items).';
	populateUrlButton.style.cssText =
		'background: #6c757d; color: white; border: none; padding: 6px 12px; border-radius: 4px; font-size: 12px; cursor: pointer; transition: background-color 0.2s ease;';
	populateUrlButton.addEventListener('mouseenter', () => {
		populateUrlButton.style.background = '#545b62';
	});
	populateUrlButton.addEventListener('mouseleave', () => {
		populateUrlButton.style.background = '#6c757d';
	});

	populateUrlButton.addEventListener('click', () => {
		const lines = getBulkLines();
		if (!lines.length) {
			return;
		}

		const mainRows = getMainRows();
		const existingCount = mainRows.length;
		const lineCount = lines.length;

		if (!existingCount) {
			lines.forEach((line) => {
				const value = line || '';
				addRow(
					tbody,
					{
						text: value,
						url: value,
						target: '_self',
						hover_text: '',
						popover_placement: 'left',
						popover_trigger: 'hover focus',
						enable_popover: false,
					},
					() => updateValue()
				);
			});
		} else {
			const common = Math.min(existingCount, lineCount);
			for (let i = 0; i < common; i++) {
				const row = mainRows[i];
				const urlInput = row.querySelector('.link-list-url');
				if (urlInput) {
					urlInput.value = lines[i] || '';
				}
			}

			if (lineCount > existingCount) {
				for (let i = existingCount; i < lineCount; i++) {
					const value = lines[i] || '';
					addRow(
						tbody,
						{
							text: value,
							url: value,
							target: '_self',
							hover_text: '',
							popover_placement: 'left',
							popover_trigger: 'hover focus',
							enable_popover: false,
						},
						() => updateValue()
					);
				}
			}
		}

		updateValue();
	});

	bulkButtonRow.appendChild(populateTextButton);
	bulkButtonRow.appendChild(populateUrlButton);

	bulkContainer.appendChild(bulkLabel);
	bulkContainer.appendChild(bulkHelp);
	bulkContainer.appendChild(bulkTextarea);
	bulkContainer.appendChild(bulkButtonRow);

	container.appendChild(bulkContainer);

	// Button container
	const buttonContainer = document.createElement('div');
	buttonContainer.style.cssText = 'margin-top: 12px; display: flex; gap: 8px;';
	
	// Refresh button to sync from DOM
	const refreshButton = document.createElement('button');
	refreshButton.type = 'button';
	refreshButton.className = 'btn btn-sm';
	refreshButton.textContent = 'Sync';
	refreshButton.title = 'Refresh list from items on the page';
	refreshButton.style.cssText = 'background: #6c757d; color: white; border: none; padding: 6px 12px; border-radius: 4px; font-size: 13px; cursor: pointer; transition: background-color 0.2s ease;';
	refreshButton.addEventListener('mouseenter', () => {
		refreshButton.style.background = '#5a6268';
	});
	refreshButton.addEventListener('mouseleave', () => {
		refreshButton.style.background = '#6c757d';
	});
	refreshButton.addEventListener('click', () => {
		syncFromDOM();
	});
	buttonContainer.appendChild(refreshButton);
	
	// Add row button
	const addButton = document.createElement('button');
	addButton.type = 'button';
	addButton.className = 'btn btn-sm';
	addButton.textContent = '+ Add Link';
	addButton.style.cssText = 'flex: 1; background: #007cba; color: white; border: none; padding: 6px 12px; border-radius: 4px; font-size: 13px; font-weight: 500; cursor: pointer; transition: background-color 0.2s ease;';
	addButton.addEventListener('mouseenter', () => {
		addButton.style.background = '#005a87';
	});
	addButton.addEventListener('mouseleave', () => {
		addButton.style.background = '#007cba';
	});
	addButton.addEventListener('click', () => {
		addRow(tbody, {
			text: '',
			url: '',
			target: '_self',
			hover_text: '',
			popover_placement: 'left',
			popover_trigger: 'hover focus',
			enable_popover: false
		}, () => updateValue());
	});
	buttonContainer.appendChild(addButton);
	
	container.appendChild(buttonContainer);

	// Initialize rows - ensure existingItems is an array before iterating
	if (Array.isArray(existingItems)) {
		existingItems.forEach((item) => {
			addRow(tbody, item, () => updateValue());
		});
	}

	// If no rows, add one empty row
	if (existingItems.length === 0) {
		addRow(tbody, {
			text: '',
			url: '',
			target: '_self',
			hover_text: '',
			popover_placement: 'left',
			popover_trigger: 'hover focus',
			enable_popover: false
		}, () => updateValue());
	}

	// Function to sync from DOM and refresh the table
	const syncFromDOM = () => {
		let domItems = readItemsFromDOM(component);
		// Always sync, even if empty (to clear deleted items)
		
		// Ensure domItems is an array
		if (!Array.isArray(domItems)) {
			console.warn('domItems is not an array, resetting to empty array');
			domItems = [];
		}
		
		// Clear existing rows (both regular and advanced rows)
		const existingRows = Array.from(tbody.querySelectorAll('tr'));
		existingRows.forEach(row => row.remove());
		
		// Add rows from DOM
		if (domItems.length > 0) {
			domItems.forEach((item) => {
				addRow(tbody, item, () => updateValue());
			});
		} else {
			// If no items, add one empty row
			addRow(tbody, {
				text: '',
				url: '',
				target: '_self',
				hover_text: '',
				popover_placement: 'left',
				popover_trigger: 'hover focus',
				enable_popover: false
			}, () => updateValue());
		}
		
		// Update stored value - this ensures the table data is saved
		updateValue();
	};

	// Function to update the field value
	const updateValue = () => {
		const value = serializeTableToData(tbody);
		sidebar.updateFieldValue(originalElement, value, 'link-list-manager');
	};

	// Listen for component added events to sync
	const componentAddedHandler = (event) => {
		// Check if the added component is a link-list-item in our container
		const addedComponent = event.detail?.componentElement;
		if (addedComponent && dropContainer && dropContainer.contains(addedComponent)) {
			// Small delay to ensure DOM is updated
			setTimeout(() => {
				syncFromDOM();
			}, 100);
		}
	};
	
	// Listen for component changes (when items are edited via their own sidebar)
	const sidebarFieldChangedHandler = (event) => {
		const changedElement = event.detail?.element;
		if (changedElement && dropContainer && dropContainer.contains(changedElement.closest('[data-sdc-component="mse_sdc_components:link-list-item"]'))) {
			// Sync after a short delay to ensure all changes are applied
			setTimeout(() => {
				syncFromDOM();
			}, 100);
		}
	};
	
	document.addEventListener('componentAdded', componentAddedHandler);
	document.addEventListener('sidebar-field-changed', sidebarFieldChangedHandler);
	
	// Clean up listeners when sidebar is closed (stored in container for cleanup)
	container._cleanup = () => {
		document.removeEventListener('componentAdded', componentAddedHandler);
		document.removeEventListener('sidebar-field-changed', sidebarFieldChangedHandler);
	};
	
	// Expose sync function so it can be called before saving
	container._syncFromDOM = syncFromDOM;
	
	// Expose function to get current table data (for saving)
	container._getTableData = () => {
		return serializeTableToData(tbody);
	};

	return container;
};

/**
 * Serialize table rows back to JSON data format
 */
function serializeTableToData(tbody) {
	const rows = Array.from(tbody.querySelectorAll('tr'));
	const items = [];
	
	// Process rows in pairs (main row + advanced row)
	for (let i = 0; i < rows.length; i += 2) {
		const row = rows[i];
		const advancedRow = rows[i + 1]; // Advanced row follows main row
		
		// Skip if this is an advanced row (it will be processed with its main row)
		if (row.classList.contains('link-list-advanced-row')) {
			continue;
		}
		
		const textInput = row.querySelector('.link-list-text');
		const urlInput = row.querySelector('.link-list-url');
		const targetSelect = row.querySelector('.link-list-target');
		
		// Get advanced options from the advanced row
		const hoverTextInput = advancedRow?.querySelector('.link-list-hover-text');
		const popoverPlacementSelect = advancedRow?.querySelector('.link-list-popover-placement');
		const popoverTriggerSelect = advancedRow?.querySelector('.link-list-popover-trigger');
		const enablePopoverCheckbox = advancedRow?.querySelector('.link-list-enable-popover');
		
		if (!textInput || !urlInput) continue;

		const text = textInput.value.trim();
		const url = urlInput.value.trim();

		// Skip empty rows
		if (!text && !url) continue;

		items.push({
			text: text || 'Link',
			url: url || '#',
			target: targetSelect?.value || '_self',
			hover_text: hoverTextInput?.value?.trim() || '',
			popover_placement: popoverPlacementSelect?.value || 'left',
			popover_trigger: popoverTriggerSelect?.value || 'hover focus',
			enable_popover: enablePopoverCheckbox?.checked || false
		});
	}

	return JSON.stringify(items);
}

/**
 * Add a row to the table
 */
function addRow(tbody, itemData = {}, onChange) {
	const row = document.createElement('tr');
	row.style.cssText = 'border-bottom: 1px solid #e9ecef; transition: background-color 0.15s ease;';
	row.addEventListener('mouseenter', () => {
		row.style.backgroundColor = '#f8f9fa';
	});
	row.addEventListener('mouseleave', () => {
		row.style.backgroundColor = 'transparent';
	});

	// Text cell
	const textCell = document.createElement('td');
	textCell.style.cssText = 'padding: 10px 12px;';
	const textInput = document.createElement('input');
	textInput.type = 'text';
	textInput.className = 'link-list-text sidebar-text-input';
	textInput.value = itemData.text || '';
	textInput.placeholder = 'Link text';
	textInput.style.cssText = 'width: 100%; padding: 6px 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px; transition: border-color 0.2s ease, box-shadow 0.2s ease;';
	textInput.addEventListener('focus', () => {
		textInput.style.borderColor = '#007cba';
		textInput.style.boxShadow = '0 0 0 2px rgba(0, 124, 186, 0.2)';
	});
	textInput.addEventListener('blur', () => {
		textInput.style.borderColor = '#ddd';
		textInput.style.boxShadow = 'none';
	});
	textInput.addEventListener('input', onChange);
	textCell.appendChild(textInput);

	// URL cell
	const urlCell = document.createElement('td');
	urlCell.style.cssText = 'padding: 10px 12px;';
	const urlInput = document.createElement('input');
	urlInput.type = 'url';
	urlInput.className = 'link-list-url sidebar-text-input';
	urlInput.value = itemData.url || '';
	urlInput.placeholder = 'https://...';
	urlInput.style.cssText = 'width: 100%; padding: 6px 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px; transition: border-color 0.2s ease, box-shadow 0.2s ease;';
	urlInput.addEventListener('focus', () => {
		urlInput.style.borderColor = '#007cba';
		urlInput.style.boxShadow = '0 0 0 2px rgba(0, 124, 186, 0.2)';
	});
	urlInput.addEventListener('blur', () => {
		urlInput.style.borderColor = '#ddd';
		urlInput.style.boxShadow = 'none';
	});
	urlInput.addEventListener('input', onChange);
	urlCell.appendChild(urlInput);

	// Target cell
	const targetCell = document.createElement('td');
	targetCell.style.cssText = 'padding: 10px 12px; text-align: center;';
	const targetSelect = document.createElement('select');
	targetSelect.className = 'link-list-target sidebar-select-dropdown';
	targetSelect.style.cssText = 'width: 100%; padding: 6px 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px; background: white; color: #000; cursor: pointer; transition: border-color 0.2s ease, box-shadow 0.2s ease;';
	targetSelect.innerHTML = `
		<option value="_self" ${itemData.target === '_self' ? 'selected' : ''}>Same</option>
		<option value="_blank" ${itemData.target === '_blank' ? 'selected' : ''}>New</option>
	`;
	targetSelect.addEventListener('focus', () => {
		targetSelect.style.borderColor = '#007cba';
		targetSelect.style.boxShadow = '0 0 0 2px rgba(0, 124, 186, 0.2)';
	});
	targetSelect.addEventListener('blur', () => {
		targetSelect.style.borderColor = '#ddd';
		targetSelect.style.boxShadow = 'none';
	});
	targetSelect.addEventListener('change', onChange);
	targetCell.appendChild(targetSelect);

	// Actions cell (with expand/collapse for advanced options)
	const actionsCell = document.createElement('td');
	actionsCell.style.cssText = 'padding: 10px 12px; text-align: center;';
	
	const actionsContainer = document.createElement('div');
	actionsContainer.style.cssText = 'display: flex; gap: 6px; justify-content: center; align-items: center;';
	
	// Expand button for advanced options
	const expandBtn = document.createElement('button');
	expandBtn.type = 'button';
	expandBtn.className = 'link-list-expand-btn';
	expandBtn.textContent = '...';
	expandBtn.title = 'Advanced options';
	expandBtn.style.cssText = 'background: #f8f9fa; border: 1px solid #dee2e6; color: #495057; padding: 4px 8px; border-radius: 4px; font-size: 14px; cursor: pointer; transition: all 0.2s ease; width: 28px; height: 28px; display: flex; align-items: center; justify-content: center;';
	expandBtn.addEventListener('mouseenter', () => {
		expandBtn.style.background = '#e9ecef';
		expandBtn.style.borderColor = '#adb5bd';
	});
	expandBtn.addEventListener('mouseleave', () => {
		expandBtn.style.background = '#f8f9fa';
		expandBtn.style.borderColor = '#dee2e6';
	});
	
	// Delete button
	const deleteBtn = document.createElement('button');
	deleteBtn.type = 'button';
	deleteBtn.className = 'link-list-delete-btn';
	deleteBtn.textContent = '×';
	deleteBtn.title = 'Delete link';
	deleteBtn.style.cssText = 'background: #dc3545; border: none; color: white; padding: 4px 8px; border-radius: 4px; font-size: 18px; line-height: 1; cursor: pointer; transition: background-color 0.2s ease; width: 28px; height: 28px; display: flex; align-items: center; justify-content: center; font-weight: bold;';
	deleteBtn.addEventListener('mouseenter', () => {
		deleteBtn.style.background = '#c82333';
	});
	deleteBtn.addEventListener('mouseleave', () => {
		deleteBtn.style.background = '#dc3545';
	});
	deleteBtn.addEventListener('click', () => {
		// Find the next element sibling (advanced row) before removing the main row
		let nextElement = row.nextElementSibling;
		// Remove the main row
		row.remove();
		// Also remove the corresponding advanced row if it exists and is the expected advanced row
		if (nextElement && nextElement.classList && nextElement.classList.contains('link-list-advanced-row')) {
			nextElement.remove();
		}
		onChange();
	});
	
	actionsContainer.appendChild(expandBtn);
	actionsContainer.appendChild(deleteBtn);
	actionsCell.appendChild(actionsContainer);

	// Advanced options row (initially hidden)
	const advancedRow = document.createElement('tr');
	advancedRow.className = 'link-list-advanced-row';
	advancedRow.style.cssText = 'display: none; background: #f8f9fa; border-top: 2px solid #e9ecef;';
	
	const advancedCell = document.createElement('td');
	advancedCell.colSpan = 4;
	advancedCell.style.cssText = 'padding: 16px 12px;';
	
	const advancedContainer = document.createElement('div');
	advancedContainer.style.cssText = 'display: grid; grid-template-columns: 1fr 1fr; gap: 16px;';
	
	// Hover text
	const hoverTextGroup = document.createElement('div');
	const hoverTextLabel = document.createElement('label');
	hoverTextLabel.textContent = 'Hover Text (Popover)';
	hoverTextLabel.style.cssText = 'font-size: 12px; font-weight: 600; margin-bottom: 6px; display: block; color: #495057;';
	const hoverTextInput = document.createElement('input');
	hoverTextInput.type = 'text';
	hoverTextInput.className = 'link-list-hover-text sidebar-text-input';
	hoverTextInput.value = itemData.hover_text || '';
	hoverTextInput.placeholder = 'Optional hover text';
	hoverTextInput.style.cssText = 'width: 100%; padding: 6px 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px; transition: border-color 0.2s ease, box-shadow 0.2s ease;';
	hoverTextInput.addEventListener('focus', () => {
		hoverTextInput.style.borderColor = '#007cba';
		hoverTextInput.style.boxShadow = '0 0 0 2px rgba(0, 124, 186, 0.2)';
	});
	hoverTextInput.addEventListener('blur', () => {
		hoverTextInput.style.borderColor = '#ddd';
		hoverTextInput.style.boxShadow = 'none';
	});
	hoverTextInput.addEventListener('input', onChange);
	hoverTextGroup.appendChild(hoverTextLabel);
	hoverTextGroup.appendChild(hoverTextInput);
	
	// Enable popover checkbox
	const enablePopoverGroup = document.createElement('div');
	const enablePopoverLabel = document.createElement('label');
	enablePopoverLabel.style.cssText = 'display: flex; align-items: center; gap: 8px; font-size: 13px; color: #495057; cursor: pointer; margin-top: 24px;';
	const enablePopoverCheckbox = document.createElement('input');
	enablePopoverCheckbox.type = 'checkbox';
	enablePopoverCheckbox.className = 'link-list-enable-popover';
	enablePopoverCheckbox.checked = itemData.enable_popover || false;
	enablePopoverCheckbox.style.cssText = 'width: 16px; height: 16px; accent-color: #007cba; cursor: pointer;';
	enablePopoverCheckbox.addEventListener('change', onChange);
	enablePopoverLabel.appendChild(enablePopoverCheckbox);
	enablePopoverLabel.appendChild(document.createTextNode('Enable Popover'));
	enablePopoverGroup.appendChild(enablePopoverLabel);
	
	// Popover placement
	const popoverPlacementGroup = document.createElement('div');
	const popoverPlacementLabel = document.createElement('label');
	popoverPlacementLabel.textContent = 'Popover Placement';
	popoverPlacementLabel.style.cssText = 'font-size: 12px; font-weight: 600; margin-bottom: 6px; display: block; color: #495057;';
	const popoverPlacementSelect = document.createElement('select');
	popoverPlacementSelect.className = 'link-list-popover-placement sidebar-select-dropdown';
	popoverPlacementSelect.style.cssText = 'width: 100%; padding: 6px 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px; background: white; color: #000; cursor: pointer; transition: border-color 0.2s ease, box-shadow 0.2s ease;';
	popoverPlacementSelect.innerHTML = `
		<option value="left" ${itemData.popover_placement === 'left' ? 'selected' : ''}>Left</option>
		<option value="right" ${itemData.popover_placement === 'right' ? 'selected' : ''}>Right</option>
		<option value="top" ${itemData.popover_placement === 'top' ? 'selected' : ''}>Top</option>
		<option value="bottom" ${itemData.popover_placement === 'bottom' ? 'selected' : ''}>Bottom</option>
		<option value="auto" ${itemData.popover_placement === 'auto' ? 'selected' : ''}>Auto</option>
	`;
	popoverPlacementSelect.addEventListener('focus', () => {
		popoverPlacementSelect.style.borderColor = '#007cba';
		popoverPlacementSelect.style.boxShadow = '0 0 0 2px rgba(0, 124, 186, 0.2)';
	});
	popoverPlacementSelect.addEventListener('blur', () => {
		popoverPlacementSelect.style.borderColor = '#ddd';
		popoverPlacementSelect.style.boxShadow = 'none';
	});
	popoverPlacementSelect.addEventListener('change', onChange);
	popoverPlacementGroup.appendChild(popoverPlacementLabel);
	popoverPlacementGroup.appendChild(popoverPlacementSelect);
	
	// Popover trigger
	const popoverTriggerGroup = document.createElement('div');
	const popoverTriggerLabel = document.createElement('label');
	popoverTriggerLabel.textContent = 'Popover Trigger';
	popoverTriggerLabel.style.cssText = 'font-size: 12px; font-weight: 600; margin-bottom: 6px; display: block; color: #495057;';
	const popoverTriggerSelect = document.createElement('select');
	popoverTriggerSelect.className = 'link-list-popover-trigger sidebar-select-dropdown';
	popoverTriggerSelect.style.cssText = 'width: 100%; padding: 6px 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 13px; background: white; color: #000; cursor: pointer; transition: border-color 0.2s ease, box-shadow 0.2s ease;';
	popoverTriggerSelect.innerHTML = `
		<option value="hover focus" ${itemData.popover_trigger === 'hover focus' ? 'selected' : ''}>Hover & Focus</option>
		<option value="hover" ${itemData.popover_trigger === 'hover' ? 'selected' : ''}>Hover</option>
		<option value="focus" ${itemData.popover_trigger === 'focus' ? 'selected' : ''}>Focus</option>
		<option value="click" ${itemData.popover_trigger === 'click' ? 'selected' : ''}>Click</option>
	`;
	popoverTriggerSelect.addEventListener('focus', () => {
		popoverTriggerSelect.style.borderColor = '#007cba';
		popoverTriggerSelect.style.boxShadow = '0 0 0 2px rgba(0, 124, 186, 0.2)';
	});
	popoverTriggerSelect.addEventListener('blur', () => {
		popoverTriggerSelect.style.borderColor = '#ddd';
		popoverTriggerSelect.style.boxShadow = 'none';
	});
	popoverTriggerSelect.addEventListener('change', onChange);
	popoverTriggerGroup.appendChild(popoverTriggerLabel);
	popoverTriggerGroup.appendChild(popoverTriggerSelect);
	
	advancedContainer.appendChild(hoverTextGroup);
	advancedContainer.appendChild(enablePopoverGroup);
	advancedContainer.appendChild(popoverPlacementGroup);
	advancedContainer.appendChild(popoverTriggerGroup);
	
	advancedCell.appendChild(advancedContainer);
	advancedRow.appendChild(advancedCell);
	
	// Toggle advanced options - ensure it starts closed
	expandBtn.textContent = '...';
	expandBtn.style.transform = 'rotate(0deg)';
	
	expandBtn.addEventListener('click', () => {
		const isVisible = advancedRow.style.display !== 'none' && advancedRow.style.display !== '';
		advancedRow.style.display = isVisible ? 'none' : 'table-row';
		expandBtn.textContent = isVisible ? '...' : 'v';
		expandBtn.style.transform = isVisible ? 'rotate(0deg)' : 'rotate(180deg)';
	});

	row.appendChild(textCell);
	row.appendChild(urlCell);
	row.appendChild(targetCell);
	row.appendChild(actionsCell);
	tbody.appendChild(row);
	tbody.appendChild(advancedRow);
}

export default createLinkListManager;

