import { ImageCropper } from '../../utils/ImageCropper.js';

const createMultiImageSelector = (sidebar, fieldName, originalElement) => {
	const container = document.createElement('div');
	container.className = 'sidebar-multi-image-selector';

	const maxImages = parseInt(originalElement.getAttribute('data-sdc-max-images')) || 20;
	const currentImageUuids = sidebar.getFieldValue(originalElement, 'multi-image-selector') || [];
	const currentCropData = sidebar.getFieldValue(originalElement, 'image-crop') || null;
	
	// Check if the component has an imageCrop field
	const component = originalElement.closest('[data-sdc-component]');
	const hasImageCropField = component ? !!component.querySelector('[data-sdc-field="imageCrop"]') : false;
	
	let cropperInstance = null;
	let selectedImageData = null;

	const previewContainer = document.createElement('div');
	previewContainer.className = 'multi-image-preview-container';

	// Crop button (only shown for single image selectors)
	const cropBtn = document.createElement('button');
	cropBtn.type = 'button';
	cropBtn.className = 'crop-image-btn';
	cropBtn.innerHTML = '<i class="fa-solid fa-crop" aria-hidden="true"></i> Crop Image';
	cropBtn.style.display = 'none';

	// Cropper container (initially hidden)
	const cropperContainer = document.createElement('div');
	cropperContainer.className = 'image-cropper-wrapper';
	cropperContainer.style.display = 'none';

	const renderPreviews = (imageUuids) => {
		previewContainer.innerHTML = '';

		if (imageUuids && imageUuids.length > 0) {
			imageUuids.forEach((uuid, index) => {
				const previewItem = document.createElement('div');
				previewItem.className = 'multi-image-preview-item';
				previewItem.setAttribute('data-uuid', uuid);
				previewItem.setAttribute('data-index', index);

				const thumbnail = document.createElement('div');
				thumbnail.className = 'image-thumbnail';
				thumbnail.innerHTML = `<div class="image-placeholder-small">Image ${index + 1}</div>`;

				const removeBtn = document.createElement('button');
				removeBtn.type = 'button';
				removeBtn.className = 'remove-image-btn';
				removeBtn.innerHTML = '&times;';
				removeBtn.title = 'Remove image';

				removeBtn.addEventListener('click', (event) => {
					event.stopPropagation();
					const updatedUuids = imageUuids.filter((_, i) => i !== index);
					sidebar.updateFieldValue(originalElement, JSON.stringify(updatedUuids), 'multi-image-selector');
					renderPreviews(updatedUuids);
				});

				previewItem.appendChild(thumbnail);
				previewItem.appendChild(removeBtn);
				previewContainer.appendChild(previewItem);
			});

			// Show crop button only for single-image selectors that have an imageCrop field
			if (maxImages === 1 && imageUuids.length === 1 && hasImageCropField) {
				cropBtn.style.display = 'block';
			} else {
				cropBtn.style.display = 'none';
			}
		} else {
			previewContainer.innerHTML = '<div class="no-images">No images selected</div>';
			cropBtn.style.display = 'none';
		}
	};

	renderPreviews(currentImageUuids ?? []);

	const addBtn = document.createElement('button');
	addBtn.type = 'button';
	addBtn.className = 'add-images-btn';
	addBtn.textContent = '+ Add Images';

	addBtn.addEventListener('click', () => {
		sidebar.openMediaLibrary(
			(selectedImages) => {
				const existing = sidebar.getFieldValue(originalElement, 'multi-image-selector') || [];
				const newUuids = selectedImages.map((img) => img.uuid);
				const updatedUuids = [...existing, ...newUuids];

				// Store first selected image for cropping
				if (selectedImages.length > 0) {
					selectedImageData = selectedImages[0];
				}

				sidebar.updateFieldValue(originalElement, JSON.stringify(updatedUuids), 'multi-image-selector');
				renderPreviews(updatedUuids);
			},
			true,
			maxImages,
		);
	});

	// Crop button handler
	cropBtn.addEventListener('click', () => {
		const currentUuids = sidebar.getFieldValue(originalElement, 'multi-image-selector') || [];
		if (!currentUuids.length) {
			return;
		}

		// Find the imageCrop field element in the component
		const component = originalElement.closest('[data-sdc-component]');
		const imageCropElement = component ? component.querySelector('[data-sdc-field="imageCrop"]') : null;
		
		if (!imageCropElement) {
			console.error('Could not find imageCrop field element');
			return;
		}

		// Get existing crop data
		const currentCropData = sidebar.getFieldValue(imageCropElement, 'image-crop') || {};

		// Get the image URL from the rendered component
		let imageUrl = '';
		if (component) {
			const img = component.querySelector('img');
			if (img && img.src) {
				imageUrl = img.src;
			}
		}

		// Fallback to selectedImageData if available
		if (!imageUrl && selectedImageData) {
			imageUrl = selectedImageData.url || selectedImageData.thumbnail || '';
		}

		// If still no image URL, show an error
		if (!imageUrl) {
			console.error('No image URL found for cropping');
			alert('Please save the component first, then try cropping the image.');
			return;
		}

		// Show cropper interface
		cropperContainer.style.display = 'block';
		buttonContainer.style.display = 'none';
		previewContainer.style.display = 'none';

		if (!cropperInstance) {
			// Parse existing crop data if available
			let initialCrop = { x: 50, y: 50, scale: 1 };
			if (currentCropData && typeof currentCropData === 'object') {
				initialCrop = { ...initialCrop, ...currentCropData };
			}

			cropperInstance = new ImageCropper({
				imageUrl: imageUrl,
			aspectRatio: null,
			initialCrop: initialCrop,
			onCropChange: (cropData) => {
				// Store crop data in the hidden image-crop field
				sidebar.updateFieldValue(imageCropElement, JSON.stringify(cropData), 'image-crop');
			},
		});			const cropperElement = cropperInstance.createCropperElement();
			cropperContainer.appendChild(cropperElement);

			// Done button
			const doneBtn = document.createElement('button');
			doneBtn.type = 'button';
			doneBtn.className = 'crop-done-btn';
			doneBtn.textContent = 'Done';

			doneBtn.addEventListener('click', () => {
				cropperContainer.style.display = 'none';
				buttonContainer.style.display = 'flex';
				previewContainer.style.display = 'grid';
			});

			cropperContainer.appendChild(doneBtn);
		} else {
			// Update existing cropper with new image if needed
			if (imageUrl && imageUrl !== cropperInstance.imageUrl) {
				cropperInstance.updateImage(imageUrl);
			}
		}
	});

	const clearBtn = document.createElement('button');
	clearBtn.type = 'button';
	clearBtn.className = 'clear-images-btn';
	clearBtn.textContent = 'Clear All';

	clearBtn.addEventListener('click', () => {
		sidebar.updateFieldValue(originalElement, JSON.stringify([]), 'multi-image-selector');
		renderPreviews([]);
	});

	const buttonContainer = document.createElement('div');
	buttonContainer.className = 'multi-image-buttons';
	buttonContainer.appendChild(addBtn);
	buttonContainer.appendChild(clearBtn);
	
	// Only add crop button for single-image selectors that have an imageCrop field
	if (maxImages === 1 && hasImageCropField) {
		buttonContainer.appendChild(cropBtn);
	}

	container.appendChild(previewContainer);
	container.appendChild(buttonContainer);
	container.appendChild(cropperContainer);

	return container;
};

export default createMultiImageSelector;


