const BRAND_ICON_NAMES = [
	'facebook-f',
	'youtube',
	'linkedin-in',
	'instagram',
];

const GENERAL_ICON_NAMES = [
	'caret-down',
	'caret-up',
	'caret-right',
	'paperclip',
	'arrow-left',
	'y-mount',
	'tree-wisdom',
	'arrow-right',
	'chevron-right',
	'search',
	'magnifying-glass',
	'pencil',
	'briefcase',
	'phone',
	'envelope',
	'house',
	'timeline',
	'tree',
	'clipboard',
	'file-text',
	'ufo-beam',
	'pencil-square',
	'graduation-cap',
	'map-marker',
	'print',
	'xmark',
	'clock',
	'location-dot',
	'download',
	'spinner',
	'times',
	'trash',
	'award',
	'x',
	'brain',
	'refresh',
	'face-smile',
	'face-meh',
	'face-frown',
	'headphones',
	'calendar',
	'dollar-sign',
	'address-book',
	'newspaper',
	'books',
	'hand-holding-heart',
	'sapling',
	'users-line',
	'chart-network',
	'book-open-cover',
	'waveform-lines',
	'comments',
	'person-chalkboard',
	'chalkboard-user',
	'users',
	'head-side-headphones',
	'phone-office',
	'file-lines',
	'clipboard-list',
	'book',
	'head-side-brain',
	'calendar-week',
	'hand-holding-dollar',
	'id-badge',
	'chart-line',
	'id-card-clip',
	'user-magnifying-glass',
	'school',
	'memo-circle-check',
	'people-group',
	'handshake-angle',
	'messages',
	'whistle',
	'dumbbell',
	'flag',
	'book-font',
	'magnifying-glass-chart',
	'square-list',
	'paintbrush-fine',
	'share-nodes',
	'screen-users',
	'paintbrush-pencil',
	'scale-balanced',
	'image-user',
	'person-hiking',
	'hands-holding-heart',
	'clock-three',
	'notebook',
	'list-check',
	'hands-holding-child',
	'file-chart-column',
	'diagram-project',
	'file-spreadsheet',
	'user-pen',
	'ballot-check',
	'comment-heart',
	'users-rectangle',
	'book-blank',
	'building-memo',
	'people-roof',
	'clipboard-user',
	'file-circle-check',
	'folder-open',
	'podcast',
	'book-open-reader',
	'browser',
	'clapperboard-play',
	'family',
	'handshake',
	'lightbulb-on',
	'chart-diagram',
	'pencil-mechanical',
	'pen-ruler',
	'object-ungroup',
	'grid-dividers',
	'memo-circle-info',
	'folders',
	'file-magnifying-glass',
	'briefcase-blank',
	'shapes',
	'calendar-check',
	'square-plus',
	'chart-simple',
	'comments-question-check',
	'diploma',
	'palette',
	'pen',
	'fingerprint',
	'apple-whole',
	'user-graduate',
	'handshake-simple',
	'fish-fins',
	'user',
	'screwdriver-wrench',
	'desktop',
	'object-group',
	'chart-mixed',
	'film',
	'cloud-arrow-up',
	'display-arrow-down',
	'laptop',
	'tablet-screen-button',
	'robot',
	'projector',
	'computer-mouse',
	'presentation-screen',
	'info',
	'rabbit',
	'bed',
	'up-right-and-down-left-from-center',
];

const GENERAL_VARIANT_CLASS_MAP = {
	light: ['fa', 'fa-light'],
	solid: ['fa', 'fa-solid'],
};

const DEFAULT_GENERAL_VARIANT = 'light';

const styleToClassMap = {
	brand: ['fa', 'fa-brands'],
};

const toLabel = (name) =>
	name
		.split('-')
		.map((part) => part.charAt(0).toUpperCase() + part.slice(1))
		.join(' ');

const buildClassName = (classes = [], name = '') => {
	const iconClass = `fa-${name}`;
	return `${[...classes, iconClass].join(' ')}`.trim();
};

const buildIconEntry = (name, style) => {
	if (style === 'general') {
		const variants = Object.fromEntries(
			Object.entries(GENERAL_VARIANT_CLASS_MAP).map(([variantKey, classes]) => [
				variantKey,
				buildClassName(classes, name),
			]),
		);

		return {
			name,
			style,
			label: toLabel(name),
			variants,
			defaultVariant: DEFAULT_GENERAL_VARIANT,
			className: variants[DEFAULT_GENERAL_VARIANT],
			searchTerms: `${name} ${toLabel(name)} ${style} ${Object.keys(variants).join(' ')}`.toLowerCase(),
		};
	}

	const baseClasses = styleToClassMap[style] || ['fa'];
	const className = buildClassName(baseClasses, name);

	return {
		name,
		style,
		label: toLabel(name),
		variants: {
			[style]: className,
		},
		defaultVariant: style,
		className,
		searchTerms: `${name} ${toLabel(name)} ${style}`.toLowerCase(),
	};
};

export const GENERAL_STYLE_VARIANTS = Object.keys(GENERAL_VARIANT_CLASS_MAP).map((key) => ({
	key,
	label: toLabel(key),
}));

export const ICONS = [
	...BRAND_ICON_NAMES.map((name) => buildIconEntry(name, 'brand')),
	...GENERAL_ICON_NAMES.map((name) => buildIconEntry(name, 'general')),
];

export const getIconClassName = (icon, requestedVariant) => {
	if (!icon) {
		return '';
	}

	if (icon.style !== 'general') {
		const variantKey = Object.keys(icon.variants || {})[0];
		return (icon.variants && icon.variants[variantKey]) || icon.className;
	}

	const variantKey =
		requestedVariant && GENERAL_VARIANT_CLASS_MAP[requestedVariant]
			? requestedVariant
			: icon.defaultVariant || DEFAULT_GENERAL_VARIANT;

	if (icon.variants && icon.variants[variantKey]) {
		return icon.variants[variantKey];
	}

	return buildClassName(GENERAL_VARIANT_CLASS_MAP[variantKey] || GENERAL_VARIANT_CLASS_MAP[DEFAULT_GENERAL_VARIANT], icon.name);
};

export const getIconByClassName = (className = '') => {
	const normalized = className.trim().replace(/\s+/g, ' ');

	if (!normalized) {
		return null;
	}

	for (const icon of ICONS) {
		const variants = icon.variants || {};

		for (const [variantKey, variantClass] of Object.entries(variants)) {
			if (variantClass === normalized) {
				return {
					...icon,
					className: variantClass,
					matchedVariant: icon.style === 'general' ? variantKey : null,
				};
			}
		}
	}

	return null;
};


