/**
 * @file
 * ImageCropper utility class
 *
 * Provides image cropping functionality with drag-to-pan capability
 * and real-time preview for components.
 */

export class ImageCropper {
	constructor(options = {}) {
		this.imageUrl = options.imageUrl || null;
		this.aspectRatio = options.aspectRatio || null; // e.g., 16/9, 3/2, null for free
		this.initialCrop = options.initialCrop || { x: 50, y: 50 }; // x, y as percentage
		this.onCropChange = options.onCropChange || (() => {});
		this.containerElement = null;
		this.currentCrop = { ...this.initialCrop };
	}

	/**
	 * Create the cropper interface
	 * @returns {HTMLElement} The cropper container element
	 */
	createCropperElement() {
		const container = document.createElement('div');
		container.className = 'image-cropper-container';

		// Instructions
		const instructions = document.createElement('div');
		instructions.className = 'image-cropper-instructions';
		instructions.innerHTML = `
			<i class="fa-solid fa-arrows-alt" aria-hidden="true"></i>
			<span>Use arrow controls to adjust image position</span>
		`;

		// Controls container
		const controls = document.createElement('div');
		controls.className = 'image-cropper-controls';

		// Position controls
		const positionControl = this.createPositionControl();
		controls.appendChild(positionControl);

		// Reset button
		const resetBtn = document.createElement('button');
		resetBtn.type = 'button';
		resetBtn.className = 'image-cropper-reset-btn';
		resetBtn.innerHTML = '<i class="fa-solid fa-rotate-left" aria-hidden="true"></i> Reset';
		resetBtn.addEventListener('click', () => this.resetCrop());
		controls.appendChild(resetBtn);

		container.appendChild(instructions);
		container.appendChild(controls);

		this.containerElement = container;

		return container;
	}



	/**
	 * Create position fine-tune controls
	 */
	createPositionControl() {
		const positionWrapper = document.createElement('div');
		positionWrapper.className = 'image-cropper-position-control';

		const label = document.createElement('label');
		label.textContent = 'Position';
		label.className = 'image-cropper-control-label';

		const buttonsWrapper = document.createElement('div');
		buttonsWrapper.className = 'image-cropper-position-buttons';

		// Create buttons in grid layout (plus sign)
		// Row 1: Up arrow (centered)
		// Row 2: Left, Center, Right
		// Row 3: Down arrow (centered)
		const directions = [
			{ dir: 'up', icon: 'fa-arrow-up', dx: 0, dy: 2, gridArea: 'up' },
			{ dir: 'left', icon: 'fa-arrow-left', dx: 2, dy: 0, gridArea: 'left' },
			{ dir: 'center', icon: 'fa-bullseye', dx: 0, dy: 0, center: true, gridArea: 'center' },
			{ dir: 'right', icon: 'fa-arrow-right', dx: -2, dy: 0, gridArea: 'right' },
			{ dir: 'down', icon: 'fa-arrow-down', dx: 0, dy: -2, gridArea: 'down' },
		];

		directions.forEach(({ dir, icon, dx, dy, center, gridArea }) => {
			const btn = document.createElement('button');
			btn.type = 'button';
			btn.className = `image-cropper-position-btn ${center ? 'center' : ''}`;
			btn.style.gridArea = gridArea;
			btn.innerHTML = `<i class="fa-solid ${icon}" aria-hidden="true"></i>`;
			btn.setAttribute('aria-label', center ? 'Center image' : `Move ${dir}`);

			btn.addEventListener('click', () => {
				if (center) {
					this.currentCrop.x = 50;
					this.currentCrop.y = 50;
				} else {
					this.currentCrop.x = Math.max(0, Math.min(100, this.currentCrop.x + dx));
					this.currentCrop.y = Math.max(0, Math.min(100, this.currentCrop.y + dy));
				}
				this.onCropChange(this.currentCrop);
			});

			buttonsWrapper.appendChild(btn);
		});

		positionWrapper.appendChild(label);
		positionWrapper.appendChild(buttonsWrapper);

		return positionWrapper;
	}





	/**
	 * Reset crop to initial state
	 */
	resetCrop() {
		this.currentCrop = { x: 50, y: 50 };
		this.onCropChange(this.currentCrop);
	}

	/**
	 * Update the image URL
	 */
	updateImage(newImageUrl) {
		this.imageUrl = newImageUrl;
	}

	/**
	 * Update the crop data
	 */
	updateCrop(cropData) {
		this.currentCrop = { ...cropData };
	}

	/**
	 * Get current crop data
	 */
	getCropData() {
		return { ...this.currentCrop };
	}

	/**
	 * Destroy the cropper and clean up
	 */
	destroy() {
		if (this.containerElement) {
			this.containerElement.remove();
		}
		this.containerElement = null;
	}
}

export default ImageCropper;
