<?php

namespace Drupal\sdc_inline_editor\Component;
use Drupal\Core\Plugin\Component;
/**
 * Encapsulates component information for SDC components.
 * Mostly used for Backend communication
 */
class ComponentInfo
{
    public function __construct(
        public readonly string $name,
        public readonly ?string $friendlyName = null,
        public readonly ?string $categoryName = null,
        public readonly ?string $library = null,
        public readonly ?string $thumbnail = null,
        public readonly array $metadata = []
    ) {
    }

    /**
     * Get the component name in the format expected by Drupal's component system.
     */
    public function getComponentName(): string
    {
        return $this->name;
    }

    /**
     * Get the library name if available.
     */
    public function getLibraryName(): ?string
    {
        return $this->library;
    }

    /**
     * Get the thumbnail path if available.
     */
    public function getThumbnailPath(): ?string
    {
        return $this->thumbnail;
    }

    /**
     * Get additional metadata.
     */
    public function getMetadata(): array
    {
        return $this->metadata;
    }

    /**
     * Get the category name if available.
     */
    public function getCategoryName(): ?string
    {
        return $this->categoryName;
    }

    /**
     * Create ComponentInfo from a component plugin instance.
     */
    public static function fromComponent(Component $component): self
    {
        $library = $component->getLibraryName();
        $name = self::extractComponentName($library);
        $friendlyName = $component->metadata->name;
        $categoryName = null;

        $plugin_def = $component->getPluginDefinition();

        // The plugin definition has a 'path' key that points to the component directory
        if (!empty($plugin_def['path'])) {
            $component_path = $plugin_def['path'];
            $component_id = $plugin_def['id'] ?? '';

            // Extract the component folder name from the ID (format: module:component)
            if (strpos($component_id, ':') !== FALSE) {
                $parts = explode(':', $component_id);
                $component_folder = end($parts);
                $yaml_file = $component_path . '/' . $component_folder . '.component.yml';

                if (file_exists($yaml_file)) {
                    try {
                        $yaml_content = file_get_contents($yaml_file);
                        // Simple regex to extract category value
                        if (preg_match('/^category:\s*(.+)$/m', $yaml_content, $matches)) {
                            $categoryName = trim($matches[1]);
                        }
                    } catch (\Exception $e) {
                        // Silently fail
                    }
                }
            }
        }

        // Get the proper file URL for the thumbnail
        $thumbnail_path = $component->metadata->getThumbnailPath();
        $thumbnail = null;
        if ($thumbnail_path) {
            // Use Drupal's file URL generator to get the proper URL
            $thumbnail = \Drupal::service('file_url_generator')->generateAbsoluteString($thumbnail_path);
        }

        return new self(
            name: $name,
            friendlyName: $friendlyName,
            categoryName: $categoryName,
            library: $library,
            thumbnail: $thumbnail,
            metadata: []
        );
    }

    /**
     * Extract component name from library name.
     */
    private static function extractComponentName(string $component_library): string
    {
        $component_name = str_replace('core/components.', '', $component_library);
        $component_name = str_replace('--', ':', $component_name);
        return $component_name;
    }
}
