<?php

namespace Drupal\sdc_inline_editor\Component;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\Markup;
use Drupal\Component\Utility\Xss;

class SDCComponent extends BaseComponent
{
    public ComponentInfo $componentInfo;
    public array $props;
    // Slots are currently not used, as these are mostly for Drupal Experience Builder. We won't be making use of them.
    public array $slots;
    
    public function __construct(
        ComponentInfo $componentInfo,
        array $props,
    ) {
        $this->componentInfo = $componentInfo;
        $this->props = $props;
        $this->slots = [];
    }

    public function buildRenderArray(): array
    {
        $component_name = $this->componentInfo->getComponentName();
        
        // Validate that component name is not empty
        if (empty(trim($component_name))) {
            \Drupal::logger('sdc_inline_editor')->error('Attempted to build render array for component with empty name');
            return [
                '#markup' => '<!-- Invalid component: empty name -->',
            ];
        }
        
        // Add component_name to props so it's available in the Twig template
        $props = $this->props;
        $props['component_name'] = $component_name;
        
        // For other components, use standard prop handling
        if (isset($props['html']) && is_string($props['html'])) {
            $props['html'] = Markup::create($props['html']);
        }
        
        if (isset($props['text']) && is_string($props['text'])) {
            $decoded_text = html_entity_decode($props['text'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $props['text'] = Markup::create($decoded_text);
        }
        
        return [
            '#type' => 'component',
            '#component' => $component_name,
            '#props' => $props,
        ];
    }

    /**
     * Get the component name for backward compatibility.
     */
    public function getName(): string
    {
        return $this->componentInfo->getComponentName();
    }

    /**
     * Get the component info object.
     */
    public function getComponentInfo(): ComponentInfo
    {
        return $this->componentInfo;
    }
}