<?php

namespace Drupal\sdc_inline_editor\Controller;

use Drupal\media_library\MediaLibraryState;
use Drupal\media_library\MediaLibraryUiBuilder;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\media\MediaTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

class AjaxController extends ControllerBase {

    /**
     * The request stack.
     *
     * @var \Symfony\Component\HttpFoundation\RequestStack
     */
    protected $requestStack;
    /**
     * The medial library builder.
     *
     * @var \Drupal\media_library\MediaLibraryUiBuilder|null
     */
    private ?MediaLibraryUiBuilder $medialLibraryBuilder;

    final public function __construct(MediaLibraryUiBuilder $medialLibraryBuilder, RequestStack $requestStack) {
        $this->requestStack = $requestStack;
        $this->medialLibraryBuilder = $medialLibraryBuilder;
    }

    public static function create(ContainerInterface $container) {

        $medialLibraryBuilder = $container->get('media_library.ui_builder');
        $requestStack = $container->get('request_stack');
        return new static(
            $medialLibraryBuilder,
            $requestStack
        );
    }

    public function openImageMediaLibrary() {
        $sdc_inline_editor_settings = $this->config('sdc_inline_editor.settings');
        $cardinality = $this->requestStack->getCurrentRequest()->query->get('remaining_slots');
        $uuid = $this->requestStack->getCurrentRequest()->query->get('uuid');
        $content_type = $this->requestStack->getCurrentRequest()->query->get('content_type');
        
        // Get allowed media types based on content type
        $allowed_media_types = $this->getAllowedMediaTypes($content_type);
        
        // Use the first media type as the selected type
        $selected_media_type = !empty($allowed_media_types) ? $allowed_media_types[0] : 'sdc-inline-editor-image';

        $state = MediaLibraryState::create(
            opener_id: 'media_library.opener.sdc_inline_editor',
            allowed_media_type_ids: $allowed_media_types,
            selected_type_id: $selected_media_type,
            remaining_slots: $cardinality ?: 1,
            opener_parameters: $uuid ? ['uuid' => $uuid] : []
        );

        $response = $this->medialLibraryBuilder->buildUi($state);
        
        // Ensure the sdc_inline_editor_media CSS library is attached to the response
        if (isset($response['#attached'])) {
            $response['#attached']['library'][] = 'sdc_inline_editor_media/view';
        } else {
            $response['#attached'] = [
                'library' => ['sdc_inline_editor_media/view']
            ];
        }
        
        return $response;
    }

    /**
     * Get allowed media types based on content type.
     *
     * @param string|null $content_type
     *   The content type machine name.
     *
     * @return array
     *   Array of allowed media type IDs.
     */
    protected function getAllowedMediaTypes($content_type) {
        // Default media type
        $default_media_types = ['sdc-inline-editor-image'];
        
        if (!$content_type) {
            return $default_media_types;
        }

        // Get the configuration
        $config = $this->config('sdc_inline_editor.settings');
        $content_type_settings = $config->get('content_type_settings') ?: [];
        
        // Check if this content type has custom media type configuration
        if (isset($content_type_settings[$content_type]['allowed_media_types'])) {
            $allowed_media_types = $content_type_settings[$content_type]['allowed_media_types'];
            
            // If media types are configured, use them; otherwise use default
            if (!empty($allowed_media_types)) {
                return array_values($allowed_media_types);
            }
        }

        // Return default media types if no configuration exists
        return $default_media_types;
    }
}