<?php

namespace Drupal\sdc_inline_editor\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\sdc_inline_editor\Service\PageEditorService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for handling page editor operations.
 */
class EditorController extends ControllerBase {

  /**
   * The page editor service.
   *
   * @var \Drupal\sdc_inline_editor\Service\PageEditorService
   */
  protected $pageEditorService;

  /**
   * Constructs an EditorController object.
   *
   * @param \Drupal\sdc_inline_editor\Service\PageEditorService $page_editor_service
   *   The page editor service.
   */
  public function __construct(PageEditorService $page_editor_service) {
    $this->pageEditorService = $page_editor_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sdc_inline_editor.page_editor_service')
    );
  }

  /**
   * Get HTML content from a node.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response with the content.
   */
  public function getContent(Request $request): JsonResponse {
    // Check if user has permission
    if (!$this->currentUser()->hasPermission('sdc_inline_editor.edit_content')) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Access denied. You do not have permission to edit content.',
      ], Response::HTTP_FORBIDDEN);
    }

    // Get node_id from query parameters
    $node_id = $request->query->get('node_id');

    // Validate required parameters
    if (!$node_id) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Missing required parameter: node_id is required.',
      ], Response::HTTP_BAD_REQUEST);
    }

    $node_id = (int) $node_id;

    // Validate node_id is a positive integer
    if ($node_id <= 0) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Invalid node_id. Must be a positive integer.',
      ], Response::HTTP_BAD_REQUEST);
    }

    // Get the content using the page editor service
    $result = $this->pageEditorService->getNodeContent($node_id);

    // Return appropriate HTTP status code based on result
    $status_code = $result['success'] ? Response::HTTP_OK : Response::HTTP_BAD_REQUEST;

    return new JsonResponse($result, $status_code);
  }

  /**
   * Save component-based content to a node.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response with the result.
   */
  public function saveComponents(Request $request): JsonResponse {
    // Check if user has permission
    if (!$this->currentUser()->hasPermission('sdc_inline_editor.edit_content')) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Access denied. You do not have permission to edit content.',
      ], Response::HTTP_FORBIDDEN);
    }

    // Get request data
    $raw_content = $request->getContent();
    \Drupal::logger('sdc_inline_editor')->debug('EditorController::saveComponents: Raw request content: @content', [
      '@content' => substr($raw_content, 0, 2000) . (strlen($raw_content) > 2000 ? '...' : ''),
    ]);
    
    $data = json_decode($raw_content, TRUE);
    
    // Log decoded data
    \Drupal::logger('sdc_inline_editor')->debug('EditorController::saveComponents: Decoded JSON data: @data', [
      '@data' => print_r($data, TRUE),
    ]);

    // Validate required parameters
    if (!isset($data['node_id']) || !isset($data['components'])) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Missing required parameters: node_id and components are required.',
      ], Response::HTTP_BAD_REQUEST);
    }

    $node_id = (int) $data['node_id'];
    $components = $data['components'];
    
    // Log components structure
    \Drupal::logger('sdc_inline_editor')->debug('EditorController::saveComponents: Components array structure: @components', [
      '@components' => print_r($components, TRUE),
    ]);

    // Validate node_id is a positive integer
    if ($node_id <= 0) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Invalid node_id. Must be a positive integer.',
      ], Response::HTTP_BAD_REQUEST);
    }

    // Validate components is an array
    if (!is_array($components)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Invalid components. Must be an array.',
      ], Response::HTTP_BAD_REQUEST);
    }

    // Save the components using the page editor service
    $result = $this->pageEditorService->saveComponentsToNode($node_id, $components);

    // Return appropriate HTTP status code based on result
    $status_code = $result['success'] ? Response::HTTP_OK : Response::HTTP_BAD_REQUEST;

    return new JsonResponse($result, $status_code);
  }

}
