<?php

namespace Drupal\sdc_inline_editor\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\sdc_inline_editor\Service\SDCService;
use Drupal\sdc_inline_editor\Component\ComponentInfo;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for SDC (Single Directory Component) operations.
 */
class SDCController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * The SDC service.
   *
   * @var \Drupal\sdc_inline_editor\Service\SDCService
   */
  protected $sdcService;

  /**
   * Constructs a new SDCController object.
   *
   * @param \Drupal\sdc_inline_editor\Service\SDCService $sdc_service
   *   The SDC service.
   */
  public function __construct(SDCService $sdc_service) {
    $this->sdcService = $sdc_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sdc_inline_editor.sdc_service')
    );
  }

  /**
   * Get the HTML for a specific component.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the component HTML.
   */
  public function getComponentHtml(Request $request): JsonResponse {
    $component_name = $request->query->get('component');
    $props = [];

    // Support JSON payloads for POST requests.
    if ($request->isMethod('POST')) {
      $payload = json_decode($request->getContent(), TRUE) ?? [];
      if (isset($payload['component'])) {
        $component_name = $payload['component'];
      }
      if (isset($payload['props'])) {
        $props = is_array($payload['props']) ? $payload['props'] : [];
      }
    }

    // Fallback to query string props if provided (legacy support).
    if (empty($props) && $request->query->has('props')) {
      $component_props = $request->query->get('props');
      $decoded_props = json_decode($component_props, TRUE);
      $props = is_array($decoded_props) ? $decoded_props : [];
    }
    
    if (empty($component_name)) {
      return new JsonResponse([
        'error' => 'Component name is required',
        'html' => ''
      ], 400);
    }

    try {
      // Get all accessible components
      $accessible_components = $this->sdcService->getAccessibleComponents();
      
      // Find the requested component
      $component = null;
      foreach ($accessible_components as $comp) {
        if ($comp->getComponentName() === $component_name) {
          $component = $comp;
          break;
        }
      }

      if (!$component) {
        return new JsonResponse([
          'error' => 'Component not found or not accessible',
          'html' => ''
        ], 404);
      }

      // Get the rendered HTML for the component
      $html = $this->sdcService->getRenderedHTMLForComponent($component, $props);

      return new JsonResponse([
        'success' => TRUE,
        'component' => $component_name,
        'html' => $html,
        'metadata' => [
          'friendly_name' => $component->friendlyName,
          'friendly_category_name' => $component->categoryName,
          'library' => $component->library,
          'thumbnail' => $component->thumbnail,
        ]
      ]);

    } catch (\Exception $e) {
      return new JsonResponse([
        'error' => 'Failed to render component: ' . $e->getMessage(),
        'html' => ''
      ], 500);
    }
  }

  /**
   * Get all accessible components with their metadata.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the list of accessible components.
   */
  public function getAccessibleComponents(Request $request): JsonResponse {
    $content_type = $request->query->get('content_type');
    
    try {
      $components = $this->sdcService->getAccessibleComponents($content_type);
      
      $component_data = [];
      foreach ($components as $component) {
        $component_data[] = [
          'name' => $component->getComponentName(),
          'friendly_name' => $component->friendlyName,
          'friendly_category_name' => $component->categoryName,
          'library' => $component->library,
          'thumbnail' => $component->thumbnail,
          'metadata' => $component->metadata,
        ];
      }

      return new JsonResponse([
        'success' => TRUE,
        'components' => $component_data,
        'count' => count($component_data),
        'content_type' => $content_type,
      ]);

    } catch (\Exception $e) {
      return new JsonResponse([
        'error' => 'Failed to get accessible components: ' . $e->getMessage(),
        'components' => []
      ], 500);
    }
  }

  /**
   * Check if a specific component is accessible.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response indicating if the component is accessible.
   */
  public function checkComponentAccess(Request $request): JsonResponse {
    $component_name = $request->query->get('component');
    $content_type = $request->query->get('content_type');
    
    if (empty($component_name)) {
      return new JsonResponse([
        'error' => 'Component name is required',
        'accessible' => FALSE
      ], 400);
    }

    try {
      $is_accessible = $this->sdcService->isComponentAccessible($component_name, $content_type);
      $config_source = $this->sdcService->getConfigurationSource($content_type);

      return new JsonResponse([
        'success' => TRUE,
        'component' => $component_name,
        'accessible' => $is_accessible,
        'content_type' => $content_type,
        'config_source' => $config_source,
      ]);

    } catch (\Exception $e) {
      return new JsonResponse([
        'error' => 'Failed to check component access: ' . $e->getMessage(),
        'accessible' => FALSE
      ], 500);
    }
  }

}
