<?php

namespace Drupal\sdc_inline_editor\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\sdc_inline_editor\Service\SDCService;
use Drupal\node\Entity\NodeType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for SDC Inline Editor component settings.
 */
class SdcInlineEditorConfigForm extends ConfigFormBase {

  /**
   * The SDC service.
   *
   * @var \Drupal\sdc_inline_editor\Service\SDCService
   */
  protected $sdcService;

  /**
   * Constructs a SdcInlineEditorConfigForm object.
   *
   * @param \Drupal\sdc_inline_editor\Service\SDCService $sdc_service
   *   The SDC service.
   */
  public function __construct(SDCService $sdc_service) {
    $this->sdcService = $sdc_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('sdc_inline_editor.sdc_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['sdc_inline_editor.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'sdc_inline_editor_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('sdc_inline_editor.settings');
    
    // Get all available SDC components
    $available_components_info = $this->sdcService->getSDCComponents();
    $available_components = array_map(fn($component) => $component->getComponentName(), $available_components_info);
    $accessible_components = $config->get('accessible_components') ?: [];
    $enable_all = $config->get('enable_all_components') ?: FALSE;
    
    // Get existing content type settings
    $content_type_settings = $config->get('content_type_settings') ?: [];

    // Global settings
    $form['global_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Global Settings'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $form['global_settings']['enable_all_components'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable All Components Globally'),
      '#description' => $this->t('When enabled, all SDC components will be accessible from the front-end builder by default.'),
      '#default_value' => $enable_all,
    ];

    $form['global_settings']['accessible_components'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Default Accessible Components'),
      '#description' => $this->t('Select which components should be accessible by default (can be overridden per content type). When "Enable All Components Globally" is checked, this setting is ignored.'),
      '#options' => array_combine($available_components, $available_components),
      '#default_value' => $accessible_components,
      '#states' => [
        'invisible' => [
          ':input[name="global_settings[enable_all_components]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Content type configurations - Accordion style
    $form['content_type_configs'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Type Specific Configurations'),
      '#description' => $this->t('Configure SDC component access for specific content types. These settings override global settings.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    // Get all available content types
    $content_types = NodeType::loadMultiple();
    
    foreach ($content_types as $type_id => $type) {
      $this->buildContentTypeAccordion($form, $type_id, $type->label(), $available_components, $content_type_settings);
    }

    // Module info
    $form['info_link'] = [
      '#type' => 'item',
      '#markup' => $this->t('<p><strong>Module Information:</strong> <a href="/admin/modules" target="_blank">View module details</a> | <a href="/admin/reports/status" target="_blank">System status</a></p>'),
      '#weight' => 100,
    ];

    // Submit button
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Configuration'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Build accordion-style content type configuration.
   */
  protected function buildContentTypeAccordion(array &$form, $type_id, $type_label, $available_components, $content_type_settings) {
    // Get saved settings for this content type
    $current_settings = $content_type_settings[$type_id] ?? [];
    
    // Check if this content type has any configuration
    $has_config = !empty($current_settings);
    
    $form['content_type_configs'][$type_id] = [
      '#type' => 'details',
      '#title' => $this->t('@type (@id)', ['@type' => $type_label, '@id' => $type_id]),
      '#open' => $has_config, // Open if configuration exists
      '#description' => $this->t('Configure SDC component access for the @type content type.', ['@type' => $type_label]),
    ];

    // Enable configuration checkbox
    $form['content_type_configs'][$type_id]['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable custom configuration for @type', ['@type' => $type_label]),
      '#description' => $this->t('When checked, this content type will use the settings below instead of global settings.'),
      '#default_value' => $has_config,
      '#ajax' => [
        'callback' => [$this, 'ajaxToggleContentTypeConfig'],
        'wrapper' => 'content-type-config-' . $type_id,
        'event' => 'change',
      ],
    ];

    // Container for conditional elements
    $form['content_type_configs'][$type_id]['config_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'content-type-config-' . $type_id],
      '#states' => [
        'visible' => [
          ':input[name="content_type_configs[' . $type_id . '][enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Enable all components checkbox
    $form['content_type_configs'][$type_id]['config_container']['enable_all_components'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable All Components'),
      '#description' => $this->t('When enabled, all SDC components will be accessible for this content type.'),
      '#default_value' => $current_settings['enable_all_components'] ?? FALSE,
    ];

    // Component selection
    $form['content_type_configs'][$type_id]['config_container']['accessible_components'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Accessible Components'),
      '#description' => $this->t('Select which components should be accessible for this content type.'),
      '#options' => array_combine($available_components, $available_components),
      '#default_value' => $current_settings['accessible_components'] ?? [],
      '#states' => [
        'invisible' => [
          ':input[name="content_type_configs[' . $type_id . '][config_container][enable_all_components]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Media type selection
    $media_types = $this->getMediaTypeOptions();
    $form['content_type_configs'][$type_id]['config_container']['allowed_media_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Media Types'),
      '#description' => $this->t('Select which media types should be available in the media library for this content type. If none are selected, the default "SDC Inline Editor Image" will be used.'),
      '#options' => $media_types,
      '#default_value' => $current_settings['allowed_media_types'] ?? [],
    ];
  }

  /**
   * Get all available media type options.
   *
   * @return array
   *   An associative array of media type machine names and labels.
   */
  protected function getMediaTypeOptions() {
    $media_types = \Drupal::entityTypeManager()
      ->getStorage('media_type')
      ->loadMultiple();
    
    $options = [];
    foreach ($media_types as $media_type) {
      $options[$media_type->id()] = $media_type->label();
    }
    
    // Sort alphabetically by label
    asort($options);
    
    return $options;
  }

  /**
   * AJAX callback for toggling content type configuration visibility.
   */
  public function ajaxToggleContentTypeConfig(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $type_id = $triggering_element['#parents'][1]; // Get the content type ID from the form structure
    
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand(
      '#content-type-config-' . $type_id,
      $form['content_type_configs'][$type_id]['config_container']
    ));
    
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    
    // No validation needed for accordion approach since each content type
    // has its own section and there are no duplicates possible
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    
    $config = $this->config('sdc_inline_editor.settings');
    
    // Save global settings
    $global_settings = $form_state->getValue('global_settings', []);
    
    // Filter out unchecked checkboxes (FALSE values) from accessible_components
    $accessible_components = $global_settings['accessible_components'] ?? [];
    $filtered_accessible_components = array_filter($accessible_components, function($value) {
      return $value !== FALSE && $value !== 0;
    });
    
    $config->set('enable_all_components', $global_settings['enable_all_components'] ?? FALSE)
           ->set('accessible_components', $filtered_accessible_components);

    // Save content type specific settings
    $content_type_settings = [];
    $content_type_configs = $form_state->getValue('content_type_configs', []);

    foreach ($content_type_configs as $type_id => $type_config) {
      // Only save if the configuration is enabled
      if (!empty($type_config['enabled'])) {
        // Get the config container data
        $config_container = $type_config['config_container'] ?? [];
        
        // Filter out unchecked checkboxes (FALSE values) from accessible_components
        $type_accessible_components = $config_container['accessible_components'] ?? [];
        $filtered_type_accessible_components = array_filter($type_accessible_components, function($value) {
          return $value !== FALSE && $value !== 0;
        });
        
        // Filter out unchecked checkboxes (FALSE values) from allowed_media_types
        $type_allowed_media_types = $config_container['allowed_media_types'] ?? [];
        $filtered_type_allowed_media_types = array_filter($type_allowed_media_types, function($value) {
          return $value !== FALSE && $value !== 0;
        });
        
        $content_type_settings[$type_id] = [
          'enable_all_components' => $config_container['enable_all_components'] ?? FALSE,
          'accessible_components' => $filtered_type_accessible_components,
          'allowed_media_types' => array_values($filtered_type_allowed_media_types),
        ];
      }
    }
    
    $config->set('content_type_settings', $content_type_settings);
    $config->save();

    // Invalidate caches that depend on this configuration
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['config:sdc_inline_editor.settings']);

    // Add success message
    $this->messenger()->addStatus($this->t('SDC Inline Editor configuration has been saved.'));
  }

}

