<?php
namespace Drupal\sdc_inline_editor\Service;

use DOMElement;
use Drupal\Component\Utility\Html;
use Drupal\sdc_inline_editor\Component\SDCComponent;
use Drupal\sdc_inline_editor\Component\RawComponent;
use Drupal\sdc_inline_editor\Component\ComponentInfo;


class ContentValidatorService {

    public function __construct(
        private SDCValidatorService $sdcValidatorService,
    ) {}

    /**
     * Find all elements inside <body> that are direct children,
     * validate sdc- components and return component objects.
     *
     * @param string $html
     * @return \Drupal\sdc_inline_editor\Component\SDCComponent[]|\Drupal\sdc_inline_editor\Component\RawComponent[]
     */
    public function validateBodyContent(string $html): array
    {
        $components = [];
        
        try {
            $domDocument = Html::load($html);
        } catch (\Throwable $e) {
            $domDocument = NULL;
        }

        if (!$domDocument instanceof \DOMDocument) {
            $htmlString = trim($html);
            if ($htmlString === '') {
                return [];
            }

            $componentInfo = new ComponentInfo('sdc_inline_editor:custom-html-block');

            $components[] = $this->sdcValidatorService->validateSDCComponent(
                $componentInfo,
                ["html" => $htmlString]
            );

            return $components;
        }

        $elementStorage = new \SplObjectStorage();

        // Get head element and find direct children: script and style
        $headElements = $domDocument->getElementsByTagName('head');
        if ($headElements->length > 0) {
            $head = $headElements->item(0);
            foreach ($head->childNodes as $child) {
                if ($child instanceof DOMElement && ($child->tagName === 'script' || $child->tagName === 'style')) {
                    $elementStorage->attach($child);
                }
            }
        }

        // Get body element and find all direct children
        $bodyElements = $domDocument->getElementsByTagName('body');
        if ($bodyElements->length > 0) {
            $body = $bodyElements->item(0);
            foreach ($body->childNodes as $child) {
                if ($child instanceof DOMElement) {
                    $elementStorage->attach($child);
                }
            }
        }

        // If no elements found, try html > *
        if ($elementStorage->count() === 0) {
            $htmlElements = $domDocument->getElementsByTagName('html');
            if ($htmlElements->length > 0) {
                $htmlElement = $htmlElements->item(0);
                foreach ($htmlElement->childNodes as $child) {
                    if ($child instanceof DOMElement) {
                        $elementStorage->attach($child);
                    }
                }
            }
        }

        // If still no elements found, get all elements
        if ($elementStorage->count() === 0) {
            $allElements = $domDocument->getElementsByTagName('*');
            foreach ($allElements as $element) {
                if ($element instanceof DOMElement) {
                    $elementStorage->attach($element);
                }
            }
        }

        if ($elementStorage->count() === 0) {
            $htmlString = trim($html);
            if ($htmlString === '') {
                return [];
            }

            $componentInfo = new ComponentInfo('sdc_inline_editor:custom-html-block');

            $components[] = $this->sdcValidatorService->validateSDCComponent(
                $componentInfo,
                ["html" => $htmlString]
            );

            return $components;
        }

        foreach ($elementStorage as $element) {
        // DomCrawler returns DOMElement instances
            if ($element instanceof DOMElement) {
                if (strpos($element->tagName, 'sdc-') === 0) {
                    // Use your private helper methods to get props and element name
                    $elementProps = $this->getSDCPropsFromElement($element);
                    $elementName = $this->getElementNameFromElement($element);
                    
                    $componentInfo = new ComponentInfo($elementName);

                    $components[] = $this->sdcValidatorService->validateSDCComponent(
                        $componentInfo,
                        $elementProps ?? []
                    );
                }
                else {
                    // Use RawComponent for non-sdc elements
                    $htmlString = $element->ownerDocument->saveHTML($element);                
                    
                    $elementProps = [
                        "html" => $htmlString
                    ];
                    
                    $componentInfo = new ComponentInfo('sdc_inline_editor:custom-html-block');
                    
                    $components[] = $this->sdcValidatorService->validateSDCComponent(
                        $componentInfo,
                        $elementProps ?? []
                    );
                }
            }
        }

        return $components;
    }


    /**
     * Get attributes as props from a DOMElement.
     *
     * @param \DOMElement $element
     * @return array
     */
    private function getSDCPropsFromElement(DOMElement $element): array
    {
        $props = [];
        foreach ($element->attributes as $attribute) {
            if($attribute->nodeName === 'data-sdc-props') {
                $props = json_decode($attribute->nodeValue, true);
            }
        }
        return $props;
    }

    /**
     * Convert sdc- element name to sdc_inline_editor: format.
     *
     * @param \DOMElement $element
     * @return string
     */
    private function getElementNameFromElement(DOMElement $element): string
    {
        $library = $element->getAttribute('data-sdc-library');
        if ($library) {
            return $library . ':' . str_replace('sdc-', '', $element->tagName);
        }
        return str_replace('sdc-', 'sdc_inline_editor:', $element->tagName);
    }

}
