<?php

namespace Drupal\sdc_inline_editor\Service;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\media_library\MediaLibraryOpenerInterface;
use Drupal\media_library\MediaLibraryState;
use Drupal\image\Entity\ImageStyle;

/**
 * The media library opener for SDC Inline Editor.
 */
class MediaLibrarySdcInlineEditorOpener implements MediaLibraryOpenerInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module extensions list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleExtensionList;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * MediaLibrarySdcInlineEditorOpener constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ModuleExtensionList $module_extension_list, RendererInterface $renderer) {
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleExtensionList = $module_extension_list;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public function checkAccess(MediaLibraryState $state, AccountInterface $account): AccessResultInterface {
    return AccessResult::allowedIfHasPermission($account, 'sdc_inline_editor.edit_content');
  }

  /**
   * {@inheritdoc}
   */
  public function getSelectionResponse(MediaLibraryState $state, array $selected_ids): AjaxResponse {
    $response = new AjaxResponse();

    // Debug logging
    \Drupal::logger('sdc_inline_editor')->info('MediaLibrarySdcInlineEditorOpener::getSelectionResponse called with selected_ids: @ids', [
      '@ids' => implode(', ', $selected_ids)
    ]);

    // If nothing was selected, we should just return the response without
    // actions.
    if (empty($selected_ids)) {
      \Drupal::logger('sdc_inline_editor')->warning('No media items selected');
      return $response;
    }

    $opener_parameters = $state->getOpenerParameters();

    // Set "drupalSettings" data. We should put "uuid" as it requires for
    // detecting the field where media entities should be added and
    // "entities" - this parameter related to the default data returned by
    // "Media Browser" module.
    $drupal_settings['sdcInlineEditor']['mediaLibrary']['uuid'] = $opener_parameters['uuid'] ?? NULL;
    $drupal_settings['sdcInlineEditor']['mediaLibrary']['entities'] = [];

    /** @var \Drupal\media\MediaInterface[] $selected_media */
    $selected_media = $this->entityTypeManager->getStorage('media')->loadMultiple($selected_ids);
    foreach ($selected_media as $media) {
      // Get the dynamic field name for this media type
      $image_field_name = $this->getImageFieldName($media);
      
      if (!$image_field_name) {
        \Drupal::logger('sdc_inline_editor')->warning('Could not determine image field for media @id', [
          '@id' => $media->id()
        ]);
        continue;
      }

      // Get the image field and check if it has a value
      $image_field = $media->get($image_field_name);
      if ($image_field->isEmpty() || !$image_field->entity) {
        \Drupal::logger('sdc_inline_editor')->warning('Media @id has no image in field @field', [
          '@id' => $media->id(),
          '@field' => $image_field_name
        ]);
        continue;
      }

      // Get the file entity and ensure it's a file
      $file_entity = $image_field->entity;
      if (!$file_entity instanceof \Drupal\file\FileInterface) {
        \Drupal::logger('sdc_inline_editor')->warning('Media @id field @field does not contain a file entity', [
          '@id' => $media->id(),
          '@field' => $image_field_name
        ]);
        continue;
      }

      // Media thumbnail - use core's thumbnail style
      $image_uri = $file_entity->getFileUri();
      $image_style = ImageStyle::load('thumbnail');
      // Fallback to original image if thumbnail style doesn't exist
      $thumbnail_url = $image_style ? $image_style->buildUrl($image_uri) : \Drupal::service('file_url_generator')->generateAbsoluteString($image_uri);
      
      $drupal_settings['sdcInlineEditor']['mediaLibrary']['entities'][] = [
        $media->id(),
        $media->get('uuid')->getString(),
        $media->getEntityTypeId(),
        $thumbnail_url,
      ];
    }



    // Inject js script for adding media entities data to the "drupalSettings"
    // object.
    $encoded_drupal_settings = json_encode($drupal_settings['sdcInlineEditor']['mediaLibrary']);

    // Inject js script to trigger media selection callback
    $execute_media_selection = <<<JS
      (function () {
        try {
          // Trigger a custom event for media selection
          const event = new CustomEvent('sdcInlineEditorMediaSelected', {
            detail: $encoded_drupal_settings
          });
          // Try dispatching to both document and window
          window.parent.document.dispatchEvent(event);
          window.parent.dispatchEvent(event);
        } catch (error) {
          console.error('Error in media selection script:', error);
        }
      })();
    JS;

    // Inject js script for the closing iframe modal window.
    $close_iframe = <<<JS
      (function () {
        const modal = window.parent.document.querySelector('.sdc-inline-editor-media-modal');
        if (modal) {
          modal.remove();
        }
      })();
    JS;

    return $response
      // All these <script> tags will be injected in the iframe, not in the main
      // document.
      // But "body" of the scripts will be executed for the main document.
      ->addCommand(new HtmlCommand('body', '<script>' . $execute_media_selection . '</script>'))
      ->addCommand(new HtmlCommand('body', '<script>' . $close_iframe . '</script>'));
  }

  /**
   * Get the image field name from a media entity.
   *
   * @param \Drupal\media\MediaInterface $media_entity
   *   The media entity.
   *
   * @return string|null
   *   The image field name or NULL if not found.
   */
  private function getImageFieldName(\Drupal\media\MediaInterface $media_entity): ?string {
    if (!$media_entity) {
      return NULL;
    }
    
    $media_type = $media_entity->bundle();
    $media_type_entity = $this->entityTypeManager->getStorage('media_type')->load($media_type);
    
    if (!$media_type_entity || !$media_type_entity instanceof \Drupal\media\MediaTypeInterface) {
      return NULL;
    }
    
    $media_source = $media_entity->getSource();
    if (!$media_source) {
      return NULL;
    }
    
    $field_definition = $media_source->getSourceFieldDefinition($media_type_entity);
    if (!$field_definition) {
      return NULL;
    }
    
    return $field_definition->getName();
  }

}

