<?php

namespace Drupal\sdc_inline_editor\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\sdc_inline_editor\Service\ContentValidatorService;

/**
 * Service for handling page editing operations.
 * 
 * This service provides methods to:
 * - Save HTML content to node body fields
 * - Retrieve HTML content from node body fields
 * - Validate HTML content (placeholder for future implementation)
 * - Check user permissions for editing nodes
 * 
 * Used by the EditorController to provide REST API endpoints:
 * - POST /sdc-inline-editor/api/editor/save - Save HTML content to a node
 * - GET /sdc-inline-editor/api/editor/get - Get HTML content from a node
 * - POST /sdc-inline-editor/api/editor/validate - Validate HTML content
 */
class PageEditorService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The content validator service.
   *
   * @var \Drupal\sdc_inline_editor\Service\ContentValidatorService
   */
  protected $contentValidatorService;

  /**
   * Constructs a PageEditorService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\sdc_inline_editor\Service\ContentValidatorService $content_validator_service
   *   The content validator service.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    AccountInterface $current_user,
    ContentValidatorService $content_validator_service
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->contentValidatorService = $content_validator_service;
  }

  /**
   * Check if the current user has permission to edit the node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to check access for.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function checkEditAccess(NodeInterface $node): AccessResultInterface {
    // Check if user has SDC Inline Editor edit permission
    if (!$this->currentUser->hasPermission('sdc_inline_editor.edit_content')) {
      return AccessResult::forbidden('User does not have SDC Inline Editor edit permission.');
    }

    // Check if user has permission to edit this specific node
    if (!$node->access('update', $this->currentUser)) {
      return AccessResult::forbidden('User does not have permission to edit this node.');
    }

    return AccessResult::allowed();
  }

  /**
   * Save component-based content to a node's body field.
   *
   * @param int $node_id
   *   The node ID.
   * @param array $components
   *   Array of component objects with componentType and fields.
   *
   * @return array
   *   An array with 'success' boolean and 'message' string.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function saveComponentsToNode(int $node_id, array $components): array {
    try {
      // Load the node
      $node = $this->entityTypeManager->getStorage('node')->load($node_id);
      
      if (!$node || !($node instanceof NodeInterface)) {
        return [
          'success' => FALSE,
          'message' => 'Node not found.',
        ];
      }

      // Check if user has permission to edit this node
      $access_result = $this->checkEditAccess($node);
      if (!$access_result->isAllowed()) {
        return [
          'success' => FALSE,
          'message' => 'Access denied. You do not have permission to edit this node.',
        ];
      }

      // Process components into HTML
      $html_content = $this->processComponents($components);

      // Set the body content
      $node->set('body', [
        'value' => $html_content,
        'format' => 'full_html',
      ]);

      // Enable a new revision
      $node->setNewRevision(TRUE);
      $node->setRevisionUserId(\Drupal::currentUser()->id());
      $node->setRevisionLogMessage('Updated body content programmatically by SDC Inline Editor.');
      // Optionally, update the revision creation time
      $node->setRevisionCreationTime(\Drupal::time()->getRequestTime());

      // Save the node
      $node->save();

      return [
        'success' => TRUE,
        'message' => 'Components saved successfully.',
        'node_id' => $node_id,
      ];

    } catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => 'Error saving components: ' . $e->getMessage(),
      ];
    }
  }

  /**
   * Process component data into HTML format.
   *
   * @param array $components
   *   Array of component objects.
   *
   * @return string
   *   The processed HTML content.
   */
  protected function processComponents(array $components): string {
    $html_parts = [];

    foreach ($components as $component) {
      if (!isset($component['componentType']) || !isset($component['fields'])) {
        continue;
      }

      $component_type = $component['componentType'];
      $fields = $component['fields'];

      // Convert any prefix before : to sdc- format
      $html_tag = preg_replace('/^[^:]+:/', 'sdc-', $component_type);

      // Create props data including all fields
      $combined_data = [];
      foreach ($fields as $field_name => $field_data) {
        if (isset($field_data['value'])) {
          $value = $field_data['value'];
          
          // Special handling for widths field - ensure it's always an array
          if ($field_name === 'widths' && (!is_array($value) || $value === '')) {
            $value = [];
          }
          
          $combined_data[$field_name] = $value;
        }
      }

      if (preg_match('/^([^:]+):/', $component_type, $matches)) {
        $library = $matches[1];
      } else {
        $library = 'sdc_inline_editor';
      }
        
      if (!empty($combined_data)) {
        // JSON encode the data
        // Use JSON_HEX_APOS to escape single quotes as \u0027 for HTML attribute safety
        // Use JSON_UNESCAPED_SLASHES to avoid escaping forward slashes unnecessarily
        $data = json_encode($combined_data, JSON_HEX_APOS | JSON_UNESCAPED_SLASHES);
        // For HTML attributes with single quotes, we need to escape single quotes
        // json_encode with JSON_HEX_APOS already does this, so we don't need htmlspecialchars
        // which would double-encode HTML entities like <p> to &lt;p&gt;
        $html_parts[] = "<{$html_tag} data-sdc-library='" . htmlspecialchars($library, ENT_QUOTES) . "' data-sdc-props='" . $data . "'></{$html_tag}>";
      }
    }

    return implode("\n", $html_parts);
  }

  /**
   * Get the current content of a node's body field.
   *
   * @param int $node_id
   *   The node ID.
   *
   * @return array
   *   An array with 'success' boolean, 'content' string, and 'message' string.
   */
  public function getNodeContent(int $node_id): array {
    try {
      // Load the node
      $node = $this->entityTypeManager->getStorage('node')->load($node_id);
      
      if (!$node || !($node instanceof NodeInterface)) {
        return [
          'success' => FALSE,
          'content' => '',
          'message' => 'Node not found.',
        ];
      }

      // Check if user has permission to view this node
      if (!$node->access('view', $this->currentUser)) {
        return [
          'success' => FALSE,
          'content' => '',
          'message' => 'Access denied. You do not have permission to view this node.',
        ];
      }

      // Get the body content
      $body_field = $node->get('body');
      $content = $body_field->isEmpty() ? '' : $body_field->value;

      return [
        'success' => TRUE,
        'content' => $content,
        'message' => 'Content retrieved successfully.',
        'node_id' => $node_id,
      ];

    } catch (\Exception $e) {
      return [
        'success' => FALSE,
        'content' => '',
        'message' => 'Error retrieving content: ' . $e->getMessage(),
      ];
    }
  }

}
