<?php

namespace Drupal\sdc_inline_editor\Service;

use Drupal\Core\Theme\ComponentPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\sdc_inline_editor\Constants\SdcInlineEditorConstants;
use Drupal\sdc_inline_editor\Component\ComponentInfo;
use Drupal\Core\Render\RendererInterface;

class SDCService {
    private ComponentPluginManager $componentPluginManager;
    private ConfigFactoryInterface $configFactory;
    private SDCValidatorService $sdcValidatorService;
    private RendererInterface $renderer;

    public function __construct(ComponentPluginManager $componentPluginManager, ConfigFactoryInterface $configFactory, SDCValidatorService $sdcValidatorService, RendererInterface $renderer) {
        $this->componentPluginManager = $componentPluginManager;
        $this->configFactory = $configFactory;
        $this->sdcValidatorService = $sdcValidatorService;
        $this->renderer = $renderer;
    }
    public function getSDCComponents() {

        $sdc_components = $this->componentPluginManager->getAllComponents();
        $pr_components = [];
        foreach ($sdc_components as $component) {
            $componentInfo = ComponentInfo::fromComponent($component);
            $pr_components[] = $componentInfo;
        }

        return $pr_components;
    }

    public function getAccessibleComponents($content_type = NULL) {
        $config = $this->configFactory->get('sdc_inline_editor.settings');
        $all_components = $this->getSDCComponents();
        
        // Extract component names for comparison
        $all_component_names = array_map(fn($component) => $component->getComponentName(), $all_components);
        
        // If content type is specified, check for content-type specific settings
        if ($content_type) {
            $content_type_settings = $config->get('content_type_settings') ?: [];
            if (isset($content_type_settings[$content_type])) {
                $type_settings = $content_type_settings[$content_type];
                $enable_all = $type_settings['enable_all_components'] ?? FALSE;
                
                if ($enable_all) {
                    return $all_components;
                }
                
                $accessible_components = $type_settings['accessible_components'] ?? [];
                // Filter out unchecked components (FALSE values) and get only the keys
                $filtered = array_keys(array_filter($accessible_components, function($value) {
                    return $value !== FALSE && $value !== 0;
                }));
                
                // Validate that filtered components actually exist in available components
                $valid_component_names = array_intersect($filtered, $all_component_names);
                
                // If content type has specific settings, return them
                if (!empty($valid_component_names)) {
                    return array_filter($all_components, fn($component) => in_array($component->getComponentName(), $valid_component_names));
                }
            }
        }
        
        // Global settings fallback
        $enable_all = $config->get('enable_all_components') ?: FALSE;
        
        if ($enable_all) {
            return $all_components;
        }
        
        $accessible_components = $config->get('accessible_components') ?: [];
        // Filter out unchecked components (FALSE values) and get only the keys
        $filtered = array_keys(array_filter($accessible_components, function($value) {
            return $value !== FALSE && $value !== 0;
        }));
        
        // Validate that filtered components actually exist in available components
        $valid_component_names = array_intersect($filtered, $all_component_names);
        
        return array_filter($all_components, fn($component) => in_array($component->getComponentName(), $valid_component_names));
    }

    public function getRenderedHTMLForComponent(ComponentInfo $component, array $props = []) {
        $component = $this->sdcValidatorService->validateSDCComponent($component, ["editable" => TRUE, ...$props]);
        $renderArray = $component->buildRenderArray();
        $html = $this->renderer->renderRoot($renderArray);
        return $html;
    }
    /**
     * Check if a specific component is accessible for a given content type.
     *
     * @param string|ComponentInfo $component
     *   The component name or ComponentInfo object to check.
     * @param string|null $content_type 
     *   The content type to check against, or NULL for global check.
     *
     * @return bool
     *   TRUE if the component is accessible, FALSE otherwise.
     */
    public function isComponentAccessible($component, $content_type = NULL) {
        $accessible_components = $this->getAccessibleComponents($content_type);
        
        if ($component instanceof ComponentInfo) {
            $component_name = $component->getComponentName();
        } else {
            $component_name = $component;
        }
        
        $accessible_names = array_map(fn($comp) => $comp->getComponentName(), $accessible_components);
        return in_array($component_name, $accessible_names);
    }

    /**
     * Get the configuration source for accessible components.
     *
     * @param string|null $content_type
     *   The content type to check, or NULL for global.
     *
     * @return string
     *   The source of the configuration: 'content_type', 'global', or 'none'.
     */
    public function getConfigurationSource($content_type = NULL) {
        $config = $this->configFactory->get('sdc_inline_editor.settings');
        
        if ($content_type) {
            $content_type_settings = $config->get('content_type_settings') ?: [];
            if (isset($content_type_settings[$content_type])) {
                return 'content_type';
            }
        }
        
        $global_enable_all = $config->get('enable_all_components') ?: FALSE;
        $global_accessible = $config->get('accessible_components') ?: [];
        
        if ($global_enable_all || !empty($global_accessible)) {
            return 'global';
        }
        
        return 'none';
    }


}