<?php

namespace Drupal\sdc_inline_editor\Twig;

use Drupal\Core\Render\Markup;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extension for SDC attributes.
 */
class SDCAttributesExtension extends AbstractExtension {

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('sdc_component_attributes', [$this, 'getComponentAttributes']),
      new TwigFunction('sdc_field_attributes', [$this, 'getFieldAttributes']),
      new TwigFunction('sdc_html_editor_attributes', [$this, 'getHtmlEditorAttributes']),
      new TwigFunction('load_media_entity', [$this, 'loadMediaEntity']),
      new TwigFunction('media_image_url', [$this, 'getMediaImageUrl']),
      new TwigFunction('media_image_alt', [$this, 'getMediaImageAlt']),
      new TwigFunction('render_child_component', [$this, 'renderChildComponent']),
      new TwigFunction('get_available_image_styles', [$this, 'getAvailableImageStyles']),
      new TwigFunction('get_image_style_dropdown_options', [$this, 'getImageStyleDropdownOptions']),
    ];
  }

  /**
   * Generate SDC component attributes.
   *
   * @param string|null $component_name
   *   The component name.
   * @param array $additional_attributes
   *   Additional attributes to include.
   *
   * @return \Drupal\Core\Render\Markup
   *   The formatted attributes string.
   */
  public function getComponentAttributes(?string $component_name = NULL, array $additional_attributes = []): Markup {
    $attributes = [];
    
    if ($component_name) {
      $attributes['data-sdc-component'] = $component_name;
    }
    
    // Merge any additional attributes
    $attributes = array_merge($attributes, $additional_attributes);
    
    return $this->formatAttributes($attributes);
  }

  /**
   * Generate SDC field attributes.
   *
   * @param string $field_name
   *   The field name.
   * @param bool $editable
   *   Whether the field is editable.
   * @param array $additional_attributes
   *   Additional attributes to include.
   *
   * @return \Drupal\Core\Render\Markup
   *   The formatted attributes string.
   */
  public function getFieldAttributes(string $field_name, bool $editable = FALSE, array $additional_attributes = []): Markup {
    $attributes = [
      'data-sdc-field' => $field_name,
      'data-sdc-editable' => $editable ? 'true' : 'false',
    ];
    
    // Merge any additional attributes
    $attributes = array_merge($attributes, $additional_attributes);
    
    return $this->formatAttributes($attributes);
  }

  /**
   * Generate SDC HTML editor attributes.
   *
   * @param string|null $component_name
   *   The component name.
   * @param string $field_name
   *   The field name (defaults to 'html').
   * @param bool $editable
   *   Whether the field is editable.
   * @param string $type
   *   The editor type (defaults to 'html-editor').
   * @param array $additional_attributes
   *   Additional attributes to include.
   *
   * @return \Drupal\Core\Render\Markup
   *   The formatted attributes string.
   */
  public function getHtmlEditorAttributes(?string $component_name = NULL, string $field_name = 'html', bool $editable = FALSE, string $type = 'html-editor', array $additional_attributes = []): Markup {
    $attributes = [];
    
    if ($component_name) {
      $attributes['data-sdc-component'] = $component_name;
    }
    
    $attributes['data-sdc-field'] = $field_name;
    $attributes['data-sdc-editable'] = $editable ? 'true' : 'false';
    $attributes['data-sdc-type'] = $type;
    
    // Merge any additional attributes
    $attributes = array_merge($attributes, $additional_attributes);
    
    return $this->formatAttributes($attributes);
  }

  /**
   * Load a media entity by UUID.
   *
   * @param string $uuid
   *   The media entity UUID.
   * @param string|null $image_style
   *   Optional image style to apply to the image.
   *
   * @return \Drupal\media\MediaInterface|null
   *   The loaded media entity or NULL if not found.
   */
  public function loadMediaEntity(string $uuid, ?string $image_style = NULL): ?\Drupal\media\MediaInterface {
    if (empty($uuid)) {
      return NULL;
    }

    try {
      $entity_type_manager = \Drupal::entityTypeManager();
      $media_storage = $entity_type_manager->getStorage('media');
      $media_entities = $media_storage->loadByProperties(['uuid' => $uuid]);
      
      $media_entity = !empty($media_entities) ? reset($media_entities) : NULL;
      
      // If an image style is specified and media entity exists, store the image style
      // as a property on the media entity for use in templates
      if ($media_entity && $image_style) {
        $media_entity->_image_style = $image_style;
      }
      
      return $media_entity;
    } catch (\Exception $e) {
      \Drupal::logger('sdc_inline_editor')->error('Failed to load media entity with UUID @uuid: @message', [
        '@uuid' => $uuid,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Get the image field name from a media entity.
   *
   * @param \Drupal\media\MediaInterface $media_entity
   *   The media entity.
   *
   * @return string|null
   *   The image field name or NULL if not found.
   */
  private function getImageFieldName(?\Drupal\media\MediaInterface $media_entity): ?string {
    if (!$media_entity) {
      return NULL;
    }
    $media_type = $media_entity->bundle();
    $media_type_entity = \Drupal::entityTypeManager()->getStorage('media_type')->load($media_type);
    $media_source = $media_entity->getSource();
    if(!$media_type_entity || !($media_type_entity instanceof \Drupal\media\MediaTypeInterface)) {
      return NULL;
    }
    if(!$media_source) {
      return NULL;
    }
    $field_definition = $media_source->getSourceFieldDefinition($media_type_entity);
    if(!$field_definition) {
      return NULL;
    }
    return $field_definition->getName();
  }

  /**
   * Get the image URL from a media entity, optionally with image style applied.
   *
   * @param \Drupal\media\MediaInterface|null $media_entity
   *   The media entity.
   * @param string|null $image_style
   *   Optional image style to apply.
   *
   * @return string|null
   *   The image URL or NULL if not found.
   */
  public function getMediaImageUrl(?\Drupal\media\MediaInterface $media_entity, ?string $image_style = NULL): ?string {
    if (!$media_entity) {
      return NULL;
    } 

    try {
      // Get the image field
      $field_name = $this->getImageFieldName($media_entity);

      $image_field = $media_entity->get($field_name);
      if ($image_field->isEmpty()) {
        return NULL;
      }

      $file_entity = $image_field->entity;
      if (!$file_entity) {
        return NULL;
      }

      $image_uri = $file_entity->getFileUri();
      
      // If image style is specified, use it to build the URL
      if ($image_style) {
        $image_style_entity = \Drupal::entityTypeManager()->getStorage('image_style')->load($image_style);
        if ($image_style_entity && $image_style_entity instanceof \Drupal\image\ImageStyleInterface) {
          return $image_style_entity->buildUrl($image_uri);
        }
      }
      
      // Fallback to direct file URL
      return \Drupal::service('file_url_generator')->generateAbsoluteString($image_uri);
      
    } catch (\Exception $e) {
      \Drupal::logger('sdc_inline_editor')->error('Failed to get image URL from media entity @id: @message', [
        '@id' => $media_entity->id(),
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }
  /**
   * Get the image alt text from a media entity.
   *
   * @param \Drupal\media\MediaInterface|null $media_entity
   *   The media entity.
   *
   * @return string
   *   The image alt text or 'Media Image' if not found.
   */
  public function getMediaImageAlt(?\Drupal\media\MediaInterface $media_entity) {
    if (!$media_entity) {
      return 'Media Image';
    }

    $field_name = $this->getImageFieldName($media_entity);
    if(!$field_name) {
      return 'Media Image';
    }

    $image_field = $media_entity->get($field_name);
    if ($image_field->isEmpty()) {
      return 'Media Image';
    }

    $image_items = $image_field->getValue();
    if (!empty($image_items[0]['alt'])) {
      return $image_items[0]['alt'];
    }
    return 'Media Image';
  }

  /**
   * Render a child component from component data.
   *
   * @param array $component_data
   *   The component data array containing componentType and fields.
   * @param bool $editable
   *   Whether the component should be editable.
   *
   * @return array
   *   A Drupal render array for the component.
   */
  public function renderChildComponent(array $component_data, bool $editable = false): array {
    if (empty($component_data['componentType'])) {
      return [];
    }

    $component_type = $component_data['componentType'];
    
    // Validate that the component type is not empty after trimming
    if (empty(trim($component_type))) {
      \Drupal::logger('sdc_inline_editor')->warning('Attempted to render component with empty type');
      return [];
    }
    
    // Try to validate the component exists
    try {
      $component_plugin_manager = \Drupal::service('plugin.manager.sdc');
      $definitions = $component_plugin_manager->getDefinitions();
      
      // Check if the component exists
      $component_exists = false;
      foreach ($definitions as $definition) {
        if (isset($definition['id']) && $definition['id'] === $component_type) {
          $component_exists = true;
          break;
        }
      }
      
      if (!$component_exists) {
        \Drupal::logger('sdc_inline_editor')->warning('Attempted to render non-existent component: @component', [
          '@component' => $component_type,
        ]);
        return [];
      }
    } catch (\Exception $e) {
      \Drupal::logger('sdc_inline_editor')->error('Error validating component @component: @message', [
        '@component' => $component_type,
        '@message' => $e->getMessage(),
      ]);
      return [];
    }

    $fields = $component_data['fields'] ?? [];

    // Extract field values into props
    $props = [];
    foreach ($fields as $field_name => $field_data) {
      if (isset($field_data['value'])) {
        $value = $field_data['value'];
        
        // Special handling for widths field - ensure it's always an array
        if ($field_name === 'widths' && (!is_array($value) || $value === '')) {
          $value = [];
        }
        
        // Special handling for HTML fields - decode entities and mark as safe
        if ($field_name === 'html' && is_string($value)) {
          $value = Markup::create(html_entity_decode($value, ENT_QUOTES | ENT_HTML5, 'UTF-8'));
        }
        
        // Special handling for text fields with text-editor type - decode entities and mark as safe
        // Note: This will decode ALL HTML entities, including those that might represent literal text
        // The browser/CKEditor will then parse the HTML, which may normalize invalid structures
        if (isset($field_data['type']) && $field_data['type'] === 'text-editor' && is_string($value)) {
          $value = Markup::create(html_entity_decode($value, ENT_QUOTES | ENT_HTML5, 'UTF-8'));
        }
        
        // Special handling for checkbox-toggle fields - ensure boolean values are preserved
        if (isset($field_data['type']) && $field_data['type'] === 'checkbox-toggle') {
          // Convert string "true"/"false" to boolean if needed
          if (is_string($value)) {
            $value = ($value === 'true' || $value === '1' || $value === 'on');
          }
          // Ensure it's a boolean
          $value = (bool) $value;
        }
        
        $props[$field_name] = $value;
      }
    }

    // Add component_name to props
    $props['component_name'] = $component_type;
    
    // Add editable state as string to match Drupal's component system
    $props['editable'] = $editable ? TRUE : FALSE;

    // Create a render array
    return [
      '#type' => 'component',
      '#component' => $component_type,
      '#props' => $props,
    ];
  }

  /**
   * Get available image styles.
   *
   * @param string|null $prefix
   *   Optional prefix to filter image styles (e.g., 'sdc_inline_editor_').
   *
   * @return array
   *   Array of image style machine names.
   */
  public function getAvailableImageStyles(?string $prefix = NULL): array {
    try {
      $image_styles = \Drupal::entityTypeManager()
        ->getStorage('image_style')
        ->loadMultiple();
      
      $styles = [];
      foreach ($image_styles as $style) {
        $style_id = $style->id();
        if ($prefix === NULL || strpos($style_id, $prefix) === 0) {
          $styles[] = $style_id;
        }
      }
      
      return $styles;
    } catch (\Exception $e) {
      \Drupal::logger('sdc_inline_editor')->error('Failed to get available image styles: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Get image style dropdown options formatted for SDC select-dropdown.
   *
   * @param string|null $prefix
   *   Optional prefix to filter image styles (e.g., 'sdc_inline_editor_').
   *
   * @return string
   *   JSON string of dropdown options.
   */
  public function getImageStyleDropdownOptions(?string $prefix = 'sdc_inline_editor_'): string {
    try {
      $image_styles = \Drupal::entityTypeManager()
        ->getStorage('image_style')
        ->loadMultiple();
      
      $options = [];
      
      // Always include 'raw' option
      $options[] = ['value' => 'raw', 'label' => 'Raw (Original)'];
      
      foreach ($image_styles as $style) {
        $style_id = $style->id();
        if ($prefix === NULL || strpos($style_id, $prefix) === 0) {
          $label = $style->label();
          // Clean up label - remove "SDC Inline Editor, " prefix if present
          $label = preg_replace('/^SDC Inline Editor,?\s*/i', '', $label);
          $options[] = [
            'value' => $style_id,
            'label' => $label,
          ];
        }
      }
      
      // Sort options alphabetically by label, but keep 'raw' first
      usort($options, function($a, $b) {
        if ($a['value'] === 'raw') return -1;
        if ($b['value'] === 'raw') return 1;
        return strcasecmp($a['label'], $b['label']);
      });
      
      return json_encode($options);
    } catch (\Exception $e) {
      \Drupal::logger('sdc_inline_editor')->error('Failed to get image style dropdown options: @message', [
        '@message' => $e->getMessage(),
      ]);
      return json_encode([['value' => 'raw', 'label' => 'Raw (Original)']]);
    }
  }

  /**
   * Format attributes array into a string.
   *
   * @param array $attributes
   *   The attributes array.
   *
   * @return \Drupal\Core\Render\Markup
   *   The formatted attributes string as Markup object.
   */
  private function formatAttributes(array $attributes): Markup {

    $formatted = [];
    
    foreach ($attributes as $key => $value) {
      if (is_bool($value)) {
        $value = $value ? 'true' : 'false';
      }
      // Don't HTML encode for Twig attributes - Twig will handle the escaping
      $formatted[] = $key . '="' . $value . '"';
    }
    
    return Markup::create(implode(' ', $formatted));
  }

}
