<?php

namespace Drupal\search_api_algolia\Plugin\search_api\backend;

use Algolia\AlgoliaSearch\Exceptions\AlgoliaException;
use Algolia\AlgoliaSearch\SearchIndex;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\search_api\Backend\BackendPluginBase;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Item\ItemInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Query\ConditionGroupInterface;
use Drupal\search_api\Query\QueryInterface;
use Drupal\search_api\Utility\Utility;
use Drupal\search_api_algolia\Plugin\search_api\processor\ItemSplitter;
use Drupal\search_api_autocomplete\SearchInterface;
use Drupal\search_api_autocomplete\Suggestion\SuggestionFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Algolia SearchApiBackend definition.
 *
 * @SearchApiBackend(
 *   id = "search_api_algolia",
 *   label = @Translation("Algolia"),
 *   description = @Translation("Index items using a Algolia Search.")
 * )
 */
class SearchApiAlgoliaBackend extends BackendPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * Algolia Index.
   *
   * @var \Algolia\AlgoliaSearch\SearchIndex
   */
  protected $algoliaIndex = NULL;

  /**
   * A connection to the Algolia server.
   *
   * @var \Algolia\AlgoliaSearch\SearchClient
   */
  protected $algoliaClient;

  /**
   * The logger to use for logging messages.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Search API Algolia Helper service.
   *
   * @var \Drupal\search_api_algolia\SearchApiAlgoliaHelper
   */
  protected $helper;

  /**
   * Search query filter helper.
   *
   * @var \Drupal\search_api_algolia\SearchQueryHelper
   */
  protected $searchQueryHelper;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->languageManager = $container->get('language_manager');
    $instance->configFactory = $container->get('config.factory');
    $instance->helper = $container->get('search_api_algolia.helper');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->logger = $container->get('logger.channel.search_api_algolia');
    $instance->searchQueryHelper = $container->get('search_api_algolia.search_query_helper');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'application_id' => '',
      'api_key' => '',
      'disable_truncate' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['help'] = [
      '#markup' => '<p>' . $this->t('The application ID and API key an be found and configured at <a href="@link" target="blank">@link</a>.', ['@link' => 'https://algolia.com/account/api-keys']) . '</p>',
    ];

    $form['application_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Application ID'),
      '#description' => $this->t('The application ID from your Algolia subscription.'),
      '#default_value' => $this->getApplicationId(),
      '#required' => TRUE,
      '#size' => 60,
      '#maxlength' => 128,
    ];

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('The API key from your Algolia subscription.'),
      '#default_value' => $this->getApiKey(),
      '#required' => TRUE,
      '#size' => 60,
      '#maxlength' => 128,
    ];

    $form['disable_truncate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable truncation'),
      '#description' => $this->t('If checked, fields of type text and strong will not be truncated at 10000 characters. It will be site owner or developer responsibility to limit the characters. Truncating is always disabled when the item splitter processor is added to an index.'),
      '#default_value' => $this->configuration['disable_truncate'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function viewSettings() {
    $info = [];
    $indexes = [];

    try {
      $indexes = $this->listIndexes();
      $is_connected = TRUE;
    }
    catch (\Exception $e) {
      $is_connected = FALSE;
      $this->logger->warning('Could not connect to Algolia backend.');
    }

    // Application ID.
    $info[] = [
      'label' => $this->t('Application ID'),
      'info' => $this->getApplicationId(),
    ];

    // API Key.
    $info[] = [
      'label' => $this->t('API Key'),
      'info' => $this->getApiKey(),
    ];

    // Connection status.
    $message = $is_connected
      ? $this->t('The Algolia backend could be reached.')
      : $this->t('The Algolia backend could not be reached.');
    $info[] = [
      'label' => $this->t('Connection status'),
      'info' => $message,
      'status' => $is_connected ? 'success' : 'error',
    ];

    // Available indexes.
    $info[] = [
      'label' => $this->t('Available Algolia indexes'),
      'info' => implode(', ', $indexes),
    ];

    return $info;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\search_api\SearchApiException
   */
  public function removeIndex($index) {
    // Only delete the index's data if the index isn't read-only.
    if (!is_object($index) || empty($index->get('read_only'))) {
      $this->deleteAllIndexItems($index);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function indexItems(IndexInterface $index, array $items) {
    $objects = [];

    if ($index->isValidProcessor('algolia_item_splitter')) {
      $splitter = $index->getProcessor('algolia_item_splitter');
      assert($splitter instanceof ItemSplitter);
    }

    foreach ($items as $id => $item) {
      assert($item instanceof ItemInterface);
      $object = $this->prepareItem($index, $item);

      if (isset($splitter)) {
        // Delete existing splits of this item.
        try {
          $this->connect($index);
          $this->deleteSplits($item->getId());
        }
        catch (AlgoliaException $e) {
          $this->getLogger()->warning(Html::escape($e->getMessage()));
        }

        // Add new splits to the objects array.
        $splits = $splitter->splitItem($object);
        $objects = array_merge($objects, $splits);
      } else {
        $objects[$id] = $object;
      }
    }

    // Let other modules alter objects before sending them to Algolia.
    $this->alterAlgoliaObjects($objects, $index, $items);

    if (count($objects) > 0) {
      $itemsToIndex = [];

      if ($this->languageManager->isMultilingual()) {
        foreach ($objects as $item) {
          $itemsToIndex[$item['search_api_language']][] = $item;
        }
      }
      else {
        $itemsToIndex[''] = $objects;
      }

      foreach ($itemsToIndex as $language => $itemsPerLanguage) {
        // Allow adding objects to logs for investigation.
        if ($this->isDebugActive()) {
          foreach ($itemsPerLanguage as $item) {
            $this->logger->notice('Data pushed to Algolia for Language @language : @data', [
              '@data' => json_encode($item),
              '@language' => $language,
            ]);
          }
        }

        try {
          $this->connect($index, '', $language);
          if ($index->getOption('partially_update_objects', FALSE)) {
            $this->getAlgoliaIndex()
              ->partialUpdateObjects($itemsPerLanguage, ['createIfNotExists' => TRUE]);
          }
          else {
            $this->getAlgoliaIndex()->saveObjects($itemsPerLanguage);
          }
        }
        catch (AlgoliaException $e) {
          $this->logger->warning(Html::escape($e->getMessage()));
        }
      }
    }

    return array_keys($items);
  }

  /**
   * Indexes a single item on the specified index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index for which the item is being indexed.
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to index.
   */
  protected function indexItem(IndexInterface $index, ItemInterface $item) {
    $this->indexItems($index, [$item->getId() => $item]);
  }

  /**
   * Prepares a single item for indexing.
   *
   * Used as a helper method in indexItem()/indexItems().
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   Index.
   * @param \Drupal\search_api\Item\ItemInterface $item
   *   The item to index.
   *
   * @return array
   *   Item to index.
   */
  protected function prepareItem(IndexInterface $index, ItemInterface $item) {
    $item_id = $item->getId();
    $item_to_index = ['objectID' => $item_id];
    $object_id_field = $index->getOption('object_id_field');
    // Change objectID if some other field is used in the config.
    if ($object_id_field) {
      $entity = $item->getOriginalObject()->getValue();
      if ($entity instanceof ContentEntityInterface) {
        // Use the value of the field set in object_id_field config as objectID.
        $object_id = $entity->hasField($object_id_field) ? $entity->get($object_id_field)->getString() : '';
        if ($object_id) {
          $item_to_index['objectID'] = $object_id;
        }
      }
    }

    $item_fields = $item->getFields();
    $item_fields += $this->getSpecialFields($index, $item);
    $truncate = $this->isTruncatingEnabled($index);

    /** @var \Drupal\search_api\Item\FieldInterface $field */
    foreach ($item_fields as $field) {
      $type = $field->getType();
      $values = NULL;
      $field_values = $field->getValues();
      if (empty($field_values)) {
        continue;
      }
      foreach ($field_values as $field_value) {
        switch ($type) {
          case 'uri':
            $field_value .= '';
            if (mb_strlen($field_value) > 10000) {
              $field_value = mb_substr(trim($field_value), 0, 10000);
            }
            $values[] = $field_value;
            break;

          case 'text':
          case 'string':
            $field_value .= '';
            if ($truncate && mb_strlen($field_value) > 10000) {
              $field_value = mb_substr(trim($field_value), 0, 10000);
            }
            $values[] = $field_value;
            break;

          case 'integer':
          case 'duration':
          case 'decimal':
            $values[] = 0 + $field_value;
            break;

          case 'boolean':
            $values[] = $field_value ? TRUE : FALSE;
            break;

          case 'date':
            if (is_numeric($field_value) || !$field_value) {
              $values[] = 0 + $field_value;
              break;
            }
            $values[] = strtotime($field_value);
            break;

          default:
            $values[] = $field_value;
        }
      }
      if (is_array($values) && count($values) <= 1) {
        $values = reset($values);
      }
      $item_to_index[$field->getFieldIdentifier()] = $values;
    }

    return $item_to_index;
  }

  /**
   * Applies custom modifications to indexed Algolia objects.
   *
   * This method allows subclasses to easily apply custom changes before the
   * objects are sent to Algolia.
   *
   * @param array $objects
   *   An array of objects ready to be indexed, generated from $items array.
   * @param \Drupal\search_api\IndexInterface $index
   *   The search index for which items are being indexed.
   * @param array $items
   *   An array of items being indexed.
   *
   * @see hook_search_api_algolia_objects_alter()
   */
  protected function alterAlgoliaObjects(array &$objects, IndexInterface $index, array $items) {
    $this->moduleHandler->alter('search_api_algolia_objects', $objects, $index, $items);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteItems(IndexInterface $index, array $ids) {
    // When using custom field for object id, we handle the deletion of
    // objects in separate code.
    if ($index->getOption('object_id_field')) {
      return;
    }

    // Deleting all items included in the $ids array.
    foreach ($this->getLanguages($index) as $key) {
      // If algolia_index_batch_deletion enabled delete in batches
      // with drush command.
      if ($index->getOption('algolia_index_batch_deletion')) {
        $this->helper->scheduleForDeletion($index, $ids, $key);
        continue;
      }

      try {
        // Connect to the Algolia index for specific language.
        $this->connect($index, '', $key);
      }
      catch (\Exception $e) {
        $this->logger->error('Failed to connect to Algolia index while deleting indexed items, Error: @message', [
          '@message' => $e->getMessage(),
        ]);

        continue;
      }

      // Delete the splits first.
      foreach ($ids as $id) {
        $this->deleteSplits($id);
      }

      $response = $this->getAlgoliaIndex()->deleteObjects($ids);

      if ($this->isDebugActive()) {
        $this->logger->notice('Deletion requested for IDs: @ids on Algolia for Index: @index, Response: @response.', [
          '@response' => json_encode($response),
          '@index' => $this->getAlgoliaIndex()->getIndexName(),
          '@ids' => implode(',', $ids),
        ]);
      }

      // Wait for the deletion to be completed.
      if ($this->shouldWaitForDeleteToFinish()) {
        $response->wait();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllIndexItems(?IndexInterface $index = NULL, $datasource_id = NULL) {
    if (empty($index)) {
      return;
    }

    foreach ($this->getLanguages($index) as $key) {
      // Connect to the Algolia service.
      $this->connect($index, '', $key);

      // Clearing the full index.
      $response = $this->getAlgoliaIndex()->clearObjects();

      if ($this->isDebugActive()) {
        $this->logger->notice('Deletion requested for full index on Algolia Index: @index, Response: @response.', [
          '@response' => json_encode($response),
          '@index' => $this->getAlgoliaIndex()->getIndexName(),
        ]);
      }

      // Wait for the deletion to be completed.
      if ($this->shouldWaitForDeleteToFinish()) {
        $response->wait();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function search(QueryInterface $query) {
    $results = $query->getResults();
    $options = $query->getOptions();
    $sorts = $query->getSorts() ?? [];
    $search_api_index = $query->getIndex();
    $suffix = '';

    // Allow other modules to remove sorts handled in index rankings.
    $this->moduleHandler->alter('search_api_algolia_sorts', $sorts, $search_api_index);

    // Get the first sort to build replica name.
    // Replicas must be created with format PRIMARYINDEXNAME_FIELD_DIRECTION.
    // For instance index_stock_desc.
    foreach ($sorts as $field => $direction) {
      $suffix = '_' . strtolower($field . '_' . $direction);
      break;
    }

    try {
      $this->connect($search_api_index, $suffix);
      $index = $this->getAlgoliaIndex();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to connect to Algolia index while searching with suffix: @suffix, Error: @message', [
        '@message' => $e->getMessage(),
        '@suffix' => $suffix,
      ]);

      return $results;
    }

    $facets = isset($options['search_api_facets'])
      ? array_column($options['search_api_facets'], 'field')
      : [];

    $algolia_options = [
      'attributesToRetrieve' => [
        'search_api_id',
      ],
      'facets' => $facets,
      'clickAnalytics' => TRUE,
    ];

    if (!empty($options['limit'])) {
      $algolia_options['length'] = $options['limit'];
      $algolia_options['offset'] = $options['offset'];
    }

    // Allow Algolia specific options to be set dynamically.
    if (isset($options['algolia_options']) && is_array($options['algolia_options'])) {
      $algolia_options += $options['algolia_options'];
    }

    $condition_group = $query->getConditionGroup();
    $this->addLanguageConditions($condition_group, $query);
    $this->extractConditions($condition_group, $algolia_options, $facets, $query);

    // Algolia expects indexed arrays, remove the keys.
    if (isset($algolia_options['facetFilters'])) {
      $algolia_options['facetFilters'] = array_values($algolia_options['facetFilters']);
    }
    if (isset($algolia_options['disjunctiveFacets'])) {
      $algolia_options['disjunctiveFacets'] = array_values($algolia_options['disjunctiveFacets']);
    }

    // Filters and disjunctiveFacets are not supported together by Algolia.
    if (!empty($algolia_options['filters']) && !empty($algolia_options['disjunctiveFacets'])) {
      unset($algolia_options['disjunctiveFacets']);
    }

    $keys = $query->getOriginalKeys();
    $search = empty($keys) ? '*' : $keys;
    try {
      $data = $index->search($search, $algolia_options);
    }
    catch (\Throwable $e) {
      $this->logger->error('Search request failed. Index: @index. Error: @message. Request: @params', [
        '@index' => $index->getIndexName(),
        '@message' => $e->getMessage(),
        '@params' => json_encode([
          'query' => $search,
          'request_options' => $algolia_options,
        ]),
      ]);

      return $results;
    }

    $results->setResultCount($data['nbHits']);
    foreach ($data['hits'] ?? [] as $row) {
      $item = $this->getFieldsHelper()->createItem($query->getIndex(), $row['search_api_id']);
      if (!empty($row['_snippetResult'])) {
        $item->setExcerpt(implode('&hellip;', array_column($row['_snippetResult'], 'value')));
      }
      $results->addResultItem($item);
    }

    if (isset($data['facets'])) {
      $results->setExtraData(
        'search_api_facets',
        $this->extractFacetsData($facets, $data['facets'])
      );
    }

    if (isset($data['queryID'])) {
      $results->setExtraData(
        'queryID',
        $data['queryID']
      );
    }

    return $results;
  }

  /**
   * Creates a connection to the Algolia Search server as configured.
   *
   * @param \Drupal\search_api\IndexInterface|null $index
   *   Index to connect to.
   * @param string $index_suffix
   *   Index suffix, specified when connecting to replica or query suggestion.
   * @param string $langcode
   *   Language code to connect to.
   *   Specified when doing operations on both languages together.
   */
  protected function connect(?IndexInterface $index = NULL, $index_suffix = '', $langcode = '') {
    $client = $this->getAlgolia();

    if ($index && $index instanceof IndexInterface) {
      $indexId = ($index->getOption('algolia_index_name'))
        ? $index->getOption('algolia_index_name')
        : $index->get('id');

      if ($this->isLanguageSuffixEnabled($index)) {
        $langcode = $langcode ?: $this->languageManager->getCurrentLanguage()->getId();
        $indexId .= '_' . $langcode;
      }

      $indexId .= $index_suffix;
      $this->setAlgoliaIndex($client->initIndex($indexId));
    }
  }

  /**
   * Retrieves the list of available Algolia indexes.
   *
   * Also used in search_api_algolia_form_search_api_index_edit_form_alter.
   *
   * @return array
   *   List of indexes on Algolia.
   */
  public function listIndexes() {
    $indexes = [];

    $response = $this->getAlgolia()->listIndices();
    $items = $response['items'] ?? [];
    foreach ($items as $index) {
      $indexes[] = $index['name'];
    }

    return $indexes;
  }

  /**
   * Returns the AlgoliaSearch client.
   *
   * @return \Algolia\AlgoliaSearch\SearchClient
   *   The algolia instance object.
   */
  public function getAlgolia() {
    if (!$this->algoliaClient) {
      $this->algoliaClient = $this->helper->buildAlgoliaSearchClient($this->getApplicationId(), $this->getApiKey());
    }

    return $this->algoliaClient;
  }

  /**
   * Get the Algolia index.
   *
   * @returns \Algolia\AlgoliaSearch\SearchIndex
   *   Index.
   */
  protected function getAlgoliaIndex() {
    return $this->algoliaIndex;
  }

  /**
   * Set the Algolia index.
   */
  protected function setAlgoliaIndex(SearchIndex $index) {
    $this->algoliaIndex = $index;
  }

  /**
   * Deletes the split records associated with an item.
   *
   * @param string $itemId
   *   The ID of item object.
   */
  protected function deleteSplits(string $itemId) : void {
    // Look for split records in Algolia.
    $options = [
      'attributesToRetrieve' => ['objectID'],
      'filters' => sprintf('parent_record:%s', str_replace(':', '-', $itemId)),
    ];
    $records = $this->getAlgoliaIndex()->browseObjects($options);
    foreach ($records as $hit) {
      if (!empty($hit['objectID'])) {
        $splits_to_delete[] = $hit['objectID'];
      }
    }
    if (!empty($splits_to_delete)) {
      // Delete the record.
      $this->getAlgoliaIndex()->deleteObjects($splits_to_delete);
    }
  }

  /**
   * Get the ApplicationID (provided by Algolia).
   */
  protected function getApplicationId() {
    return $this->configuration['application_id'];
  }

  /**
   * Get the API key (provided by Algolia).
   */
  protected function getApiKey() {
    return $this->configuration['api_key'];
  }

  /**
   * Get whether truncation of records is enabled for a certain index.
   */
  protected function isTruncatingEnabled(IndexInterface $index): bool {
    return empty($this->configuration['disable_truncate'])
      && !$index->isValidProcessor('algolia_item_splitter');
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedFeatures() {
    return [
      'search_api_autocomplete',
      'search_api_facets',
      'search_api_facets_operator_or',
    ];
  }

  /**
   * Extract facets data from response.
   *
   * @param array $facets
   *   Facets to extract.
   * @param array $data
   *   Facets data from response.
   *
   * @return array
   *   Facets data in format required by Drupal.
   */
  private function extractFacetsData(array $facets, array $data) {
    $facets_data = [];

    foreach ($data as $field => $facet_data) {
      if (!in_array($field, $facets)) {
        continue;
      }

      foreach ($facet_data as $value => $count) {
        $facets_data[$field][] = [
          'count' => $count,
          'filter' => '"' . $value . '"',
        ];
      }
    }

    return $facets_data;
  }

  /**
   * Adds item language conditions to the condition group, if applicable.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   The condition group on which to set conditions.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The query to inspect for language settings.
   *
   * @see \Drupal\search_api\Query\QueryInterface::getLanguages()
   */
  protected function addLanguageConditions(ConditionGroupInterface $condition_group, QueryInterface $query) {
    $languages = $query->getLanguages();
    if (!empty($languages)) {
      $condition_group->addCondition('search_api_language', $languages, 'IN');
    }
  }

  /**
   * Extract conditions.
   *
   * @todo Move to \Drupal\search_api_algolia\SearchQueryHelper.
   *
   * @param \Drupal\search_api\Query\ConditionGroupInterface $condition_group
   *   Condition group.
   * @param array $options
   *   Algolia options to updatesearch_api_algolia.module.
   * @param array $facets
   *   Facets.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   A query representing the completed user input so far.
   */
  private function extractConditions(ConditionGroupInterface $condition_group, array &$options, array $facets, QueryInterface $query) {
    static $index_fields = [];
    $index = $query->getIndex();
    $index_id = $index->id();

    if (!isset($index_fields[$index_id])) {
      $index_fields[$index_id] = array_merge(
        $this->getSpecialFields($index),
        $index->getFields(TRUE)
      );
    }

    foreach ($condition_group->getConditions() as $condition) {
      if ($condition instanceof ConditionGroupInterface) {
        $this->extractConditions($condition, $options, $facets, $query);
        continue;
      }

      $field = $condition->getField();
      $raw_value = $condition->getValue();

      if ($raw_value === NULL) {
        throw new \LogicException('Algolia doesn’t support filtering on null values or missing attributes. More information: https://www.algolia.com/doc/guides/managing-results/refine-results/filtering/how-to/filter-by-attributes/#filter-by-null-or-missing-attributes');
      }

      /** @var \Drupal\search_api\Query\Condition $condition */
      // We support limited operators for now.
      if ($condition->getOperator() === '=') {
        $value = $this->searchQueryHelper->formatFilterValue($raw_value, $index_fields[$index_id][$field]);
        $query_str = $field . ':' . $value;

        if (in_array($field, $facets)) {
          $options['facetFilters'][$field][] = $query_str;
          $options['disjunctiveFacets'][$field] = $field;
        }
        else {
          $options['filters'] = !empty($options['filters'])
            ? $options['filters'] . ' AND ' . $query_str
            : $query_str;
        }
      }
      elseif (in_array($condition->getOperator(), ['<', '>', '<=', '>='])) {
        $value = $this->searchQueryHelper->formatFilterValue($raw_value, $index_fields[$index_id][$field]);
        $options['numericFilters'][] = $field . ' ' . $condition->getOperator() . ' ' . $value;
      }
      elseif (in_array($condition->getOperator(), ['IN', 'NOT IN'], TRUE)) {
        $values = $condition->getValue();
        $sub_query_str = '';
        foreach ($values as $value) {
          $value = $this->searchQueryHelper->formatFilterValue($value, $index_fields[$index_id][$field]);
          $query_str = $field . ':' . $value;
          if ($condition->getOperator() === 'NOT IN') {
            $query_str = 'NOT ' . $query_str;
          }
          $sub_query_str = !empty($sub_query_str)
            ? $sub_query_str . ' ' . ($condition->getOperator() === 'NOT IN' ? 'AND' : 'OR') . ' ' . $query_str
            : $query_str;
        }
        if (count($values) > 1) {
          $sub_query_str = '(' . $sub_query_str . ')';
        }
        $options['filters'] = !empty($options['filters'])
          ? $options['filters'] . ' AND ' . $sub_query_str
          : $sub_query_str;
      }
    }
  }

  /**
   * Implements autocomplete compatible to AutocompleteBackendInterface.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   A query representing the completed user input so far.
   * @param \Drupal\search_api_autocomplete\SearchInterface $search
   *   An object containing details about the search the user is on, and
   *   settings for the autocompletion. See the class documentation for details.
   *   Especially $search->options should be checked for settings, like whether
   *   to try and estimate result counts for returned suggestions.
   * @param string $incomplete_key
   *   The start of another fulltext keyword for the search, which should be
   *   completed. Might be empty, in which case all user input up to now was
   *   considered completed. Then, additional keywords for the search could be
   *   suggested.
   * @param string $user_input
   *   The complete user input for the fulltext search keywords so far.
   *
   * @return \Drupal\search_api_autocomplete\Suggestion\SuggestionInterface[]
   *   An array of suggestions.
   *
   * @see \Drupal\search_api_autocomplete\AutocompleteBackendInterface
   */
  public function getAutocompleteSuggestions(QueryInterface $query, SearchInterface $search, $incomplete_key, $user_input) {
    // This function will be used only is search_api_autocomplete is enabled
    // and used. We have it here to add the support but it might never be used
    // in normal cases.
    $suggestions = [];

    try {
      $factory = new SuggestionFactory($user_input);
    }
    catch (\Exception $e) {
      return $suggestions;
    }

    $search_api_index = $query->getIndex();

    try {
      $this->connect($search_api_index, '_query');
      $index = $this->getAlgoliaIndex();
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to connect to Algolia index with suffix: @suffix, Error: @message', [
        '@message' => $e->getMessage(),
        '@suffix' => '_query',
      ]);

      return $suggestions;
    }

    $algolia_options = [
      'attributesToRetrieve' => [
        'query',
      ],
      'analytics' => TRUE,
    ];

    try {
      $data = $index->search($user_input, $algolia_options);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to load autocomplete suggestions from Algolia. Query: @query, Error: @message', [
        '@message' => $e->getMessage(),
        '@query' => $user_input,
      ]);

      return $suggestions;
    }

    foreach ($data['hits'] ?? [] as $row) {
      $suggestions[] = $factory->createFromSuggestedKeys($row['query']);
    }

    return $suggestions;
  }

  /**
   * Wrapper function to check if debug mode is active or not as per config.
   *
   * @return bool
   *   TRUE if debug mode is active.
   */
  protected function isDebugActive() {
    static $debug_active = NULL;

    if (is_null($debug_active)) {
      $debug_active = $this->configFactory
        ->get('search_api_algolia.settings')
        ->get('debug') ?? FALSE;
    }

    return $debug_active;
  }

  /**
   * Wrapper to check if we need to wait for delete operation to finish.
   *
   * @return bool
   *   TRUE if we should wait.
   */
  protected function shouldWaitForDeleteToFinish() {
    static $should_wait = NULL;

    if (is_null($should_wait)) {
      $should_wait = $this->configFactory
        ->get('search_api_algolia.settings')
        ->get('wait_for_delete') ?? FALSE;
    }

    return $should_wait;
  }

  /**
   * Wrapper function to check if multi-lingual language suffix is enabled.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   Index to check for.
   *
   * @return bool
   *   If language suffix is enabled.
   */
  protected function isLanguageSuffixEnabled(IndexInterface $index) {
    return $this->languageManager->isMultilingual() && $index->getOption('algolia_index_apply_suffix');
  }

  /**
   * Get all the languages supported by the Index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   Index.
   *
   * @return array
   *   Supported languages for the index.
   */
  protected function getLanguages(IndexInterface $index) {
    $languages = [];

    if (!($this->isLanguageSuffixEnabled($index))) {
      // If not multi-lingual or suffix not supported, we simply do it once
      // with empty language code.
      return [''];
    }

    foreach ($index->getDatasources() as $datasource) {
      $config = $datasource->getConfiguration();

      $always_valid = [
        LanguageInterface::LANGCODE_NOT_SPECIFIED,
        LanguageInterface::LANGCODE_NOT_APPLICABLE,
      ];

      foreach ($this->languageManager->getLanguages() as $language) {
        if (Utility::matches($language->getId(), $config['languages'])
          || in_array($language->getId(), $always_valid)) {
          $languages[$language->getId()] = $language->getId();
        }
      }
    }

    return $languages;
  }

}
