<?php

namespace Drupal\Tests\search_api_algolia\Unit;

use Drupal\search_api\Item\FieldInterface;
use Drupal\search_api_algolia\SearchQueryHelper;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the SearchQueryHelper service.
 *
 * @group search_api_algolia
 */
class SearchQueryHelperTest extends UnitTestCase {

  /**
   * Tests formating filter with boolean value.
   *
   * @dataProvider formatFilterBooleanValueDataProvider
   */
  public function testFormatFilterBooleanValue($value, $expectedValue) {
    $field = $this->createMock(FieldInterface::class);
    $field->expects($this->once())
      ->method('getType')
      ->willReturn('boolean');

    $helper = new SearchQueryHelper();
    static::assertEquals($expectedValue, $helper->formatFilterValue($value, $field));
  }

  /**
   * Data provider for testFormatFilterBooleanValue().
   *
   * @see https://www.algolia.com/doc/api-reference/api-parameters/filters/#boolean-filters
   */
  public static function formatFilterBooleanValueDataProvider(): array {
    return [
      [0, 'false'],
      [1, 'true'],
    ];
  }

  /**
   * Tests formating filter with string value.
   *
   * @dataProvider formatFilterStringValueDataProvider
   */
  public function testFormatFilterStringValue($value, $expectedValue) {
    $field = $this->createMock(FieldInterface::class);
    $field->expects($this->once())
      ->method('getType')
      ->willReturn('string');

    $helper = new SearchQueryHelper();
    static::assertEquals($expectedValue, $helper->formatFilterValue($value, $field));
  }

  /**
   * Data provider for testFormatFilterStringValue().
   *
   * @see https://www.algolia.com/doc/api-reference/api-parameters/filters/#handle-attributes-with-spaces
   * @see https://www.algolia.com/doc/api-reference/api-parameters/filters/#handle-attributes-conflicting-with-keywords
   * @see https://www.algolia.com/doc/api-reference/api-parameters/filters/#handle-attributes-with-single-quotes
   * @see https://www.algolia.com/doc/api-reference/api-parameters/filters/#handle-attributes-with-double-quotes
   */
  public static function formatFilterStringValueDataProvider(): array {
    return [
      'common' => ['test', "'test'"],
      'with spaces' => ['Books and Comics', "'Books and Comics'"],
      'conflicting with keywords' => ['OR', "'OR'"],
      'with single quotes' => ["It's a wonderful day", "'It\\'s a wonderful day'"],
      'with double quotes' => ['She said "Hello World"', '\'She said \"Hello World\"\''],
    ];
  }

}
