<?php

namespace Drupal\search_api_decoupled\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\search_api_decoupled\SearchApiEndpointInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller routines for search api decoupled filters.
 */
class SearchApiDecoupledFilterLibrary extends ControllerBase {

  /**
   * Shows a list of search filters that can be added.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Drupal\search_api_decoupled\SearchApiEndpointInterface $search_api_endpoint
   *   The search api endpoint object.
   *
   * @return array
   *   A render array as expected by the renderer.
   */
  public function list(Request $request, SearchApiEndpointInterface $search_api_endpoint) {
    $headers = [
      ['data' => $this->t('Field name')],
      ['data' => $this->t('Field type')],
      ['data' => $this->t('Operations')],
    ];

    $definitions = $search_api_endpoint->getIndexFieldsDefinitions();
    uasort($definitions, function ($a, $b) {
      return Unicode::strcasecmp($a->getLabel(), $b->getLabel());
    });

    $rows = [];
    foreach ($definitions as $field_name => $field_definition) {
      $row = [];
      $row['title']['data'] = [
        '#type' => 'inline_template',
        '#template' => '<div class="block-filter-text-source">{{ label }}</div>',
        '#context' => [
          'label' => $field_definition->getLabel(),
        ],
      ];
      $row['type']['data'] = [
        '#type' => 'markup',
        '#markup' => $field_definition->getType(),
      ];
      $links['add'] = [
        'title' => $this->t('Add'),
        'url' => Url::fromRoute('search_api_decoupled.filter_add_form', [
          'search_api_endpoint' => $search_api_endpoint->id(),
          'filter' => $field_name,
        ]),
        'attributes' => [
          'class' => ['use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 700,
          ]),
        ],
      ];
      $row['operations']['data'] = [
        '#type' => 'operations',
        '#links' => $links,
      ];
      $rows[] = $row;
    }

    $build['#attached']['library'][] = 'block/drupal.block.admin';

    $build['filter'] = [
      '#type' => 'search',
      '#title' => $this->t('Search'),
      '#title_display' => 'invisible',
      '#size' => 30,
      '#placeholder' => $this->t('Search by filter name'),
      '#attributes' => [
        'class' => ['block-filter-text'],
        'data-element' => '.block-add-table',
        'title' => $this->t('Enter a part of the filter name to look for it.'),
      ],
    ];

    $build['blocks'] = [
      '#type' => 'table',
      '#header' => $headers,
      '#rows' => $rows,
      '#empty' => $this->t('No elements available.'),
      '#attributes' => [
        'class' => ['block-add-table'],
      ],
    ];

    return $build;
  }

}
