<?php

namespace Drupal\search_api_decoupled\Form;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api_decoupled\Entity\SearchApiEndpoint;
use Drupal\search_api_decoupled\SearchApiDecoupledFilterInterface;
use Drupal\search_api_decoupled\SearchApiDecoupledFilterPluginManager;
use Drupal\search_api_decoupled\SearchApiEndpointInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Provides a base form for search api decoupled filters.
 */
abstract class SearchApiDecoupledFilterFormBase extends FormBase {

  /**
   * The search endpoint.
   *
   * @var \Drupal\search_api_decoupled\SearchApiEndpointInterface
   */
  protected $searchApiEndpoint;

  /**
   * The search api decoupled filter.
   *
   * @var \Drupal\search_api_decoupled\SearchApiDecoupledFilterInterface
   */
  protected $filter;


  /**
   * The search api decoupled filter manager.
   *
   * @var \Drupal\search_api_decoupled\SearchApiDecoupledFilterPluginManager
   */
  protected $searchFilterManager;

  /**
   * Constructs a new SearchUiElementAddForm.
   *
   * @param \Drupal\search_api_decoupled\SearchApiDecoupledFilterPluginManager $search_filter_manager
   *   The search api decoupled filter manager.
   */
  public function __construct(SearchApiDecoupledFilterPluginManager $search_filter_manager) {
    $this->searchFilterManager = $search_filter_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.search_api_decoupled.filter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'search_api_decoupled_filter_form';
  }

  /**
   * Gets the search api decoupled filter.
   *
   * @return \Drupal\search_api_decoupled\SearchApiDecoupledFilterInterface
   *   The filter for this form.
   */
  public function getFilter() {
    return $this->filter;
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   A nested array form elements comprising the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param \Drupal\search_api_decoupled\SearchApiEndpointInterface $search_api_endpoint
   *   The search config.
   * @param string $filter
   *   The search api decoupled filter.
   *
   * @return array
   *   The form structure.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?SearchApiEndpointInterface $search_api_endpoint = NULL, $filter = NULL) {
    $this->searchApiEndpoint = $search_api_endpoint;
    try {
      $this->filter = $this->prepareFilter($filter);
    }
    catch (PluginNotFoundException $e) {
      throw new NotFoundHttpException("Invalid filter id: '$filter'.");
    }

    if (!($this->filter instanceof SearchApiDecoupledFilterInterface)) {
      throw new NotFoundHttpException();
    }

    $form['uuid'] = [
      '#type' => 'value',
      '#value' => $this->filter->getUuid(),
    ];
    $form['id'] = [
      '#type' => 'value',
      '#value' => $this->filter->getPluginId(),
    ];
    $form['field'] = [
      '#type' => 'value',
      '#value' => $this->filter->getField(),
    ];
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#default_value' => $this->filter->getLabel() ?? $this->getDefaultLabel($this->filter->getField()),
    ];
    $form['operator'] = [
      '#type' => 'select',
      '#title' => $this->t('Operator'),
      '#options' => SearchApiEndpoint::getOperators(),
      '#default_value' => $this->filter->getOperator(),
      '#states' => [
        'visible' => [
          ':input[name="exposed"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Value'),
      '#default_value' => $this->filter->getValue(),
      '#states' => [
        'visible' => [
          ':input[name="exposed"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['exposed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Expose filter'),
      '#default_value' => $this->filter->isExposed(),
    ];
    $expose_settings = $this->filter->getExposeSettings();
    $form['expose'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      '#title' => $this->t('Expose options'),
      '#states' => [
        'visible' => [
          ':input[name="exposed"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['expose']['identifier'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Identifier'),
      '#default_value' => $expose_settings['identifier'] ?? $this->filter->getField(),
      '#states' => [
        'required' => [
          ':input[name="exposed"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['expose']['multiple'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow multiple selections'),
      '#default_value' => $expose_settings['multiple'] ?? FALSE,
    ];

    $form['expose']['use_operator'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Expose operator'),
      '#default_value' => $expose_settings['use_operator'] ?? FALSE,
    ];

    $form['expose']['operator'] = [
      '#type' => 'select',
      '#title' => $this->t('Operator'),
      '#options' => SearchApiEndpoint::getOperators(),
      '#default_value' => $expose_settings['operator'] ?? 'equal',
      '#states' => [
        'visible' => [
          ':input[name="expose[use_operator]"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="expose[use_operator]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['expose']['operator_limit_selection'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Limit operator selection to allowed operators'),
      '#default_value' => $expose_settings['operator_limit_selection'] ?? FALSE,
      '#states' => [
        'visible' => [
          ':input[name="expose[use_operator]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['expose']['operator_list'] = [
      '#type' => 'select',
      '#multiple' => TRUE,
      '#title' => $this->t('List of allowed operators'),
      '#default_value' => $expose_settings['operator_list'] ?? [],
      '#options' => SearchApiEndpoint::getOperators(),
      '#states' => [
        'visible' => [
          ':input[name="expose[operator_limit_selection]"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="expose[operator_limit_selection]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#button_type' => 'primary',
      '#value' => $this->t('Save'),
    ];
    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => $this->searchApiEndpoint->toUrl('edit-form'),
      '#attributes' => ['class' => ['button']],
    ];
    return $form;
  }

  /**
   * Gets the default label for a field.
   *
   * @param string $field
   *   The field name.
   *
   * @return string
   *   The default label.
   */
  protected function getDefaultLabel($field) {
    foreach ($this->searchApiEndpoint->getIndexFieldsDefinitions() as $field_name => $field_definition) {
      if ($field_name === $field) {
        return $field_definition->getLabel();
      }
    }
    return $field;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_state->cleanValues();

    $this->filter->setField($form_state->getValue('field'));
    $this->filter->setExposed($form_state->getValue('exposed'));
    if ($this->filter->isExposed()) {
      $this->filter->setOperator('');
      $this->filter->setValue('');
      $this->filter->setExposeSettings($form_state->getValue('expose'));
    }
    else {
      $this->filter->setOperator($form_state->getValue('operator'));
      $this->filter->setValue($form_state->getValue('value'));
      $this->filter->setExposeSettings([]);
    }
    $this->filter->setLabel($form_state->getValue('label'));
    if (!$this->filter->getUuid()) {
      $this->searchApiEndpoint->addFilter($this->filter->getConfiguration());
    }
    $this->searchApiEndpoint->save();

    $this->messenger()->addStatus($this->t('The filter was successfully added.'));
    $form_state->setRedirectUrl($this->searchApiEndpoint->toUrl('edit-form'));
  }

  /**
   * Prepares the search api decoupled filter.
   *
   * @param string $filter
   *   The filter id.
   *
   * @return \Drupal\search_api_decoupled\SearchApiDecoupledFilterInterface
   *   The filter.
   */
  abstract protected function prepareFilter($filter);

}
