<?php

namespace Drupal\search_api_decoupled\Plugin\search_api_decoupled\filter;

use Drupal\Core\Plugin\PluginBase;
use Drupal\search_api_decoupled\SearchApiDecoupledFilterInterface;

/**
 * Base class for search api decoupled filter plugins.
 */
class SearchApiDecoupledFilterBase extends PluginBase implements SearchApiDecoupledFilterInterface {

  /**
   * The unique ID representing the search api decoupled filter.
   *
   * @var string
   */
  protected $uuid;

  /**
   * The field name.
   *
   * @var string
   */
  protected $field;

  /**
   * The filter operator.
   *
   * @var string
   */
  protected $operator;

  /**
   * The filter value.
   *
   * @var mixed
   */
  protected $value;

  /**
   * The filter label.
   *
   * @var string
   */
  protected $label;

  /**
   * The filter exposed.
   *
   * @var string
   */
  protected $exposed;

  /**
   * The filter expose options.
   *
   * @var array
   */
  protected $expose;

  /**
   * The filter weight.
   *
   * @var int
   */
  protected $weight;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->setConfiguration($configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getUuid() {
    return $this->uuid;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return [
      'id' => $this->getPluginId(),
      'uuid' => $this->uuid,
      'field' => $this->field,
      'operator' => $this->operator,
      'value' => $this->value,
      'label' => $this->label,
      'weight' => $this->weight,
      'exposed' => $this->exposed,
      'expose' => $this->expose,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $configuration += [
      'uuid' => '',
      'field' => '',
      'operator' => 'equal',
      'value' => '',
      'label' => '',
      'weight' => 0,
      'exposed' => FALSE,
      'expose' => [],
    ];
    $this->uuid = $configuration['uuid'];
    $this->field = $configuration['field'];
    $this->value = $configuration['value'];
    $this->operator = $configuration['operator'];
    $this->label = $configuration['label'];
    $this->weight = $configuration['weight'];
    $this->exposed = $configuration['exposed'];
    $this->expose = $configuration['expose'];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function isExposed() {
    return $this->exposed;
  }

  /**
   * {@inheritdoc}
   */
  public function setExposed($exposed) {
    $this->exposed = $exposed;
  }

  /**
   * {@inheritdoc}
   */
  public function setWeight($weight) {
    $this->weight = $weight;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight() {
    return $this->weight;
  }

  /**
   * {@inheritdoc}
   */
  public function setExposeSettings($expose_settings) {
    $this->expose = $expose_settings;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getExposeSettings() {
    return $this->expose;
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel() {
    return $this->label;
  }

  /**
   * {@inheritdoc}
   */
  public function setLabel(string $label) {
    $this->label = $label;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getField() {
    return $this->field;
  }

  /**
   * {@inheritdoc}
   */
  public function setField(string $field_name) {
    $this->field = $field_name;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperator() {
    return $this->operator;
  }

  /**
   * {@inheritdoc}
   */
  public function setOperator(string $operator) {
    $this->operator = $operator;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue() {
    return $this->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($value) {
    $this->value = $value;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function validateUserInput($value, $operator = 'equal') {
    // Check that operator corresponds to the user input counts.
    if (is_array($value) && !in_array($operator, ['in', 'not_in', 'between'])) {
      return FALSE;
    }
    elseif (!is_array($value) && in_array($operator, ['in', 'not_in', 'between'])) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function transformUserInput($value) {
    return $value;
  }

}
