<?php

declare(strict_types=1);

namespace Drupal\search_api_hal\Hook;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Entity\EntityInterface;
use Drupal\search_api\ServerInterface;
use Drupal\Core\Hook\Order\OrderAfter;

/**
 * Hook implementations used to alter and enhance forms.
 */
final class SearchApiHalHooks {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Implements hook_entity_operation_alter().
   *
   * Removes the "get_config_zip" operation from HAL servers.
   */
  #[Hook('entity_operation_alter')]
  public function alterServerOperations(array &$operations, EntityInterface $entity) {
    if($entity instanceof ServerInterface) {
      if($entity->getBackend()->getPluginId() == 'search_api_hal') {
        unset($operations['get_config_zip']);
      }
    }
  }

  /**
   * Implements hook_form_alter().
   *
   * Removes irrelevant configuration items.
   */
  #[Hook('form_alter', order: new OrderAfter(['search_api_solr']))]
  public function formAlter(array &$form, FormStateInterface $form_state, $form_id): void {
    if(in_array($form_id, ['search_api_server_form', 'search_api_server_edit_form'])) {
      $server = $form_state->getFormObject()->getEntity();
      $backend = $server->getBackendIfAvailable();
      if($backend && ($backend->getPluginId() == 'search_api_hal')) {
        // Hide empty fieldset "Configure Hal backend".
        $form['backend_config']['#type'] = 'container';
      } else {
        // Prevent standard Solr servers from picking the HAL connector.
        if(array_key_exists('connector', $form['backend_config'])) {
          unset($form['backend_config']['connector']['#options']['hal']);
        }
      }
    }

    // Hide "Delete all indexed data on this server" button.
    if('search_api_server_status' == $form_id) {
      $server = $form['#server'];
      $backend = $server->getBackendIfAvailable();
      if($backend && ($backend->getPluginId() == 'search_api_hal')) {
        $form['actions']['clear']['#access'] = FALSE;
      }
    }

    // Hide index items buttons from index page.
    if(('search_api_index_status' == $form_id) && !empty($form['#index'])) {
      $index = $form['#index'];
      $backend = $index->getServerInstanceIfAvailable()?->getBackendIfAvailable() ?? false;
      if($backend && ($backend->getPluginId() == 'search_api_hal')) {
        $form['actions']['#access'] = FALSE;
        $form['index']['#access'] = FALSE;
      }
    }

    if(in_array($form_id, ['search_api_index_form', 'search_api_index_edit_form'])) {
      $hal_servers = $this->getHalServers();
      $server = $form_state->getValue('server') ?? $form['server']['#default_value'];
      $is_hal_server = array_intersect([$server], array_keys($hal_servers));
      if($form_state->isRebuilding()) {
        $datasources = $form_state->getValue('datasources') ?? [];
      } else {
        $datasources = $form['datasources']['#default_value'];
      }

      // Lock solr_document configs
      if($is_hal_server && in_array('solr_document', $datasources)) {
        $form['datasource_configs']['solr_document']['id_field']['#default_value'] = 'docid';
        $form['datasource_configs']['solr_document']['id_field']['#attributes']['disabled'] = TRUE;
        $form['datasource_configs']['solr_document']['advanced']['language_field']['#access'] = FALSE;
        $form['datasource_configs']['solr_document']['advanced']['request_handler']['#access'] = FALSE;

        $form['datasource_configs']['solr_document']['id_field']['#default_value'] = 'docid';
        $form['datasource_configs']['solr_document']['id_field']['#attributes']['disabled'] = TRUE;

        $form['datasource_configs']['solr_document']['advanced']['default_query']['#default_value'] = '*:*';
        $form['datasource_configs']['solr_document']['advanced']['default_query']['#attributes']['disabled'] = TRUE;

        $form['datasource_configs']['solr_document']['advanced']['label_field']['#default_value'] = 'title_s';
        $form['datasource_configs']['solr_document']['advanced']['label_field']['#attributes']['disabled'] = TRUE;

        $form['datasource_configs']['solr_document']['advanced']['url_field']['#default_value'] = 'uri_s';
        $form['datasource_configs']['solr_document']['advanced']['url_field']['#attributes']['disabled'] = TRUE;
      }

      // Show an alert when datasources is not allowed
      if($is_hal_server && (!in_array('solr_document', $datasources) || (count($datasources) > 1))) {
        \Drupal::messenger()->addError(t('Only "Solr Document" datasource is allowed on this server.'));
      }

      // Hide irrelevant configurations.
      if($is_hal_server) {
        $form['tracker']['#access'] = false;
        $form['tracker_config']['#access'] = false;
        $form['options']['#access'] = false;
        $form['third_party_settings']['search_api_solr']['#access'] = false;
      }
    }

    if('view_preview_form' == $form_id) {
      $query = $form_state->getFormObject()->getEntity()->getExecutable()->query;
      if($query instanceof \Drupal\search_api\Plugin\views\query\SearchApiQuery) {
        $index = $query->getIndex();
        $backend = $index->getServerInstanceIfAvailable()?->getBackendIfAvailable() ?? false;
        if($backend && ($backend->getPluginId() == 'search_api_hal')) {
          // Disable live_preview
          $form['controls']['live_preview']['#default_value'] = false;
          // Show a warnign to the user
          \Drupal::messenger()->addWarning(t('You are querying Archives ouvertes HAL Solr Server. Please make sure the query you are building is lightweight to avoid performance issues.'));
        }
      }
    }

  }

  /**
   * Implements hook_form_alter().
   *
   * Warn user about performance issues on Views UI.
   */
  #[Hook('form_alter')]
  public function viewPreviewFormAlter(array &$form, FormStateInterface $form_state, $form_id): void {

    if('view_preview_form' == $form_id) {
      $query = $form_state->getFormObject()->getEntity()->getExecutable()->query;
      if($query instanceof \Drupal\search_api\Plugin\views\query\SearchApiQuery) {
        $index = $query->getIndex();
        $backend = $index->getServerInstanceIfAvailable()?->getBackendIfAvailable() ?? false;
        if($backend && ($backend->getPluginId() == 'search_api_hal')) {
          // Disable live_preview
          $form['controls']['live_preview']['#default_value'] = false;
          // Show a warning to the user
          \Drupal::messenger()->addWarning(t('You are querying Archives ouvertes HAL Solr Server. Please make sure the query you are building is lightweight to avoid performance issues.'));
        }
      }
    }

  }

  /**
   * Retrieves available hal servers as an options list.
   *
   * @return string[]
   *   An associative array mapping server IDs to their labels.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\search_api\SearchApiException
   */
  protected function getHalServers() {
    $solr_servers = [];
    /** @var \Drupal\search_api\ServerInterface[] $servers */
    $servers = $this->entityTypeManager
      ->getStorage('search_api_server')
      ->loadMultiple();
    foreach ($servers as $server) {
      if (
        $server->hasValidBackend() &&
        ($server->getBackend()->getPluginId() == 'search_api_hal')
      ) {
        $solr_servers[$server->id()] = $server;
      }
    }
    return $solr_servers;
  }
}
