<?php

namespace Drupal\search_api_hal\Plugin\SolrConnector;

use Drupal\search_api_solr\Plugin\SolrConnector\StandardSolrCloudConnector;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;
use Solarium\Core\Client\Endpoint;
use Solarium\Core\Client\Request;
use Solarium\QueryType\Select\Query\Query;
use Drupal\search_api_hal\HalEndpoint;

/**
 * Standard Solr Cloud connector.
 *
 * @SolrConnector(
 *   id = "hal",
 *   label = @Translation("Hal"),
 *   description = @Translation("A connector for a Archives ouvertes (HAL).")
 * )
 */
class HalConnector extends StandardSolrCloudConnector implements ContainerFactoryPluginInterface{

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    // Instantiate using the basic plugin constructor signature.
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    // Inject the config.factory service.
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'scheme' => 'https',
      'host' => 'api.archives-ouvertes.fr',
      'port' => null,
      'path' => '',
      'core' => null,
      'timeout' => 180,
      'index_timeout' => 30,
      'solr_version' => '6',
      'http_method' => 'GET',
      'skip_schema_check' => TRUE,
      'connector_config' => NULL,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Unset uneeded fields the port path and core of hal are irrelevant.
    unset($form['port']);
    unset($form['path']);
    unset($form['core']);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    parent::setConfiguration($configuration);
    $this->configuration['port'] = null;
  }

  /**
   * Returns a the Solr server URI.
   */
  protected function getServerUri() {
    $this->connect();
    $url_path = $this->solr->getEndpoint()->getScheme().'://'.$this->solr->getEndpoint()->getHost().$this->solr->getEndpoint()->getPath().'/';
    return $url_path;
  }

  /**
   * {@inheritdoc}
   */
  public function isCloud() {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getLuke() {
    // Get HAL fields fron config.
    $search_api_hal_settings = $this->configFactory->get('search_api_hal.settings');
    return $search_api_hal_settings->getRawData();
  }

  /**
   * {@inheritdoc}
   */
  public function executeRequest(Request $request, ?Endpoint $endpoint = NULL) {
    $request->removeParam('distrib');
    $request->removeParam('omitHeader');
    $result = parent::executeRequest($request, $endpoint);
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function search(Query $query, ?Endpoint $endpoint = NULL) {
    // We have to use /search route to query HAL (not /select)
    $query->setHandler('search');
    return parent::search($query, $endpoint);
  }

  /**
   * {@inheritdoc}
   */
  public function createEndpoint(string $key, array $additional_configuration = []) {
    $this->connect();
    $configuration = ['key' => $key, self::QUERY_TIMEOUT => $this->configuration['timeout']] + $additional_configuration + $this->configuration;
    unset($configuration['timeout']);
    $configuration['port'] = null;

    // Overriden to use our Endpoint implementation HalEndpoint
    $endpoint = new HalEndpoint($configuration);
    $this->solr->addEndpoint($endpoint);
    $this->solr->setDefaultEndpoint($endpoint);

    return $endpoint;
  }

  /**
   * {@inheritdoc}
   */
  public function getServerInfo($reset = FALSE) {
    $client = \Drupal::httpClient();
    $response = $client->get($this->getServerUri() . 'search');
    $statusCode = $response->getStatusCode();
    return ($statusCode == 200);
  }

  /**
   * {@inheritdoc}
   */
  public function pingCore(array $options = []) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getFile($file = NULL) {
    $this->connect();
    $query = $this->solr->createApi([
      'handler' => $this->configuration['core'] . '/admin/file',
    ]);
    if ($file) {
      $query->addParam('file', $file);
    }

    return $this->execute($query)->getResponse();
  }

}
