<?php

namespace Drupal\search_api_hal\Plugin\search_api\backend;

use Drupal\search_api_solr\Plugin\search_api\backend\SearchApiSolrBackend;
use Drupal\search_api\Attribute\SearchApiBackend;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Query\QueryInterface;
use Solarium\Core\Query\QueryInterface as SolariumQueryInterface;

/**
 * Archives ouvertes (HAL) backend for search api.
 */
#[SearchApiBackend(
  id: 'search_api_hal',
  label: new TranslatableMarkup('Hal'),
  description: new TranslatableMarkup('Archives ouvertes (HAL) server.')
)]
class SearchApiHalBackend extends SearchApiSolrBackend {

  /**
   * Builds the backend-specific configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function buildConnectorConfigForm(array &$form, FormStateInterface $form_state) {
    parent::buildConnectorConfigForm($form, $form_state);

    // Hide irrelevant config options.
    $form['connector_config']['#access'] = false;
    $form['multisite']['#access'] = false;
    $form['advanced']['#access'] = false;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    // Set default connecto value to hal
    return [
      'connector' => 'hal',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    // The connector is enforced to the default value wich is "hal"
    $form['connector']['#type'] = 'value';

    $this->buildConnectorConfigForm($form, $form_state);
    return $form;

  }

  /**
   * Assume Hal is always up.
   *
   * @return boolean
   */
  public function isAvailable() {
    return TRUE;
  }

  /**
   * Apply workarounds for special Solr versions before searching.
   *
   * @param \Solarium\Core\Query\QueryInterface $solarium_query
   *   The Solarium select query object.
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The \Drupal\search_api\Query\Query object representing the executed
   *   search query.
   */
  protected function applySearchWorkarounds(SolariumQueryInterface $solarium_query, QueryInterface $query) {
    parent::applySearchWorkarounds($solarium_query, $query);
    // Remove ss_search_api_language and index_filter fq filters from the queery.
    $filters = ['index_filter', 'language_filter'];
    foreach ($filters as $filter_name) {
      $filter = $solarium_query->getFilterQuery($filter_name);
      $solarium_query->removeFilterQuery($filter);
    }
  }

  /**
   * Returns a language-specific mapping from Drupal to Solr field names.
   *
   * @param string $language_id
   *   The language to get the mapping for.
   * @param \Drupal\search_api\IndexInterface $index
   *   The Search API index entity.
   *
   * @return array
   *   The language-specific mapping from Drupal to Solr field names.
   *
   * @throws \Drupal\search_api\SearchApiException
   */
  protected function formatSolrFieldNames($language_id, IndexInterface $index) {
    $field_mapping = parent::formatSolrFieldNames($language_id, $index);

    $backend = $index->getServerInstance()->getBackend();
    $connector = $backend->getSolrConnector();
    $luke = $connector->getLuke();

    // Add available hal fields to the query.
    foreach ($luke['fields'] as $field_name => $field) {
      $field_mapping[$field_name] = $field_name;
    }

    return $field_mapping;
  }


}
