<?php

declare(strict_types=1);

namespace Drupal\search_api_layout_block_filter\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\search_api\Event\IndexingItemsEvent;
use Drupal\search_api\Event\SearchApiEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listen to events relevant for this module.
 */
final class SearchApiLayoutBlockFilterSubscriber implements EventSubscriberInterface {

  /**
   * The config for this filter.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  private ImmutableConfig $config;

  /**
   * Constructs a SearchApiLayoutBlockFilterSubscriber object.
   */
  public function __construct(ConfigFactoryInterface $configFactory,) {
    $this->config = $configFactory->get('search_api_layout_block_filter.settings');
  }

  /**
   * React to the indexing of items by search API.
   */
  public function onIndexingItems(IndexingItemsEvent $event): void {
    if (!$this->config->get('enable_filter')) {
      return;
    }

    $enabled_indices = array_filter($this->config->get('indices'));
    $index = $event->getIndex();
    if (!in_array($index->id(), $enabled_indices)) {
      return;
    }

    $blocks_to_filter = array_filter($this->config->get('blocks'));
    $items = $event->getItems();
    foreach ($items as $item) {
      $entity = $item->getOriginalObject()->getValue();

      if (!($entity instanceof FieldableEntityInterface) || !$entity->hasField('layout_builder__layout')) {
        continue;
      }

      /** @var \Drupal\layout_builder\Field\LayoutSectionItemList $original_layout */
      $original_layout = $entity->get('layout_builder__layout');
      $layout = clone $original_layout;
      foreach ($layout->getSections() as $delta => $section) {
        foreach ($section->getComponents() as $uuid => $component) {
          if (in_array($component->getPluginId(), $blocks_to_filter)) {
            $section->removeComponent($uuid);
          }
        }
        // Remove empty sections.
        if (!$section->getComponents()) {
          $layout->removeSection($delta);
        }
      }
      // Set the layout property directly, since going trough the set() method
      // does not work. It is important that the entity is not saved, because
      // this would permanently remove the blocks from the layout.
      // @phpstan-ignore-next-line
      $entity->layout_builder__layout = $layout;
      $item->getOriginalObject()->setValue($entity);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      SearchApiEvents::INDEXING_ITEMS => ['onIndexingItems'],
    ];
  }

}
