<?php

declare(strict_types=1);

namespace Drupal\search_api_layout_block_filter\Form;

use Drupal\Core\Block\BlockManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Search API Layout Block filter settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The block plugin manager.
   *
   * @var \Drupal\Core\Block\BlockManagerInterface
   */
  private BlockManagerInterface $blockManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  private ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->blockManager = $container->get('plugin.manager.block');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'search_api_layout_block_filter_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['search_api_layout_block_filter.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['enable_filter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable filter'),
      '#default_value' => $this->config('search_api_layout_block_filter.settings')->get('enable_filter'),
    ];

    $form['indices'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Indices'),
      '#description' => $this->t('Select on which search API indices the filter should work.'),
      '#options' => $this->buildIndexOptions(),
      '#default_value' => $this->config('search_api_layout_block_filter.settings')->get('indices') ?? [],
    ];

    $form['blocks'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Blocks'),
      '#description' => $this->t('Select which blocks should be temporarily removed from the layout before it is rendered for indexing.'),
      '#options' => $this->buildBlockOptions(),
      '#default_value' => $this->config('search_api_layout_block_filter.settings')->get('blocks') ?? [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('search_api_layout_block_filter.settings')
      ->set('enable_filter', $form_state->getValue('enable_filter'))
      ->set('indices', $form_state->getValue('indices'))
      ->set('blocks', $form_state->getValue('blocks'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Build the options for the index form element.
   *
   * @return array
   *   The index options array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function buildIndexOptions(): array {
    /** @var \Drupal\search_api\Entity\Index[] $indices */
    $indices = $this->entityTypeManager->getStorage('search_api_index')->loadMultiple();
    return array_map(function ($index) {
      return $index->label();
    }, $indices);
  }

  /**
   * Build the options for the blocks form element.
   *
   * @return array
   *   The block options array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function buildBlockOptions(): array {
    // Get all block plugin definitions.
    $definitions = $this->blockManager->getDefinitions();
    $has_restrictions = $this->moduleHandler->moduleExists('layout_builder_restrictions');
    $has_browser = $this->moduleHandler->moduleExists('layout_builder_browser');

    if ($has_restrictions && $has_browser) {
      $lb_restriction_blocks = $this->getBlocksAllowedByRestrictions($definitions);
      $lb_browser_blocks = $this->getLayoutBuilderBrowserBlocks();
      return array_intersect_key($lb_restriction_blocks, $lb_browser_blocks);
    }
    elseif ($has_restrictions) {
      return $this->getBlocksAllowedByRestrictions($definitions);
    }
    elseif ($has_browser) {
      return $this->getLayoutBuilderBrowserBlocks();
    }

    // Fallback on displaying all available blocks if no restrictions are
    // available.
    return array_map(function ($block) {
      return $this->getBlockLabelWithCategory($block['admin_label'], $block['category']);
    }, $definitions);
  }

  /**
   * Get all blocks that are allowed via layout builder restrictions.
   *
   * @param array $definitions
   *   Array of block definitions.
   *
   * @return array
   *   Array of blocks that are allowed via layout builder restrictions.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getBlocksAllowedByRestrictions(array $definitions): array {
    // Iterate over all layouts that have Layout Builder enabled.
    $view_displays = $this->entityTypeManager->getStorage('entity_view_display')->loadByProperties([
      'third_party_settings.layout_builder.enabled' => TRUE,
    ]);

    $placeable_blocks = [];
    foreach ($view_displays as $display) {
      $settings = $display->getThirdPartySettings('layout_builder_restrictions');
      if (empty($settings)) {
        continue;
      }
      $entity_view_mode_restrictions = $settings['entity_view_mode_restriction'] ?? [];

      $allowlisted_blocks = (isset($entity_view_mode_restrictions['allowlisted_blocks'])) ? $entity_view_mode_restrictions['allowlisted_blocks'] : [];
      $denylisted_blocks = (isset($entity_view_mode_restrictions['denylisted_blocks'])) ? $entity_view_mode_restrictions['denylisted_blocks'] : [];

      // Add all allowlisted blocks to the global list.
      foreach ($allowlisted_blocks as $category => $block_list) {
        foreach ($block_list as $block_id) {
          if (isset($definitions[$block_id])) {
            $placeable_blocks[$block_id] = $this->getBlockLabelWithCategory($definitions[$block_id]['admin_label'], $category);
          }
        }
      }

      // Go over the block definitions and check for each block if it is
      // denylisted. If not, add it to the global list.
      foreach ($definitions as $id => $definition) {
        $category = $definition['category'] instanceof TranslatableMarkup ? $definition['category']->getUntranslatedString() : $definition['category'];
        if (isset($denylisted_blocks[$category])) {
          if (!in_array($id, $denylisted_blocks[$category])) {
            $placeable_blocks[$id] = $this->getBlockLabelWithCategory($definition['admin_label'], $category);
          }
        }
      }
    }
    return $placeable_blocks;
  }

  /**
   * Get all blocks that are available via layout builder browser.
   *
   * @return array
   *   Array of blocks that are available via layout builder browser.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getLayoutBuilderBrowserBlocks():array {
    $placeable_blocks = [];
    $lb_browser_blocks = $this->entityTypeManager->getStorage('layout_builder_browser_block')->loadByProperties(
      [
        'status' => TRUE,
      ]);
    foreach ($lb_browser_blocks as $lb_browser_block) {
      // LayoutBuilderBrowserBlock does not define getters, just the public
      // properties. The block class is not type hinted here because this module
      // does not declare a dependency on layout_builder_browser, therefore we
      // shouldn't reference a class defined by that module.
      // @phpstan-ignore-next-line
      $placeable_blocks[$lb_browser_block->block_id] = $this->getBlockLabelWithCategory($lb_browser_block->label(), $lb_browser_block->category);
    }
    return $placeable_blocks;
  }

  /**
   * Generate the label including the category of a block for the form options.
   *
   * @param \Stringable|string $label
   *   The actual block label, i.e. the admin label.
   * @param \Stringable|string $category
   *   The category the block belongs to.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The full label including the category.
   */
  private function getBlockLabelWithCategory(\Stringable|string $label, \Stringable|string $category): TranslatableMarkup {
    return $this->t('@label (Category: @category)', [
      '@category' => $category,
      '@label' => $label,
    ]);
  }

}
