<?php

declare(strict_types=1);

namespace Drupal\search_api_opensearch_semantic\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Plugin\views\filter\SearchApiFulltext;
use Drupal\search_api_opensearch_semantic\Plugin\search_api\data_type\Semantic;

/**
 * Base class for implementing semantic search filters.
 */
class SearchApiSemanticSearchFilterBase extends SearchApiFulltext {

  /**
   * {@inheritdoc}
   */
  public function defineOptions(): array {
    $options = parent::defineOptions();
    unset($options['operator']);
    $options['min_score'] = ['default' => 0.99];
    $options['semantic_field'] = NULL;
    $options['fulltext_fields'] = [];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);
    $form['parse_mode']['#description'] .= $this->t('<br>Parse mode <b><em>Single phrase</em></b> is recommended for semantic search.');
    // Allows to select fields of "semantic" data type.
    $form['semantic_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Searched semantic fields'),
      '#description' => $this->t('Select the fields that will be searched for semantic search.'),
      '#options' => $this->getSemanticFields(),
      '#default_value' => $this->options['semantic_field'],
    ];

    // Add field for the min_score to pass in neural query.
    // @see SemanticSearchParamBuilder for reference.
    $form['min_score'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum score'),
      '#default_value' => $this->options['min_score'] ?? 90,
      '#description' => $this->t('Min-score value for the neural query. </br> Specifies a similarity score, facilitating the retrieval of points that meet or exceed this score in relation to the query point. <br> For semantic search value can be between 0 and 1.'),
      '#min' => 0,
      '#step' => 0.01,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::submitOptionsForm($form, $form_state);
    $this->options['semantic_field'] = $form_state->getValue(['options', 'semantic_field']);
    $this->options['fulltext_fields'] = $form_state->getValue(['options', 'fields']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildExposeForm(&$form, FormStateInterface $form_state): void {
    parent::buildExposeForm($form, $form_state);
    unset($form['expose']['expose_fields']);
    unset($form['expose']['searched_fields_id']);
    unset($form['expose']['multiple']);
    $form['operator']['#access'] = FALSE;
    $form['value']['#access'] = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function adminSummary(): string {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function operators(): array {
    // @todo Check if this is really required.
    return [
      'or' => [
        'title' => $this->t('Contains any of these words'),
        'short' => $this->t('or'),
        'values' => 1,
      ],
    ];
  }

  /**
   * Retrieves a list of all available semantic fields.
   *
   * @return string[]
   *   An options list of semantic field identifiers mapped to their prefixed
   *   labels.
   */
  protected function getSemanticFields(): array {
    $fields = [];
    /** @var \Drupal\search_api\IndexInterface $index */
    $index = Index::load(substr($this->table, 17));

    $fields_info = $index->getFields();
    foreach ($index->getFulltextFields() as $field_id) {
      if ($fields_info[$field_id]->getType() !== Semantic::PLUGIN_ID) {
        continue;
      }
      $fields[$field_id] = $fields_info[$field_id]->getPrefixedLabel();
    }

    return $fields;
  }

}
