<?php

declare(strict_types=1);

namespace Drupal\search_api_opensearch_semantic\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\search_api_opensearch_semantic\Plugin\search_api\data_type\Semantic;

/**
 * Provides hooks for altering the search API index form.
 */
class IndexHooks {

  use StringTranslationTrait;

  /**
   * Adds k-nearest neighbors (k-NN) search option to the search API index form.
   */
  #[Hook('form_search_api_index_form_alter')]
  public function alterIndexForm(array &$form, FormStateInterface $form_state): void {
    $settings = [];
    /** @var \Drupal\Core\Entity\EntityFormInterface $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\search_api\IndexInterface $index */
    $index = $form_object->getEntity();
    if (!$index->isNew()) {
      $settings = $index->getThirdPartySettings('search_api_opensearch_semantic');
    }

    $form['third_party_settings']['search_api_opensearch_semantic'] = [
      '#tree' => TRUE,
      '#type' => 'details',
      '#title' => $this->t('OpenSearch Semantic (Experimental)'),
      '#collapsed' => TRUE,
    ];
    $form['third_party_settings']['search_api_opensearch_semantic']['knn'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Vector (k-NN) Index'),
      '#default_value' => $settings['knn'] ?? FALSE,
      '#description' => $this->t('See :link', [
        ':link' => 'https://docs.opensearch.org/docs/latest/vector-search/creating-vector-index/',
      ]),
    ];
    $form['third_party_settings']['search_api_opensearch_semantic']['model_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The model ID to use for semantic search'),
      '#default_value' => $settings['model_id'] ?? FALSE,
      '#description' => $this->t('See :link', [
        ':link' => 'https://docs.opensearch.org/docs/latest/vector-search/ai-search/semantic-search/#setting-a-default-model-on-an-index-or-field',
      ]),
    ];
    $form['third_party_settings']['search_api_opensearch_semantic']['search_pipeline'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The search pipeline for hybrid search'),
      '#default_value' => $settings['search_pipeline'] ?? FALSE,
      '#description' => $this->t('Search pipeline is used when we want to perform hybrid search. Hybrid search is mix of Semantic and normal text search. See :link', [
        ':link' => 'https://docs.opensearch.org/docs/latest/vector-search/ai-search/hybrid-search/index',
      ]),
    ];
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  #[Hook('form_search_api_index_fields_alter')]
  public function indexAddFieldFormAlter(array &$form, FormStateInterface $form_state): void {
    $form['#validate'][] = [self::class, 'validateIndexAddFieldForm'];
  }

  /**
   * Validate that field knn is enabled when semantic field is selected.
   */
  public static function validateIndexAddFieldForm(array &$form, FormStateInterface $form_state): void {
    $fields = $form_state->getValues()['fields'];
    /** @var \Drupal\search_api\Entity\Index $index */
    $index = $form_state->getFormObject()->getEntity();
    if ($fields === []) {
      return;
    }
    foreach ($fields as $field) {
      if ($field['type'] === Semantic::PLUGIN_ID && $index->getThirdPartySetting('search_api_opensearch_semantic', 'knn') !== TRUE) {
        $message = new TranslatableMarkup('%title field is required before semantic field type can be configured. Visit %link form to enable it. <br> You can enable it from <em>OpenSearch AI (Experimental) fieldset.</em>', [
          '%title' => 'Enable k-nearest neighbors (k-NN) search',
          '%link' => $index->toLink(new TranslatableMarkup('Edit'), 'edit-form')->toString(),
        ]);
        $form_state->setError($form, $message);
      }
    }
  }

}
