<?php

declare(strict_types=1);

namespace Drupal\search_api_opensearch_semantic\Plugin\search_api\processor;

use Drupal\Core\Form\FormStateInterface;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Processor\FieldsProcessorPluginBase;
use Drupal\search_api_opensearch\Plugin\search_api\backend\OpenSearchBackend;
use Drupal\search_api_opensearch_semantic\Plugin\search_api\data_type\Semantic;

/**
 * Text chunking processor for semantic fields.
 *
 * @SearchApiProcessor(
 *   id = "search_api_opensearch_semantic_text_chunking",
 *   label = @Translation("Text chunking"),
 *   description = @Translation("Enables text chunking for semantic fields."),
 *   stages = {
 *     "pre_index_save" = 0,
 *   },
 * )
 *
 * @see https://docs.opensearch.org/docs/latest/field-types/supported-field-types/semantic/#text-chunking
 */
class TextChunking extends FieldsProcessorPluginBase {

  use PluginFormTrait;

  /**
   * {@inheritdoc}
   */
  public static function supportsIndex(IndexInterface $index): bool {
    return $index->getServerInstance()->getBackend() instanceof OpenSearchBackend;
  }

  /**
   * {@inheritdoc}
   */
  public function supportsStage($stage): bool {
    return $stage === self::STAGE_PRE_INDEX_SAVE;
  }

  /**
   * {@inheritdoc}
   */
  protected function testType($type): bool {
    return $this->getDataTypeHelper()
      ->isTextType($type, [Semantic::PLUGIN_ID]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['fields']['#description'] = $this->t('Only semantic fields can have text chunking enabled.');
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function preIndexSave(): void {
    parent::preIndexSave();
    $fields = $this->configuration['fields'];
    foreach ($fields as $fieldName) {
      $field = $this->index->getField($fieldName);
      $configuration = $field->getConfiguration();
      $configuration['chunking'] = TRUE;
      $field->setConfiguration($configuration);
    }
  }

}
